package io.iohk.atala.prism.api

import io.iohk.atala.prism.crypto.derivation.DerivationAxis
import io.iohk.atala.prism.crypto.derivation.KeyDerivation
import io.iohk.atala.prism.crypto.keys.ECKeyPair
import io.iohk.atala.prism.identity.PublicKeyUsage
import kotlin.js.JsExport

@JsExport
public object KeyGenerator {
    /**
     * Given our root key `m`, generated from a mnemonic seed, PRISM protocol structures derivation
     * tree according to the following path:
     *
     * ```
     * m / DID_INDEX / KEY_TYPE / KEY_INDEX / '
     * ```
     *
     * Examples:
     * - m / 0' / 0' / 0' is the first master key for the first DID of a user derived from m
     * - m / 0' / 0' / 1' is the second master key derived for the first DID for a user derived from m.
     * - m / 1' / 0' / 0' is the first master key for the second DID derived from m
     * - m / 3' / 0' / 5' is the sixth master key derived for the fourth DID derived from m.
     * - m / 1' / 1' / 0' is the first issuing key for the second DID derived from m
     * - m / 3' / 1' / 1' is the second issuing key for the fourth DID derived from m
     *
     * @param seed the root BIP39 seed typically generated by
     * [io.iohk.atala.prism.crypto.derivation.KeyDerivation.binarySeed]
     * @param didIndex index of this particular DID among all DIDs used by the user (starts from 0
     * and counts up every time user needs a new DID)
     * @param keyType key type for the intended key usage
     * @param keyIndex index among keys of type [keyType] inside DID number [didIndex]
     * @return a key pair derived according to the PRISM protocol
     */
    public fun deriveKeyFromFullPath(
        seed: ByteArray,
        didIndex: Int,
        keyType: PublicKeyUsage,
        keyIndex: Int
    ): ECKeyPair =
        KeyDerivation.derivationRoot(seed)
            .derive(DerivationAxis.hardened(didIndex))
            .derive(DerivationAxis.hardened(keyType.derivationIndex()))
            .derive(DerivationAxis.hardened(keyIndex))
            .keyPair()
}
