package io.iohk.atala.prism.api

import io.iohk.atala.prism.common.PrismSdkInternal
import io.iohk.atala.prism.common.Validated
import io.iohk.atala.prism.credentials.*
import io.iohk.atala.prism.credentials.PrismCredential
import io.iohk.atala.prism.crypto.MerkleInclusionProof
import io.iohk.atala.prism.crypto.Sha256Digest
import io.iohk.atala.prism.crypto.keys.ECPublicKey
import io.iohk.atala.prism.identity.PrismDid
import io.iohk.atala.prism.identity.toModel
import io.iohk.atala.prism.protos.*
import io.iohk.atala.prism.protos.LedgerData
import io.iohk.atala.prism.protos.models.TimestampInfo
import io.iohk.atala.prism.protos.models.toModel
import kotlin.js.JsExport

internal typealias VE = VerificationError

@JsExport
@PrismSdkInternal
public data class PrismDidDocument(val didData: DIDData) {
    public fun getPublicKey(issuingKeyId: String): PublicKey? =
        didData.publicKeys.find { it.id == issuingKeyId }
}

@PrismSdkInternal
public class CredentialVerificationService(private val nodeService: NodeServiceCoroutine) {
    public suspend fun verify(
        signedCredential: PrismCredential,
        merkleInclusionProof: MerkleInclusionProof
    ): VerificationResult {
        val issuerDidValidated: Validated<PrismDid, VE> = extractValidIssuerDid(signedCredential)

        val getDidDocumentResponseValidated: Validated<GetDidDocumentResponse, VE> =
            issuerDidValidated.suspendableFlatMap { fetchPrismDidDocumentResponse(it) }

        val didDocumentValidated: Validated<PrismDidDocument, VE> = Validated.Applicative.apply(
            issuerDidValidated,
            getDidDocumentResponseValidated,
        ) { did, response -> getDidDocumentResponseValidated.flatMap { extractPrismDidDocument(response, did) } }

        val issuanceKeyIdValidated: Validated<String, VE> = extractValidIssuanceKeyId(signedCredential)

        val issuanceKeyValidated: Validated<PublicKey, VE> = Validated.Applicative.apply(
            didDocumentValidated,
            issuanceKeyIdValidated
        ) { didDocument, issuanceKey -> extractValidIssuingKeyDetailFromPrismDidDocument(didDocument, issuanceKey) }

        val issuanceKeyDetailValidated: Validated<ECPublicKey, VE> = Validated.Applicative.apply(
            issuanceKeyIdValidated,
            issuanceKeyValidated
        ) { issuanceKeyId, issuanceKey -> parseValidIssuingKeyDetail(issuanceKeyId, issuanceKey) }

        val issuingKeyAddedOnValidated: Validated<TimestampInfo, VE> = Validated.Applicative.apply(
            issuanceKeyIdValidated,
            issuanceKeyValidated
        ) { issuanceKeyId, issuanceKey -> extractValidIssuingKeyAddedOnTimestamp(issuanceKeyId, issuanceKey) }

        val issuingKeyRevokedOnValidated: Validated<TimestampInfo?, VE> =
            issuanceKeyValidated.map { extractIssuingKeyRevocationTimestamp(it) }

        val batchIdValidated: Validated<CredentialBatchId, VE> =
            issuerDidValidated.map { CredentialBatchId.fromBatchData(it.suffix, merkleInclusionProof.derivedRoot()) }

        val getBatchStateResponseValidated: Validated<GetBatchStateResponse, VE> =
            batchIdValidated.suspendableFlatMap { fetchBatchState(it) }

        val batchLedgerData: Validated<LedgerData, VE> = Validated.Applicative.apply(
            batchIdValidated,
            getBatchStateResponseValidated
        ) { batchId, getBatchStateResponse ->
            extractBatchPublicationLedgerData(
                getBatchStateResponse,
                batchId
            )
        }

        val batchIssuedOnValidated: Validated<TimestampInfo, VE> =
            Validated.Applicative.apply(batchLedgerData, batchIdValidated) { batchState, batchId ->
                extractValidBatchIssuedOnTimestamp(
                    batchState,
                    batchId
                )
            }

        val batchRevokedOnValidated: Validated<TimestampInfo?, VE> =
            getBatchStateResponseValidated.map { extractBatchRevocationTimestamp(it) }

        val getCredentialRevocationTimeResponseValidated: Validated<GetCredentialRevocationTimeResponse, VE> =
            batchIdValidated.suspendableMap { fetchGetCredentialRevocationTimeResponse(it, signedCredential) }

        val credentialRevocationTimeValidated: Validated<TimestampInfo?, VE> =
            getCredentialRevocationTimeResponseValidated.suspendableMap { extractCredentialRevocationTime(it) }

        val merkleProofValidated: Validated<Unit, VE> = validateMerkleProof(signedCredential, merkleInclusionProof)

        val credentialSignatureValidated: Validated<Unit, VE> =
            issuanceKeyDetailValidated.flatMap { validateCredentialSignature(signedCredential, it) }

        val batchNotRevokedValidated: Validated<Unit, VE> =
            batchRevokedOnValidated.flatMap { validateBatchNotRevoked(it) }

        val keyAddedBeforeIssuanceValidated: Validated<Unit, VE> = Validated.Applicative.apply(
            issuingKeyAddedOnValidated,
            batchIssuedOnValidated
        ) { issuingKey, batchIssuedOn -> validateKeyAddedBeforeIssuance(issuingKey, batchIssuedOn) }

        val keyAddedTimestampNotEqualToIssuanceValidated: Validated<Unit, VE> = Validated.Applicative.apply(
            issuingKeyAddedOnValidated,
            batchIssuedOnValidated
        ) { issuingKey, batchIssuedOn -> validateKeyAddedTimestampNotEqualToIssuance(issuingKey, batchIssuedOn) }

        val keyNotRevokedValidated: Validated<Unit, VE> = Validated.Applicative.apply(
            issuingKeyRevokedOnValidated,
            batchIssuedOnValidated
        ) { keyRevokedOn, batchIssuedOn -> validateKeyNotRevoked(keyRevokedOn, batchIssuedOn) }

        val keyRevokedTimestampNotEqualToIssuanceValidated: Validated<Unit, VE> = Validated.Applicative.apply(
            issuingKeyRevokedOnValidated,
            batchIssuedOnValidated
        ) { keyRevokedOn, batchIssuedOn ->
            validateKeyRevokedTimestampNotEqualToBatchIssuedOn(
                keyRevokedOn,
                batchIssuedOn
            )
        }

        val credentialNotRevokedValidated: Validated<Unit, VE> =
            credentialRevocationTimeValidated.flatMap { validateCredentialNotRevoked(it) }

        val lastSyncBlockTimestamp = listOfNotNull(
            getDidDocumentResponseValidated.result?.lastSyncedBlockTimestamp,
            getBatchStateResponseValidated.result?.lastSyncedBlockTimestamp,
            getCredentialRevocationTimeResponseValidated.result?.lastSyncedBlockTimestamp
        ).firstOrNull()

        return VerificationResult(
            listOf(
                issuerDidValidated,
                didDocumentValidated,
                issuanceKeyIdValidated,
                issuanceKeyValidated,
                issuanceKeyDetailValidated,
                issuingKeyAddedOnValidated,
                issuingKeyRevokedOnValidated,
                batchIdValidated,
                batchIssuedOnValidated,
                batchRevokedOnValidated,
                credentialRevocationTimeValidated,
                merkleProofValidated,
                credentialSignatureValidated,
                batchNotRevokedValidated,
                keyAddedBeforeIssuanceValidated,
                keyAddedTimestampNotEqualToIssuanceValidated,
                keyNotRevokedValidated,
                keyRevokedTimestampNotEqualToIssuanceValidated,
                credentialNotRevokedValidated
            ).mapNotNull { it.error }
                .distinct(),
            lastSyncBlockTimestamp
        )
    }

    private suspend fun fetchPrismDidDocumentResponse(did: PrismDid): Validated<GetDidDocumentResponse, VE> =
        Validated.Valid(nodeService.GetDidDocument(GetDidDocumentRequest(did = did.value)))

    private fun extractPrismDidDocument(
        response: GetDidDocumentResponse,
        did: PrismDid
    ): Validated<PrismDidDocument, VE> =
        Validated.getOrError(
            response.document?.let { PrismDidDocument(it) },
            VerificationError.IssuerDidDocumentNotFoundOnChain(did)
        )

    private suspend fun fetchBatchState(batchId: CredentialBatchId): Validated<GetBatchStateResponse, VE> =
        Validated.Valid(nodeService.GetBatchState(GetBatchStateRequest(batchId = Sha256Digest.fromHex(batchId.id).hexValue)))

    private fun extractBatchPublicationLedgerData(
        getBatchStateResponse: GetBatchStateResponse,
        batchId: CredentialBatchId
    ): Validated<LedgerData, VE> {
        return Validated.getOrError(
            getBatchStateResponse.publicationLedgerData,
            VerificationError.BatchNotFoundOnChain(batchId.id)
        )
    }

    private suspend fun fetchGetCredentialRevocationTimeResponse(
        batchId: CredentialBatchId,
        signedCredential: PrismCredential
    ): GetCredentialRevocationTimeResponse =
        nodeService.GetCredentialRevocationTime(
            GetCredentialRevocationTimeRequest(
                batchId = Sha256Digest.fromHex(batchId.id).hexValue,
                credentialHash = pbandk.ByteArr(signedCredential.hash().value)
            )
        )

    private fun extractCredentialRevocationTime(
        response: GetCredentialRevocationTimeResponse?,
    ): TimestampInfo? =
        response?.revocationLedgerData?.timestampInfo?.toModel()

    private fun validateMerkleProof(
        signedCredential: PrismCredential,
        merkleInclusionProof: MerkleInclusionProof
    ): Validated<Unit, VE> =
        Validated.validate(
            CredentialBatches.verifyInclusion(
                signedCredential,
                merkleInclusionProof.derivedRoot(),
                merkleInclusionProof
            ),
            VerificationError.InvalidMerkleProof
        )

    private fun extractValidIssuerDid(
        signedCredential: PrismCredential,
    ): Validated<PrismDid, VE> =
        Validated.getOrError(
            signedCredential.content.getIssuerDid(),
            VerificationError.IssuerDidNotFoundInCredential
        )

    private fun extractValidIssuanceKeyId(
        signedCredential: PrismCredential,
    ): Validated<String, VE> =
        Validated.getOrError(
            signedCredential.content.getIssuanceKeyId(),
            VerificationError.IssuerKeyNotFoundInCredential
        )

    private fun extractValidIssuingKeyDetailFromPrismDidDocument(
        didDocument: PrismDidDocument,
        issuingKeyId: String
    ): Validated<PublicKey, VE> =
        Validated.getOrError(
            didDocument.getPublicKey(issuingKeyId),
            VerificationError.IssuingKeyIdNotFoundOnChain(issuingKeyId, didDocument)
        )

    private fun parseValidIssuingKeyDetail(
        issuingKeyId: String,
        publicKey: PublicKey
    ): Validated<ECPublicKey, VE> =
        Validated.getOrError(
            publicKey.toModel()?.didPublicKey?.publicKey,
            VerificationError.OnChainIssuingKeyCannotBeParsed(issuingKeyId, publicKey)
        )

    private fun extractValidIssuingKeyAddedOnTimestamp(
        issuingKeyId: String,
        publicKey: PublicKey
    ): Validated<TimestampInfo, VE> =
        Validated.getOrError(
            publicKey.addedOn?.timestampInfo?.toModel(),
            VerificationError.IssuanceKeyPublicationTimestampNotFoundOnChain(
                issuingKeyId,
                publicKey
            )
        )

    private fun extractIssuingKeyRevocationTimestamp(
        publicKey: PublicKey
    ): TimestampInfo? = publicKey.revokedOn?.timestampInfo?.toModel()

    private fun validateCredentialSignature(
        signedCredential: PrismCredential,
        issuanceKeyDetail: ECPublicKey
    ): Validated<Unit, VE> =
        Validated.validate(
            signedCredential.isValidSignature(issuanceKeyDetail),
            VerificationError.InvalidSignature(signedCredential, issuanceKeyDetail)
        )

    private fun extractValidBatchIssuedOnTimestamp(
        batchLedgerData: LedgerData?,
        batchId: CredentialBatchId
    ): Validated<TimestampInfo, VE> =
        Validated.getOrError(
            batchLedgerData?.timestampInfo?.toModel(),
            VerificationError.BatchPublicationTimestampNotFoundOnChain(batchId.id)
        )

    private fun extractBatchRevocationTimestamp(
        batchStateResponse: GetBatchStateResponse,
    ): TimestampInfo? = batchStateResponse.revocationLedgerData?.timestampInfo?.toModel()

    private fun validateBatchNotRevoked(maybeBatchRevokedOn: TimestampInfo?): Validated<Unit, VE> =
        maybeBatchRevokedOn?.let { Validated.Invalid(VerificationError.BatchWasRevokedOn(it)) } ?: Validated.Valid(Unit)

    private fun validateKeyAddedBeforeIssuance(
        keyAddedOn: TimestampInfo,
        batchIssuedOn: TimestampInfo
    ): Validated<Unit, VE> =
        Validated.validate(
            !batchIssuedOn.occurredBefore(keyAddedOn),
            VerificationError.KeyAddedAfterIssuance(
                keyAddedOn = keyAddedOn,
                batchIssuedOn = batchIssuedOn
            )
        )

    private fun validateKeyAddedTimestampNotEqualToIssuance(
        keyAddedOn: TimestampInfo,
        batchIssuedOn: TimestampInfo
    ): Validated<Unit, VE> =
        Validated.validate(
            batchIssuedOn != keyAddedOn,
            VerificationError.KeyAddedTimestampEqualsIssuance(
                keyAddedOn = keyAddedOn,
                batchIssuedOn = batchIssuedOn
            )
        )

    private fun validateKeyNotRevoked(
        keyRevokedOn: TimestampInfo?,
        batchIssuedOn: TimestampInfo
    ): Validated<Unit, VE> = keyRevokedOn?.let {
        Validated.validate(
            batchIssuedOn.occurredBefore(keyRevokedOn),
            VerificationError.KeyWasRevoked(
                keyRevokedOn = keyRevokedOn,
                batchIssuedOn = batchIssuedOn
            )
        )
    } ?: Validated.Valid(Unit)

    private fun validateKeyRevokedTimestampNotEqualToBatchIssuedOn(
        keyRevokedOn: TimestampInfo?,
        batchIssuedOn: TimestampInfo
    ): Validated<Unit, VE> = keyRevokedOn?.let {
        Validated.validate(
            batchIssuedOn != keyRevokedOn,
            VerificationError.KeyRevokedTimestampEqualsIssuance(
                keyRevokedOn = keyRevokedOn,
                batchIssuedOn = batchIssuedOn
            )
        )
    } ?: Validated.Valid(Unit)

    private fun validateCredentialNotRevoked(maybeCredentialRevokedOn: TimestampInfo?): Validated<Unit, VE> =
        maybeCredentialRevokedOn?.let {
            Validated.Invalid(
                VerificationError.CredentialWasRevokedOn(
                    it
                )
            )
        }
            ?: Validated.Valid(Unit)
}

/**
 * @see io.iohk.atala.prism.api.node.NodePublicApi.verify
 *
 * This class represents the result of credential verification.
 *
 * @param verificationErrors list of errors that occurred during the verification, if empty, the
 * verification ended successfully
 * @param lastSyncBlockTimestamp if Node was contacted, the latest synchronized Cardano block
 * timestamp; null otherwise
 */
@JsExport
public data class VerificationResult(
    val verificationErrors: List<VE>,
    val lastSyncBlockTimestamp: pbandk.wkt.Timestamp?
)

/**
 * Represents an errors that can occur during credential verification.
 *
 * Full list of possible errors:
 * - `InvalidMerkleProof` - provided inclusion proof does not match the credential batch
 * - `IssuerDidNotFoundInCredential` - credential does not contain a valid issuer DID
 * - `IssuerKeyNotFoundInCredential` - credential does not contain a valid issuer key with which the
 * credential was issued
 * - `IssuerDidDocumentNotFoundOnChain` - issuer's DID was not found on Cardano ledger accroding to
 * Node
 * - `IssuingKeyIdNotFoundOnChain` - issuer's DID does not contain the key with which the credential
 * was issued
 * - `OnChainIssuingKeyCannotBeParsed` - the issuing key contained in issuer's DID is malformed
 * - `IssuanceKeyPublicationTimestampNotFoundOnChain` - the issuing key contained in issuer's DID
 * is missing a publication timestamp
 * - `InvalidSignature` - credential's signature cannot be verified
 * - `BatchNotFoundOnChain` - credential batch was not published on Cardano ledger accroding to Node
 * - `BatchPublicationTimestampNotFoundOnChain` - credential batch is missing publication timestamp
 * - `BatchWasRevokedOn` - the entire credential batch has already been revoked
 * - `KeyAddedAfterIssuance` - the issuing key contained in issuer's DID was added after the
 * credential batch was issued
 * - `KeyAddedTimestampEqualsIssuance` - the issuing key contained in issuer's DID was added on the
 * exact same timestamp the credential batch was issued
 * - `KeyWasRevoked` - the issuing key contained in issuer's DID was revoked before the credential
 * batch was issued
 * - `KeyRevokedTimestampEqualsIssuance` - the issuing key contained in issuer's DID was revoked on
 * the exact same timestamp the credential batch was issued
 * - `CredentialWasRevokedOn` - the credential has already been revoked
 *
 * @param errorMessage human-readable error message
 */
@JsExport
public sealed class VerificationError(public val errorMessage: String) {

    public object InvalidMerkleProof : VerificationError("Invalid merkle proof")

    public object IssuerDidNotFoundInCredential : VerificationError("Issuer PrismDid not found in credential")

    public object IssuerKeyNotFoundInCredential : VerificationError("Issuer Key not found in credential")

    public data class IssuerDidDocumentNotFoundOnChain(val did: PrismDid) :
        VerificationError("Issuer PrismDid Document not found for PrismDid:$did.value")

    public data class IssuingKeyIdNotFoundOnChain(val issuerKeyId: String, val didDocument: PrismDidDocument) :
        VerificationError("Issuing Key not found on chain. issuingKeyId=$issuerKeyId didDocument=$didDocument")

    public data class OnChainIssuingKeyCannotBeParsed(val issuerKeyId: String, val publicKey: PublicKey) :
        VerificationError("Unable to parse on chain Issuing Key. issuingKeyId=$issuerKeyId publicKey=$publicKey")

    public data class IssuanceKeyPublicationTimestampNotFoundOnChain(
        val issuerKeyId: String,
        val publicKey: PublicKey
    ) :
        VerificationError("Issuance Key publication timestamp not found on chain. issuingKeyId=$issuerKeyId publicKey=$publicKey")

    public data class InvalidSignature(val signedCredential: PrismCredential, val publicKey: ECPublicKey) :
        VerificationError("Invalid signature. credential=$signedCredential publicKey=$publicKey")

    public data class BatchNotFoundOnChain(val batchId: String) :
        VerificationError("Batch not found on chain. BatchId:$batchId")

    public data class BatchPublicationTimestampNotFoundOnChain(val batchId: String) :
        VerificationError("Publication timestamp not found on chain. BatchId:$batchId")

    public data class BatchWasRevokedOn(val timestamp: TimestampInfo) :
        VerificationError("Batch was revoked. RevokedOn:$timestamp")

    public data class KeyAddedAfterIssuance(val keyAddedOn: TimestampInfo, val batchIssuedOn: TimestampInfo) :
        VerificationError("Invalid key. Key added after issuance. keyAddedOn:$keyAddedOn batchIssuedOn:$batchIssuedOn")

    public data class KeyAddedTimestampEqualsIssuance(val keyAddedOn: TimestampInfo, val batchIssuedOn: TimestampInfo) :
        VerificationError("Key added time should never be equal to issuance time. keyAddedOn:$keyAddedOn batchIssuedOn:$batchIssuedOn")

    public data class KeyWasRevoked(val keyRevokedOn: TimestampInfo, val batchIssuedOn: TimestampInfo) :
        VerificationError("Key was revoked before credential issuance. keyRevokedOn:$keyRevokedOn batchIssuedOn:$batchIssuedOn")

    public data class KeyRevokedTimestampEqualsIssuance(
        val keyRevokedOn: TimestampInfo,
        val batchIssuedOn: TimestampInfo
    ) :
        VerificationError("Key revoked time should never be equal to issuance time. keyRevokedOn:$keyRevokedOn batchIssuedOn:$batchIssuedOn")

    public data class CredentialWasRevokedOn(val timestamp: TimestampInfo) :
        VerificationError("Credential was revoked. RevokedOn:$timestamp")
}
