package io.iohk.atala.prism.api

import com.benasher44.uuid.bytes
import com.benasher44.uuid.uuid4
import com.benasher44.uuid.uuidOf
import io.iohk.atala.prism.api.common.AuthDetails
import io.iohk.atala.prism.api.connector.ConnectorAuthApi
import io.iohk.atala.prism.common.PrismSdkInternal
import io.iohk.atala.prism.credentials.PrismCredential
import io.iohk.atala.prism.crypto.EC
import io.iohk.atala.prism.crypto.keys.ECPrivateKey
import io.iohk.atala.prism.crypto.keys.ECPublicKey
import io.iohk.atala.prism.crypto.signature.ECSignature
import io.iohk.atala.prism.crypto.util.Random
import io.iohk.atala.prism.protos.AtalaMessage
import io.iohk.atala.prism.protos.CredentialAndVerificationNonce
import io.iohk.atala.prism.protos.util.Base64Utils
import pbandk.ByteArr
import pbandk.encodeToByteArray
import io.iohk.atala.prism.protos.CredentialOwnershipVerificationRequest as CredentialOwnershipVerificationRequestProto
import io.iohk.atala.prism.protos.CredentialOwnershipVerificationResponse as CredentialOwnershipVerificationResponseProto

@PrismSdkInternal
public object CredentialOwnershipVerificationService {

    /**
     * @param connector ConnectorAuthApi instance
     * @param credential credential ownership of which user wants to verify
     * @param connectionId ConnectionId of a contact user wants to ask for verification ownership
     * @param authDetails Authentication parameters to authenticate a request to connector
     */
    public suspend fun sendCredentialOwnershipVerificationRequest(
        connector: ConnectorAuthApi,
        credential: PrismCredential,
        connectionId: String,
        authDetails: AuthDetails
    ): CredentialOwnershipRequestReference {

        val nonce = Random.bytesOfLength(1024)

        val credentialOwnershipVerificationRequestMessage = AtalaMessage.Message.CredentialOwnershipVerificationRequest(
            CredentialOwnershipVerificationRequestProto(
                credentialAndVerificationNonce = CredentialAndVerificationNonce(
                    encodedCredential = credential.canonicalForm,
                    nonce = ByteArr(nonce)
                )
            )
        )

        val id = uuidOf(uuid4().bytes).toString()

        val message = AtalaMessage(message = credentialOwnershipVerificationRequestMessage)

        val sentMessageReference = connector.sendMessage(
            id = id,
            message = message,
            connectionId = connectionId,
            authDetails = authDetails
        )

        return CredentialOwnershipRequestReference(
            messageId = sentMessageReference.id,
            base64UrlNonce = Base64Utils.encode(nonce)
        )
    }

    /**
     * @param key Private key associated with public key inside a credential to sign the nonce
     * @param credential Credential, ownership of which holder is trying to prove
     * @param nonce Nonce that verifier sent for ownership verification
     */
    public fun solveCredentialOwnershipRequestChallenge(
        key: ECPrivateKey,
        credential: PrismCredential,
        nonce: ByteArray
    ): ECSignature {
        val credentialAndVerificationNonce = CredentialAndVerificationNonce(
            credential.canonicalForm,
            ByteArr(nonce)
        )

        return EC.signBytes(credentialAndVerificationNonce.encodeToByteArray(), key)
    }

    /**
     * @param connector ConnectorAuthApi instance
     * @param connectionId ConnectionId of a contact user wants to ask for verification ownership
     * @param authDetails Aauthentication parameters to authenticate a request to connector
     * @param replyTo AtalaMessage id of the received message that requested credential ownership verification
     * @param signature Prove of the ownership of the credential
     * @param nonce Nonce that was sent by the verifier to verify ownership
     */
    public suspend fun sendCredentialOwnershipVerificationResponse(
        connector: ConnectorAuthApi,
        connectionId: String,
        authDetails: AuthDetails,
        replyTo: String,
        signature: ECSignature,
        nonce: ByteArray
    ): CredentialOwnershipResponseReference {

        val credentialOwnershipVerificationResponseMessage =
            AtalaMessage.Message.CredentialOwnershipVerificationResponse(
                CredentialOwnershipVerificationResponseProto(
                    ByteArr(signature.getEncoded()),
                    ByteArr(nonce)
                )
            )

        val message = AtalaMessage(replyTo, credentialOwnershipVerificationResponseMessage)

        val sentMessageReference = connector.sendMessage(
            id = "",
            message = message,
            connectionId = connectionId,
            authDetails = authDetails
        )

        return CredentialOwnershipResponseReference(sentMessageReference.id)
    }

    /**
     * @param key Public key inside the credential used to verify a signature sent by the holder
     * @param credential Credential ownership of which holder is trying to verify
     * @param nonce Nonce that was sent by the verifier to verify ownership
     * @param signature Challenge solution sent by the holder
     */
    public fun verifyCredentialOwnership(
        key: ECPublicKey,
        credential: PrismCredential,
        nonce: ByteArray,
        signature: ECSignature
    ): Boolean {

        val credentialAndVerificationNonce = CredentialAndVerificationNonce(
            credential.canonicalForm,
            ByteArr(nonce)
        )

        return EC.verifyBytes(credentialAndVerificationNonce.encodeToByteArray(), key, signature)
    }
}

public data class CredentialOwnershipRequestReference(val messageId: String, val base64UrlNonce: String)

public data class CredentialOwnershipResponseReference(val messageId: String)
