package io.iohk.atala.prism.api

import io.iohk.atala.prism.credentials.CredentialBatchId
import io.iohk.atala.prism.credentials.PrismCredential
import io.iohk.atala.prism.credentials.content.CredentialContent
import io.iohk.atala.prism.credentials.json.JsonBasedCredential
import io.iohk.atala.prism.crypto.*
import io.iohk.atala.prism.crypto.keys.ECPrivateKey
import io.iohk.atala.prism.identity.CanonicalPrismDid
import io.iohk.atala.prism.identity.PrismDid
import io.iohk.atala.prism.protos.AtalaOperation
import io.iohk.atala.prism.protos.CredentialBatchData
import io.iohk.atala.prism.protos.IssueCredentialBatchOperation
import io.iohk.atala.prism.protos.SignedAtalaOperation
import kotlinx.serialization.json.*
import pbandk.ByteArr
import pbandk.encodeToByteArray
import kotlin.js.JsExport
import kotlin.jvm.JvmStatic

/**
 * @param root Merkle root that represents the credential batch
 * @param proofs list of Merkle proofs each of which can be combined with the root to verify that a
 * credential is in fact a part of the given batch, its size is guaranteed to be equal to the number of
 * credentials in the given batch
 */
@JsExport
public data class CredentialBatch(
    val root: MerkleRoot,
    val proofs: List<MerkleInclusionProof>
)

@JsExport
internal data class IssueBatchContext(
    val signedAtalaOperation: SignedAtalaOperation,
    val issuanceOperationHash: Sha256Digest,
    val batchId: CredentialBatchId,
    val credentialsAndProofs: List<CredentialInfo>,
    val merkleRoot: MerkleRoot
)

/**
 * @param signedCredential a signed credential
 * @param inclusionProof a Merkle proof that verifies that [signedCredential] belongs to a
 * credential batch (not a part of this data structure and is derived on the context)
 */
@JsExport
public data class CredentialInfo(
    val signedCredential: PrismCredential,
    val inclusionProof: MerkleInclusionProof
)

/**
 * @param subjectDid DID of the credential subject (i.e. entity receiving the credential)
 * @param content a free-form JSON content of the credential
 */
@JsExport
public data class CredentialClaim(
    val subjectDid: PrismDid,
    val content: JsonObject
)

@JsExport
public object CredentialBatches {
    /**
     * Computes a credential batch based on the list of provided signed credentials.
     *
     * @param signedCredentials array of signed credentials to include in the batch
     * @return credential batch consisting of the same amount of proofs as the size of
     * [signedCredentials]
     */
    @JvmStatic
    public fun batch(
        signedCredentials: List<PrismCredential>
    ): CredentialBatch {
        val merkleProofs = generateProofs(
            signedCredentials.map { it.hash() }
        )

        return CredentialBatch(merkleProofs.root, merkleProofs.proofs)
    }

    /**
     * Verifies that the signed credential belongs to the credential batch (represented by
     * [merkleRoot]) by using Merkle proof (represented by [inclusionProof]).
     *
     * @param signedCredential signed credential
     * @param merkleRoot Merkle root, typically taken out of [CredentialBatch] that is produced by
     * [batch]
     * @param inclusionProof Merkle tree inclusion proof for the given credential hash, typically
     * taken out of [CredentialBatch] that is produced by [batch]
     */
    @JvmStatic
    public fun verifyInclusion(
        signedCredential: PrismCredential,
        merkleRoot: MerkleRoot,
        inclusionProof: MerkleInclusionProof
    ): Boolean {
        return signedCredential.hash() == inclusionProof.hash &&
            verifyProof(merkleRoot, inclusionProof)
    }

    /**
     * Computes credential batch id based on issuer's DID and merkle root representing the batch.
     *
     * @param did credential batch issuer's DID
     * @param merkleRoot Merkle root, typically taken out of [CredentialBatch] that is produced by
     * [batch]
     */
    @JvmStatic
    public fun computeCredentialBatchId(did: PrismDid, merkleRoot: MerkleRoot): CredentialBatchId {
        val data = CredentialBatchData(did.suffix, ByteArr(merkleRoot.hash.value))
        return CredentialBatchId.fromDigest(Sha256.compute(data.encodeToByteArray()))
    }

    internal fun createBatchAtalaOperation(
        issuerDid: PrismDid,
        signingKeyId: String,
        issuingPrivateKey: ECPrivateKey,
        credentialsClaims: List<CredentialClaim>,
    ): IssueBatchContext {
        require(issuerDid is CanonicalPrismDid) {
            "DID should be in canonical form, found DID: $issuerDid"
        }

        val signedCredentials =
            credentialsClaims.map { credentialClaim ->
                JsonBasedCredential(
                    CredentialContent(
                        JsonObject(
                            mapOf(
                                Pair("id", JsonPrimitive(issuerDid.value)),
                                Pair("keyId", JsonPrimitive(signingKeyId)),
                                Pair(
                                    "credentialSubject",
                                    JsonObject(
                                        (
                                            credentialClaim.content +
                                                Pair(
                                                    "id",
                                                    JsonPrimitive(credentialClaim.subjectDid.value)
                                                )
                                            )
                                    )
                                ),
                            )
                        )
                    )
                ).sign(issuingPrivateKey)
            }

        val (credentialsMerkleRoot, credentialsMerkleProofs) = batch(signedCredentials)

        val issueCredentialBatchAtalaOperation =
            AtalaOperation(
                AtalaOperation.Operation.IssueCredentialBatch(
                    IssueCredentialBatchOperation(
                        CredentialBatchData(
                            issuerDid = issuerDid.suffix,
                            merkleRoot = ByteArr(credentialsMerkleRoot.hash.value)
                        )
                    )
                )
            )

        val credentialContexts = signedCredentials.zip(credentialsMerkleProofs).map {
            val (signedCredential, proof) = it
            CredentialInfo(signedCredential, proof)
        }

        return IssueBatchContext(
            signedAtalaOperation = ProtoUtils.signedAtalaOperation(
                issuingPrivateKey,
                signingKeyId,
                issueCredentialBatchAtalaOperation
            ),
            issuanceOperationHash = Sha256.compute(issueCredentialBatchAtalaOperation.encodeToByteArray()),
            batchId = computeCredentialBatchId(issuerDid, credentialsMerkleRoot),
            credentialsAndProofs = credentialContexts,
            merkleRoot = credentialsMerkleRoot
        )
    }
}
