/*
 * Copyright (c) 2015-2019 Rocket Partners, LLC
 * https://github.com/inversion-api
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.inversion.dynamodb;

import com.amazonaws.services.dynamodbv2.AmazonDynamoDB;
import com.amazonaws.services.dynamodbv2.document.*;
import com.amazonaws.services.dynamodbv2.document.api.QueryApi;
import com.amazonaws.services.dynamodbv2.document.api.ScanApi;
import com.amazonaws.services.dynamodbv2.document.spec.BatchGetItemSpec;
import com.amazonaws.services.dynamodbv2.document.spec.GetItemSpec;
import com.amazonaws.services.dynamodbv2.document.spec.QuerySpec;
import com.amazonaws.services.dynamodbv2.document.spec.ScanSpec;
import com.amazonaws.services.dynamodbv2.model.AttributeValue;
import com.amazonaws.services.dynamodbv2.model.KeysAndAttributes;
import io.inversion.Collection;
import io.inversion.Index;
import io.inversion.*;
import io.inversion.query.Projection;
import io.inversion.query.Select;
import io.inversion.query.Where;
import io.inversion.query.Group;
import io.inversion.query.Order;
import io.inversion.query.Page;
import io.inversion.query.From;
import io.inversion.query.Query;
import io.inversion.rql.*;
import io.inversion.utils.Utils;

import java.util.*;

/**
 * IMPLEMENTATION NOTE: Helpful DynamoDb Links
 * <ul>
 *  <li>https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Expressions.OperatorsAndFunctions.html
 *  <li>https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_Query.html
 *  <li>https://docs.aws.amazon.com/AWSJavaSDK/latest/javadoc/index.html
 *  <li>https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Introduction.html
 *  <li>https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/QueryingJavaDocumentAPI.html
 *  <li>https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Query.html#FilteringResults
 *  <li>https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Expressions.ExpressionAttributeNames.html
 *  <li>https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/LegacyConditionalParameters.KeyConditions.html
 *  <li>https://stackoverflow.com/questions/34349135/how-do-you-query-for-a-non-existent-null-attribute-in-dynamodb
 *  <li>https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/LegacyConditionalParameters.QueryFilter.html
 * </ul>
 * TODO: are all of these covered? EQ | NE | LE | LT | GE | GT | NOT_NULL | NULL | CONTAINS | NOT_CONTAINS | BEGINS_WITH | IN | BETWEEN
 */
public class DynamoDbQuery extends Query<DynamoDbQuery, DynamoDb, Select<Select<Select, DynamoDbQuery>, DynamoDbQuery>, From<From<From, DynamoDbQuery>, DynamoDbQuery>, Where<Where<Where, DynamoDbQuery>, DynamoDbQuery>, Group<Group<Group, DynamoDbQuery>, DynamoDbQuery>, Order<Order<Order, DynamoDbQuery>, DynamoDbQuery>, Page<Page<Page, DynamoDbQuery>, DynamoDbQuery>> {

    public static final Map<String, String> OPERATOR_MAP = new HashMap<>();
    public static final Map<String, String> FUNCTION_MAP = new HashMap<>();

    static {
        OPERATOR_MAP.put("eq", "=");
        OPERATOR_MAP.put("ne", "<>");
        OPERATOR_MAP.put("gt", ">");
        OPERATOR_MAP.put("ge", ">=");
        OPERATOR_MAP.put("lt", "<");
        OPERATOR_MAP.put("le", "<=");

        FUNCTION_MAP.put("w", "contains");
        FUNCTION_MAP.put("sw", "begins_with");
        FUNCTION_MAP.put("attribute_not_exists", "attribute_not_exists");
        FUNCTION_MAP.put("attribute_exists", "attribute_exists");

        //attribute_not_exists
        //attribute_exists

        //https://stackoverflow.com/questions/34349135/how-do-you-query-for-a-non-existent-null-attribute-in-dynamodb
        //FUNCTION_MAP.put("nn", "attribute_exists");//needs to be
        //FUNCTION_MAP.put("n", "attribute_not_exists");
    }

    com.amazonaws.services.dynamodbv2.document.Table dynamoTable = null;
    Index                                            index;

    Term partKey = null;
    Term sortKey = null;

    public DynamoDbQuery() {
    }

    public DynamoDbQuery(DynamoDb db, Collection table, List<Term> terms) {
        super(db, table, terms);
    }

    @Override
    protected Where createWhere() {
        Where where = new Where(this) {

            /**
             * Overridden to allow columns that are not defined in the schema to be included in queries
             */
            @Override
            protected boolean isInvalidColumn(Collection collection, String column) {
                boolean invalid = column.startsWith("_") || !column.matches("^[a-zA-Z0-9_]+$");
                //                if (invalid) {
                //                    System.out.println("Skipping column  " + column);
                //                }
                return invalid;
            }

        };
        where.withFunctions("_key", "eq", "ne", "gt", "ge", "lt", "le", "w", "wo", "sw", /* "ew" is not supported */ "nn", "n", "emp", "nemp", "in", "out", "and", "or", "not", "attribute_not_exists", "attribute_exists");
        return where;
    }

    protected boolean addTerm(String token, Term term) {
        index = null;

        if (term.hasToken("n", "nn", "emp", "nemp")) {
            if (term.size() > 1)
                throw ApiException.new400BadRequest("The n() and nn() functions only take one column name arg.");

            if (term.hasToken("n", "emp")) {
                Term eqNull        = Term.term(term.getParent(), "eq", term.getTerm(0), "null");
                Term attrNotExists = Term.term(null, "attribute_not_exists", term.getTerm(0));

                term = Term.term(term.getParent(), "or", attrNotExists, eqNull);
            } else if (term.hasToken("nn", "nemp")) {
                Term neNull     = Term.term(term.getParent(), "ne", term.getTerm(0), "null");
                Term attrExists = Term.term(null, "attribute_exists", term.getTerm(0));
                term = Term.term(term.getParent(), "and", attrExists, neNull);
            }
        }

        if (term.hasToken("like")) {
            String val     = term.getToken(1);
            int    firstWc = val.indexOf("*");
            if (firstWc > -1) {
                int wcCount = val.length() - val.replace("*", "").length();
                int lastWc  = val.lastIndexOf("*");
                if (wcCount > 2//
                        || (wcCount == 1 && firstWc != val.length() - 1)//
                        || (wcCount == 2 && !(firstWc == 0 && lastWc == val.length() - 1)))
                    throw ApiException.new400BadRequest("DynamoDb only supports a 'value*' or '*value*' wildcard formats which are equivalant to the 'sw' and 'w' operators.");

                boolean sw = firstWc == val.length() - 1;

                val = val.replace("*", "");
                term.getTerm(1).withToken(val);

                if (sw)
                    term.withToken("sw");
                else
                    term.withToken("w");
            }
        }

        if (term.hasToken("sw"))//sw (startswith) includes a implicit trailing wild card
        {
            String val = term.getTerm(1).getToken();
            while (val != null && val.endsWith("*")) {
                val = val.substring(0, val.length() - 1);
            }
            term.getTerm(1).withToken(val);
        }

        if (term.hasToken("wo")) {
            term.withToken("w");
            term = Term.term(null, "not", term);//getParent().replaceTerm(, newTerm)

        }

        return super.addTerm(token, term);
    }

    public com.amazonaws.services.dynamodbv2.document.Table getDynamoTable() {
        return dynamoTable;
    }

    public DynamoDbQuery withDynamoTable(com.amazonaws.services.dynamodbv2.document.Table dynamoTable) {
        this.dynamoTable = dynamoTable;
        return this;
    }

    public Results doSelect() throws ApiException {
        Results.LAST_QUERY = null;
        try {
            return doSelect0();
        } catch (Exception ex) {
            if (Results.LAST_QUERY != null) {
                System.out.println("Error after query: " + Results.LAST_QUERY);
                System.out.println(ex.getMessage());
            }

            Utils.rethrow(ex);
        }
        return null;
    }

    protected Results doSelect0() throws Exception {

        Results results = doSelect1();

        if (results.size() > 0) {
            if (this.index != null) {
                String type = this.index.getType();
                if (!type.equalsIgnoreCase(DynamoDb.PRIMARY_INDEX_TYPE)) {
                    Projection proj = this.index.getProjection();

                    if (proj != null) {

                        HashSet<String> includeColumns = new HashSet(getSelect().getIncludeColumns());
                        if (includeColumns.size() == 0) {
                            for (Property p : collection.getProperties()) {
                                includeColumns.add(p.getColumnName());
                            }
                        }

                        HashSet<String> projectionColumns = new HashSet();
                        for (String column : proj.keySet()) {
                            projectionColumns.add(column);
                        }

                        boolean columnsCovered = true;
                        for (String includeCol : includeColumns) {
                            if (!projectionColumns.contains(includeCol)) {
                                columnsCovered = false;
                                break;
                            }
                        }
                        if (!columnsCovered) {
                            results.withRows(batchGet(results.getRows()));
                        }
                    }
                }
            }
        }
        return results;
    }

    List<Map<String, Object>> batchGet(List<Map<String, Object>> rows){

        LinkedHashMap<String, Map<String, Object>> found = new LinkedHashMap<>();

        AmazonDynamoDB         dynamoClient = db.getDynamoClient();
        DynamoDB               dynamoDb     = new DynamoDB(dynamoClient);
        Index                  primaryIdx   = collection.getResourceIndex();
        BatchGetItemSpec       spec         = new BatchGetItemSpec();
        TableKeysAndAttributes tk           = new TableKeysAndAttributes(collection.getTableName());
        for (Map<String, Object> row : rows) {
            PrimaryKey pk = new PrimaryKey();
            for (Property prop : primaryIdx.getProperties()) {
                pk.addComponent(prop.getColumnName(), row.get(prop.getColumnName()));
            }
            tk.addPrimaryKey(pk);
            spec.withTableKeyAndAttributes(tk);
        }
        BatchGetItemOutcome out        = dynamoDb.batchGetItem(tk);
        List<Item>          items      = out.getTableItems().get(collection.getTableName());

        for (Item item : items) {
            Map map = item.asMap();
            String key = collection.encodeKeyFromColumnNames(map);
            found.put(key, map);
        }

        KeysAndAttributes unprocessed = out.getUnprocessedKeys().get(collection.getTableName());
        if(unprocessed != null){
            throw ApiException.new500InternalServerError("Unable to retrieve all items");
        }

        List<Map<String, Object>> results = new ArrayList<>();
        for(Map<String, Object> row : rows){
            String key = collection.encodeKeyFromColumnNames(row);
            results.add(found.get(key));
        }

        return results;
    }






    protected Results doSelect1() throws Exception {
        com.amazonaws.services.dynamodbv2.document.Index dynamoIndex = null;
        Results                                          result      = new Results(this);

        Object spec = getSelectSpec();

        Index index = calcIndex();
        if (!isDryRun() //
                && index != null //
                && index != collection.getIndex(DynamoDb.PRIMARY_INDEX_NAME)) {
            dynamoIndex = dynamoTable.getIndex(index.getName());
        }

        if (spec instanceof GetItemSpec) {
            GetItemSpec gis = (GetItemSpec) spec;

            StringBuilder debug = new StringBuilder("DynamoDb: ").append("GetItemSpec").append(index != null ? ":'" + index.getName() + "'" : "");
            debug.append(" key: ").append(gis.getKeyComponents());

            result.withTestQuery(debug.toString());
            Chain.debug(debug.toString());

            if (!isDryRun()) {
                Item item = dynamoTable.getItem(gis);
                if (item != null) {
                    result.withRow(item.asMap());
                }
            }
        } else if (spec instanceof QuerySpec) {
            QuerySpec qs = ((QuerySpec) spec);

            StringBuilder debug = new StringBuilder("DynamoDb: ").append("QuerySpec").append(index != null ? ":'" + index.getName() + "'" : "");

            if (qs.getMaxResultSize() != 100)
                debug.append(" maxResultSize=").append(qs.getMaxResultSize());

            if (qs.getNameMap() != null)
                debug.append(" nameMap=").append(qs.getNameMap());

            if (qs.getValueMap() != null)
                debug.append(" valueMap=").append(qs.getValueMap());

            if (qs.getFilterExpression() != null)
                debug.append(" filterExpression='").append(qs.getFilterExpression()).append("'");

            if (qs.getProjectionExpression() != null)
                debug.append(" projectionExpression='").append(qs.getProjectionExpression()).append("'");

            if (qs.getExclusiveStartKey() != null)
                debug.append(" exclusiveStartKey='").append(qs.getExclusiveStartKey());

            if (qs.getKeyConditionExpression() != null)
                debug.append(" keyConditionExpression='").append(qs.getKeyConditionExpression()).append("'");

            if (!getOrder().isAsc(0))
                debug.append(" scanIndexForward=false");

            result.withTestQuery(debug.toString());
            Chain.debug(debug.toString());

            if (!isDryRun()) {
                QueryApi                     queryApi    = dynamoIndex != null ? dynamoIndex : dynamoTable;
                ItemCollection<QueryOutcome> queryResult = queryApi.query(qs);
                for (Item item : queryResult) {
                    result.withRow(item.asMap());
                }

                result.withNext(after(index, queryResult.getLastLowLevelResult().getQueryResult().getLastEvaluatedKey()));
            }
        } else if (spec instanceof ScanSpec) {
            ScanSpec ss = ((ScanSpec) spec);

            StringBuilder debug = new StringBuilder("DynamoDb: ").append("ScanSpec").append(index != null ? ":'" + index.getName() + "'" : "");

            if (ss.getMaxResultSize() != 100)
                debug.append(" maxResultSize=").append(ss.getMaxResultSize());

            if (ss.getNameMap() != null)
                debug.append(" nameMap=").append(ss.getNameMap());

            if (ss.getValueMap() != null)
                debug.append(" valueMap=").append(ss.getValueMap());

            if (ss.getFilterExpression() != null)
                debug.append(" filterExpression='").append(ss.getFilterExpression()).append("'");

            if (ss.getProjectionExpression() != null)
                debug.append(" projectionExpression='").append(ss.getProjectionExpression()).append("'");

            if (ss.getExclusiveStartKey() != null)
                debug.append(" exclusiveStartKey='").append(ss.getExclusiveStartKey());

            result.withTestQuery(debug.toString());
            Chain.debug(debug.toString());

            if (!isDryRun()) {
                ScanApi                     scanApi    = dynamoIndex != null ? dynamoIndex : dynamoTable;
                ItemCollection<ScanOutcome> scanResult = scanApi.scan(ss);
                for (Item item : scanResult) {
                    result.withRow(item.asMap());
                }
                result.withNext(after(index, scanResult.getLastLowLevelResult().getScanResult().getLastEvaluatedKey()));
            }
        }

        return result;
    }

    protected List<Term> after(Index index, java.util.Map<String, AttributeValue> attrs) {
        if (attrs == null)
            return Collections.EMPTY_LIST;

        Term after = Term.term(null, "after");

        if (index != null) {
            after.withTerm(Term.term(after, index.getColumnName(0)));
            after.withTerm(Term.term(after, getValue(attrs.get(index.getColumnName(0))).toString()));

            if (index.getColumnName(1) != null) {
                after.withTerm(Term.term(after, index.getColumnName(1)));
                after.withTerm(Term.term(after, getValue(attrs.get(index.getColumnName(1))).toString()));
            }
        } else {
            for (String key : attrs.keySet()) {
                after.withTerm(Term.term(after, key));
                after.withTerm(Term.term(after, getValue(attrs.get(key)).toString()));
            }
        }
        return Utils.asList(after);
    }

    protected Object getValue(AttributeValue v) {
        if (v.getS() != null)
            return v.getS();
        if (v.getN() != null)
            return v.getN();
        if (v.getB() != null)
            return v.getB();
        if (v.getSS() != null)
            return v.getSS();
        if (v.getNS() != null)
            return v.getNS();
        if (v.getBS() != null)
            return v.getBS();
        if (v.getM() != null)
            return v.getM();
        if (v.getL() != null)
            return v.getL();
        if (v.getNULL() != null)
            return v.getNULL();
        if (v.getBOOL() != null)
            return v.getBOOL();

        throw ApiException.new500InternalServerError("Unable to get value from AttributeValue: {}", v);
    }

    /**
     * @return the best fit index to use for the query based on the params supplied
     */
    protected Index calcIndex() {
        //if the users requested a sort, you need to find an index with that sort: String sortBy = order.getProperty(0);
        String sortBy = order.getProperty(0);

        Term after = getPage().getAfter();
        if (after != null) {
            String afterHashKeyCol = getCollection().getProperty(after.getToken(0)).getColumnName();
            String afterSortKeyCol = after.size() > 2 ? getCollection().getProperty(after.getToken(2)).getColumnName() : null;

            for (Index idx : collection.getIndexes()) {
                if (Utils.equal(idx.getColumnName(0), afterHashKeyCol)//
                        && Utils.equal(idx.getColumnName(1), afterSortKeyCol)) {
                    index = idx;

                    partKey = findTerm(afterHashKeyCol, "eq");

                    if (partKey == null)
                        continue;

                    if (afterSortKeyCol != null) {
                        sortKey = findTerm(afterSortKeyCol, "eq");
                        if (sortKey == null) {
                            //--  WB 20200805 - there are a limited number of operators supported for sort keys in KeyCondition Expressions.
                            //--  @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_Query.html#DDB-Query-request-KeyConditionExpression
                            sortKey = findTerm(afterSortKeyCol, "gt", "ne", "gt", "ge", "lt", "le", "sw", "ew", "in", "out", "n", "nn");
                            if (sortKey != null && Utils.in(sortKey.getToken(), "in", "out")) {
                                partKey = null;
                                sortKey = null;
                                continue;
                            }
                        }
                    }

                    break;
                }
            }

            if (sortBy != null && (!sortBy.equalsIgnoreCase(afterSortKeyCol))) {
                //TODO make test
                throw ApiException.new400BadRequest("The requested sort key does not match the supplied 'after' continuation token.");
            }
        }

        if (index == null) {
            Index foundIndex   = null;
            Term  foundPartKey = null;
            Term  foundSortKey = null;

            for (io.inversion.Index idx : getCollection().getIndexes()) {
                Index index = idx;

                String partCol = index.getColumnName(0);//getHashKey().getColumnName();
                String sortCol = index.getColumnName(1);//index.getSortKey() != null ? index.getSortKey().getColumnName() : null;

                if (sortBy != null && !sortBy.equalsIgnoreCase(sortCol))
                    continue; //incompatible index. if a sort was requested, can't choose an index that has a different sort

                Term partKey = findTerm(partCol, "eq");

                if (partKey == null && sortBy == null)
                    continue;

                Term sortKey = findTerm(sortCol, "eq");

                if (sortKey == null) {
                    //--  WB 20200805 - there are a limited number of operators supported for sort keys in KeyCondition Expressions.
                    //--  @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_Query.html#DDB-Query-request-KeyConditionExpression
                    sortKey = findTerm(sortCol, "gt", "ne", "gt", "ge", "lt", "le", "sw", "ew", "in", "out", "n", "nn");
                    if (sortKey != null && Utils.in(sortKey.getToken(), "in", "out")) {
                        continue;
                    }
                }

                boolean use = false;
                if (foundPartKey == null && partKey != null)
                    use = true;

                else if (sortKey == null && foundSortKey != null)
                    use = false; //if you already have an index with a sort key match, don't replace it

                else if (foundIndex == null //
                        || (sortKey != null && foundSortKey == null) //
                        || (sortKey != null && sortKey.hasToken("eq") && !foundSortKey.hasToken("eq"))) //the new sort key has an equality condition
                    use = true;

                if (use) {
                    foundIndex = index;
                    foundPartKey = partKey;
                    foundSortKey = sortKey;
                }
            }

            if (sortBy != null && foundIndex == null) {
                //TODO: create test case to trigger this exception
                throw ApiException.new400BadRequest("Unable to find valid index to query.  The requested sort field '{}' must be the sort key of the primary index, the sort key of a global secondary index, or a local secondary secondary index.", sortBy);
            }

            if (foundPartKey == null && sortBy != null && !order.isAsc(0)) {
                //an inverse/descending sort can only be run on a QuerySpec which requires a partition key.
                throw ApiException.new400BadRequest("Unable to find valid index to query.  A descending sort on '{}' is only possible when a partition key value is supplied.", sortBy);
            }

            this.index = foundIndex;
            this.partKey = foundPartKey;
            this.sortKey = foundSortKey;
        }
        return index;
    }

    /**
     * Find a valid sort key.
     * <p>
     * IMPLEMENTATION NOTE - WB 20200805 - there are a limited number of operators supported for sort keys in KeyCondition Expressions.
     *
     * @param sortCol the name of a range key column
     * @return a sort key Term if supplied
     * @see <a href="https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_Query.html#DDB-Query-request-KeyConditionExpression">Dynamo Key Condition Expressions</a>
     */
    Term findSortKeyTerm(String sortCol) {
        Term sortKey = findTerm(sortCol);

        if (sortKey != null) {
            if (!Utils.in(sortKey.getToken(), "eq", "gt", "ne", "gt", "ge", "lt", "le", "sw"))
                return null;
        }
        return sortKey;

    }

    /**
     * Finds the primary or a secondary index to use based on
     * what parameters were passed in.
     *
     * @return a term referencing the partition key column
     * @see #calcIndex()
     */
    public Term getPartKey() {
        if (index == null) {
            calcIndex();
        }
        return partKey;
    }

    public Term getSortKey() {
        if (index == null) {
            calcIndex();
        }
        return sortKey;
    }

    public Object getSelectSpec() {
        Map nameMap  = new HashMap<>();
        Map valueMap = new HashMap<>();

        StringBuilder keyExpr    = new StringBuilder();
        StringBuilder filterExpr = new StringBuilder();

        Index index = calcIndex();

        if (index != null //
                && Utils.equal(DynamoDb.PRIMARY_INDEX_NAME, index.getName()) //
                && partKey != null //
                && sortKey != null //
                && sortKey.hasToken("eq") //
                && sortKey.getTerm(1).isLeaf())//sortKey is a single eq expression not a logic expr
        {
            String partKeyCol = partKey.getToken(0);
            String type       = collection.getProperty(partKeyCol).getType();
            Object partKeyVal = getDb().castJsonInput(type, partKey.getToken(1));

            String sortKeyCol = sortKey.getToken(0);
            Object sortKeyVal = getDb().castJsonInput(collection.getProperty(sortKeyCol).getType(), sortKey.getToken(1));

            return new GetItemSpec().withPrimaryKey(partKeyCol, partKeyVal, sortKeyCol, sortKeyVal);
        }

        if (partKey != null) {
            toString(keyExpr, partKey, nameMap, valueMap);
        }

        if (sortKey != null) {
            toString(keyExpr, sortKey, nameMap, valueMap);
        }

        for (Term term : getWhere().getTerms()) {
            if (term == partKey || term == sortKey)
                continue;
            toString(filterExpr, term, nameMap, valueMap);
        }

        boolean doQuery = partKey != null && partKey.getTerm(1).isLeaf();

        int pageSize = getPage().getPageSize();

        String projectionExpression = null;
        List   columns              = getSelect().getIncludeColumns();
        if (columns.size() > 0)
            projectionExpression = Utils.implode(",", columns);

        if (doQuery) {
            QuerySpec querySpec = new QuerySpec();//
            //querySpec.withMaxPageSize(pageSize);
            querySpec.withMaxResultSize(pageSize);
            querySpec.withScanIndexForward(getOrder().isAsc(0));

            //-- https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/SQLtoNoSQL.ReadData.Query.html
            //-- You can use Query with any table that has a composite primary key (partition key and sort key).
            //-- You must specify an equality condition for the partition key, and you can optionally provide another condition for the sort key.

            Term after = getPage().getAfter();
            if (after != null) {
                Property afterHashKeyCol = getCollection().getProperty(after.getToken(0));
                Property afterSortKeyCol = after.size() > 2 ? getCollection().getProperty(after.getToken(2)) : null;

                if (afterHashKeyCol == null || (after.size() > 2 && afterSortKeyCol == null))
                    throw ApiException.new400BadRequest("Invalid column in 'after' key: {}", after);

                Object hashValue = db.castJsonInput(afterHashKeyCol, after.getToken(1));
                Object sortValue = afterSortKeyCol != null ? db.castJsonInput(afterSortKeyCol, after.getToken(3)) : null;

                if (afterSortKeyCol != null) {
                    querySpec.withExclusiveStartKey(afterHashKeyCol.getColumnName(), hashValue, afterSortKeyCol.getColumnName(), sortValue);
                } else {
                    querySpec.withExclusiveStartKey(afterHashKeyCol.getColumnName(), hashValue);
                }
            }

            if (!Utils.empty(projectionExpression)) {
                querySpec.withProjectionExpression(projectionExpression);
            }

            if (keyExpr.length() > 0) {
                querySpec.withKeyConditionExpression(keyExpr.toString());
            }

            if (filterExpr.length() > 0) {
                querySpec.withFilterExpression(filterExpr.toString());
            }

            if (nameMap.size() > 0) {
                querySpec.withNameMap(nameMap);
            }

            if (valueMap.size() > 0) {
                querySpec.withValueMap(valueMap);
            }

            return querySpec;
        } else {
            ScanSpec scanSpec = new ScanSpec();
            //scanSpec.withMaxPageSize(pageSize);
            scanSpec.withMaxResultSize(pageSize);

            Term after = getPage().getAfter();
            if (after != null) {
                Property afterHashKeyCol = getCollection().getProperty(after.getToken(0));
                Property afterSortKeyCol = after.size() > 2 ? getCollection().getProperty(after.getToken(2)) : null;

                if (afterHashKeyCol == null || (after.size() > 2 && afterSortKeyCol == null))
                    throw ApiException.new400BadRequest("Invalid column in 'after' key: {}");

                Object hashValue = db.castJsonInput(afterHashKeyCol, after.getToken(1));
                Object sortValue = afterSortKeyCol != null ? db.castJsonInput(afterSortKeyCol, after.getToken(3)) : null;

                if (afterSortKeyCol != null) {
                    scanSpec.withExclusiveStartKey(afterHashKeyCol.getColumnName(), hashValue, afterSortKeyCol.getColumnName(), sortValue);
                } else {
                    scanSpec.withExclusiveStartKey(afterHashKeyCol.getColumnName(), hashValue);
                }
            }

            if (!Utils.empty(projectionExpression)) {
                scanSpec.withProjectionExpression(projectionExpression);
            }

            if (filterExpr.length() > 0) {
                scanSpec.withFilterExpression(filterExpr.toString());
            }
            if (nameMap.size() > 0) {
                scanSpec.withNameMap(nameMap);
            }

            if (valueMap.size() > 0) {
                scanSpec.withValueMap(valueMap);
            }

            return scanSpec;
        }
    }

    String toString(StringBuilder buff, Term term, Map nameMap, Map valueMap) {
        space(buff);

        String lc   = term.getToken().toLowerCase();
        String op   = OPERATOR_MAP.get(lc);
        String func = FUNCTION_MAP.get(lc);

        if (term.hasToken("not")) {
            if (buff.length() > 0)
                space(buff).append("and ");

            buff.append("(NOT ").append(toString(new StringBuilder(), term.getTerm(0), nameMap, valueMap)).append(")");
        } else if (term.hasToken("and", "or")) {
            buff.append("(");
            for (int i = 0; i < term.getNumTerms(); i++) {
                buff.append(toString(new StringBuilder(), term.getTerm(i), nameMap, valueMap));
                if (i < term.getNumTerms() - 1)
                    space(buff).append(term.getToken()).append(" ");
            }
            buff.append(")");
        } else if (term.hasToken("in", "out")) {
            String col     = term.getToken(0);
            String nameKey = "#var" + (nameMap.size() + 1);
            nameMap.put(nameKey, col);

            if (buff.length() > 0)
                space(buff).append("and ");

            buff.append("(");
            buff.append(term.hasToken("out") ? "NOT " : "");
            buff.append(nameKey).append(" IN (");
            for (int i = 1; i < term.size(); i++) {
                if (i > 1)
                    buff.append(", ");

                buff.append(toString(new StringBuilder(), term.getTerm(i), nameMap, valueMap));

            }
            buff.append("))");
        } else if (op != null) {
            String col = term.getToken(0);

            String nameKey = "#var" + (nameMap.size() + 1);
            nameMap.put(nameKey, col);

            String expr = toString(new StringBuilder(), term.getTerm(1), nameMap, valueMap);

            if (buff.length() > 0)
                space(buff).append("and ");

            buff.append("(").append(nameKey).append(" ").append(op).append(" ").append(expr).append(")");
        } else if (func != null) {
            if (buff.length() > 0)
                space(buff).append("and ");

            String col = term.getToken(0);

            String nameKey = "#var" + (nameMap.size() + 1);
            nameMap.put(nameKey, col);

            if (term.size() > 1) {
                String expr = toString(new StringBuilder(), term.getTerm(1), nameMap, valueMap);
                space(buff).append(func).append("(").append(nameKey).append(",").append(expr).append(")");
            } else {
                space(buff).append(func).append("(").append(nameKey).append(")");
            }
        } else if (term.isLeaf()) {
            String   colName = term.getParent().getToken(0);
            Property col     = collection.getProperty(colName);
            Object   value   = db.castJsonInput(col, term.getToken());

            if ("null".equalsIgnoreCase(value + ""))
                value = null;

            String key = ":val" + (valueMap.size() + 1);
            valueMap.put(key, value);

            space(buff).append(key);
        }

        //System.out.println("TOSTRING: " + term + " -> '" + buff + "'" + " - " + nameMap + " - " + valueMap);
        return buff.toString();
    }

    StringBuilder space(StringBuilder buff) {
        if (buff.length() > 0 && buff.charAt(buff.length() - 1) != ' ')
            buff.append(' ');

        return buff;
    }

}
