/*
 * Copyright 2022 Jeremy KUHN
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.inverno.mod.security.authentication;

/**
 * <p>
 * Credentials used to authenticate an entity based on a verifiable token.
 * </p>
 *
 * <p>
 * A token is shared between the entity and the authenticator and must be verifiable during the authentication process. A token might be generated by an authority trusted by both the entity and the
 * application, each party being then able to obtain or authenticate the token to the trusted authority. Other approaches might also include symetric or asymetric cryptographic processing.
 * </p>
 *
 * <p>
 * In order to provide proper security, implementations should choose properly secured tokens which are collision-free and hardly forgeable.
 * </p>
 * 
 * @author <a href="mailto:jeremy.kuhn@inverno.io">Jeremy Kuhn</a>
 * @since 1.5
 */
public class TokenCredentials implements Credentials {
	
	/**
	 * The token value
	 */
	private final String token;

	/**
	 * <p>
	 * Creates a token credentials with the specified value.
	 * </p>
	 * 
	 * @param token a token
	 */
	public TokenCredentials(String token) {
		this.token = token;
	}

	/**
	 * <p>
	 * Returns the token value.
	 * </p>
	 * 
	 * @return a token
	 */
	public String getToken() {
		return token;
	}
}
