package io.inai.android_sdk

import android.annotation.SuppressLint
import android.graphics.Bitmap
import android.net.Uri
import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.webkit.*
import androidx.fragment.app.Fragment
import com.google.android.material.progressindicator.CircularProgressIndicator
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import org.json.JSONObject

enum class InaiViewMode {
    Payment, AddPaymentMethod, PayWithPaymentMethod, MakePayment, GetCardInfo, ValidateFields,
}

class InaiCheckoutFragment: Fragment() {

    private lateinit var webView: WebView
    private lateinit var loadingIndicator: CircularProgressIndicator
    private lateinit var _delegate: Any
    private lateinit var _config: InaiConfig
    private var _url: String = ""
    private var inaiResult: InaiResult = InaiResult()
    private var _viewMode : InaiViewMode = InaiViewMode.Payment
    private var _extraArgs : String = ""
    private var _webViewLoaded: Boolean = false
    private var _webViewDismissed: Boolean = false
    private var _ignoreDelegateCallback: Boolean = false
    var cardIdentificationDelegate: InaiCardIdentificationDelegate? = null

    companion object {
        fun newInstance(url: String,
                        config: InaiConfig,
                        delegate: Any,
                        viewMode: InaiViewMode,
                        extraArgs: String = ""
        ): InaiCheckoutFragment {
            val checkoutFragment = InaiCheckoutFragment()
            checkoutFragment._url = url
            checkoutFragment._delegate = delegate
            checkoutFragment._config = config
            checkoutFragment._viewMode = viewMode
            checkoutFragment._extraArgs = extraArgs
            return checkoutFragment
        }
    }

    @SuppressLint("SetJavaScriptEnabled")
    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View {
        val rootView: View = inflater.inflate(R.layout.inai_checkout_fragment, container, false)

        //  Load the ui components
        loadingIndicator = rootView.findViewById(R.id.loading_indicator)
        webView = rootView.findViewById(R.id.webView)
        webView.settings.javaScriptEnabled = true
        //  Register the js handler
        webView.addJavascriptInterface(this, "inaiHandler")

        webView.webViewClient = object : WebViewClient() {
            override fun shouldOverrideUrlLoading(view: WebView?,  request: WebResourceRequest?): Boolean {

                if (request != null) {
                    val requestUrl = request.url
                    val urlString = requestUrl.toString()
                    val uri: Uri = Uri.parse(urlString)
                    val queryParams = uri.queryParameterNames
                    val queryParamsJson = JSONObject()
                    for (item in queryParams) {
                        queryParamsJson.put(item, requestUrl.getQueryParameter(item))
                    }

                    if (
                        (queryParamsJson.has("transaction_id")
                                || queryParamsJson.has("transaction-id") )
                        && queryParamsJson.has("status")
                    ) {

                        var transactionId: String? = ""
                        if (queryParamsJson.has("transaction_id")){
                            transactionId = queryParamsJson.get("transaction_id") as? String
                        } else if(queryParamsJson.has("transaction-id")){
                            transactionId = queryParamsJson.get("transaction-id") as? String
                        }
                        val status = queryParamsJson.get("status") as? String

                        if(!status.isNullOrEmpty() && !transactionId.isNullOrEmpty() ) {
                            //  We're looking for the transaction redirect url eg: https://payments.inai.io/v1/payment-status
                            //  ?transaction_id=e9a5fac1-dcc4-4cfc-a8ed-142bbdaa7ef1
                            //  &status=success
                            if (status == "success") {
                                inaiResult.status = InaiStatus.Success
                            } else {
                                inaiResult.status = InaiStatus.Failed
                            }

                            inaiResult.data = queryParamsJson

                            //  Don't let the url load and finish the page
                            dismissWebView()
                            return true
                        }
                    }
                }

                //  Don't override other urls
                return false
            }

            override fun onPageStarted(view: WebView?, url: String?, favicon: Bitmap?) {
                loadingIndicator.visibility = View.VISIBLE
            }

            override fun onPageFinished(view: WebView?, url: String?) {
                if (!_webViewLoaded) {
                    //  Call only once
                    loadingIndicator.visibility = View.GONE

                    //  Initialise the inai sdk with json data
                    webView.evaluateJavascript(
                        "window.webkit = { messageHandlers: { inaiHandler: window.inaiHandler} };",
                        null
                    )

                    //  Initialise the Inai SDK
                    val jsonString = Json.encodeToString(_config)
                    var initFn = ""

                    when (_viewMode) {
                        InaiViewMode.Payment -> {
                            initFn = "initPayment"
                        }
                        InaiViewMode.AddPaymentMethod -> {
                            initFn = "initAddPaymentMethod"
                        }
                        InaiViewMode.PayWithPaymentMethod -> {
                            initFn = "initPayWithPaymentMethod"
                        }
                        InaiViewMode.MakePayment -> {
                            initFn = "initMakePayment"
                        }
                        InaiViewMode.GetCardInfo -> {
                            initFn = "initGetCardInfo"
                        }
                        InaiViewMode.ValidateFields -> {
                            initFn = "initValidateFields"
                        }
                    }

                    if (_extraArgs.isNotEmpty()){
                        _extraArgs = ",$_extraArgs"
                    }

                    var jsCode = "let jsArgs = $jsonString;"

                    if (cardIdentificationDelegate != null) {
                        //  Add onCardIdentification callback only if the delegate is present
                        jsCode += " jsArgs.onCardIdentification = (cardInfo) => { notifyApplication(convertToJSONString({cardInfo}));};"
                    }

                    jsCode += "$initFn(jsArgs$_extraArgs);"
                    webView.evaluateJavascript(jsCode, null)
                    _webViewLoaded = true
                }
            }

            override fun onReceivedError(
                view: WebView?,
                request: WebResourceRequest?,
                error: WebResourceError?
            ) {
                loadingIndicator.visibility = View.GONE
                //  We got an error..
                /*
                if (!_webViewDismissed) {
                    var errorString = "An unknown error has occurred"
                    if (error != null && error.description.isNotEmpty()) {
                        errorString = error.description.toString()
                    }

                    inaiResult.data.put("message", errorString)
                    dismissWebView()
                }*/

            }
        }

        //  Fire off the WebView
        webView.loadUrl(_url)
        return rootView
    }

    /*
    //  Sample response from stripe
    {
      "transaction_id": "dc3292a4-2e66-4eed-84f9-bee529a82453",
      "status": "success",
      "inai_status": "success"
    }
     */
    @JavascriptInterface
    fun postMessage(message: String) {
        if(message.isNotEmpty()) {
            //  handle message from android
            val jsonObj = JSONObject(message)

            if (jsonObj.has("cardInfo")) {
                activity?.runOnUiThread {
                    this.cardIdentificationDelegate?.onCardIdentification(jsonObj.getJSONObject("cardInfo"))
                }
                return
            }

            val inaiStatus = jsonObj.getString("inai_status")
            if (inaiStatus == "success") {
                inaiResult.status = InaiStatus.Success
            } else {
                inaiResult.status = InaiStatus.Failed
            }

            jsonObj.remove("inai_status")
            inaiResult.data = jsonObj
            dismissWebView()
        }
    }

    fun evalJS(jsStr: String) {
        webView.evaluateJavascript( jsStr, null)
    }

    fun dismissWebView(ignoreDelegateCallback: Boolean = false) {
        //  Cleanup
        _webViewDismissed = true
        _ignoreDelegateCallback = ignoreDelegateCallback
        //  Call on main thread
        webView.post { webView.stopLoading() }

        //  dismiss the view
        requireActivity().supportFragmentManager
            .findFragmentByTag("inai_checkout_30d9cb17_249c_4fb6")?.let { fragment ->
                requireActivity().supportFragmentManager.beginTransaction().remove(fragment).commit()
            }
    }

    override fun onDestroy() {
        super.onDestroy()

        if (_ignoreDelegateCallback) {
            //  Do not process any delegate callbacks
            return
        }

        //  View has been closed
        //  Notify the delegate
        when(_viewMode) {
            InaiViewMode.Payment -> {
                val result = InaiPaymentResult()
                result.data = inaiResult.data
                result.status = InaiPaymentStatus.valueOf( inaiResult.status.toString() )
                (_delegate as InaiCheckoutDelegate).paymentFinished(result)
            }
            InaiViewMode.AddPaymentMethod -> {
                val result = InaiPaymentMethodResult()
                result.data = inaiResult.data
                result.status = InaiPaymentMethodStatus.valueOf( inaiResult.status.toString() )
                (_delegate as InaiPaymentMethodDelegate).paymentMethodSaved(result)
            }
            InaiViewMode.PayWithPaymentMethod -> {
                val result = InaiPaymentResult()
                result.data = inaiResult.data
                result.status = InaiPaymentStatus.valueOf( inaiResult.status.toString() )
                (_delegate as InaiCheckoutDelegate).paymentFinished(result)
            }
            InaiViewMode.MakePayment -> {
                val result = InaiPaymentResult()
                result.data = inaiResult.data
                result.status = InaiPaymentStatus.valueOf( inaiResult.status.toString() )
                (_delegate as InaiCheckoutDelegate).paymentFinished(result)
            }
            InaiViewMode.GetCardInfo -> {
                val result = InaiCardInfoResult()
                result.data = inaiResult.data
                result.status = InaiCardInfoStatus.valueOf( inaiResult.status.toString() )
                (_delegate as InaiCardInfoDelegate).cardInfoFetched(result)
            }
            InaiViewMode.ValidateFields -> {
                val result = InaiValidateFieldsResult()
                result.data = inaiResult.data
                result.status = InaiValidateFieldsStatus.valueOf( inaiResult.status.toString() )
                (_delegate as InaiValidateFieldsDelegate).fieldsValidationFinished(result)
            }
        }
    }
}