package io.inai.android_sdk

import android.content.Context
import androidx.fragment.app.FragmentActivity
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import org.json.JSONObject


enum class InaiStatus {
    Success, Failed, Canceled
}

enum class InaiPaymentStatus {
    Success, Failed, Canceled
}

enum class InaiPaymentMethodStatus {
    Success, Failed, Canceled
}

enum class InaiCardInfoStatus {
    Success, Failed
}

enum class InaiValidateFieldsStatus {
    Success, Failed
}

data class InaiResult(
    var status: InaiStatus = InaiStatus.Canceled,
    var data: JSONObject = JSONObject()
)

data class InaiPaymentResult(
    var status: InaiPaymentStatus = InaiPaymentStatus.Canceled,
    var data: JSONObject = JSONObject()
)

data class InaiPaymentMethodResult(
    var status: InaiPaymentMethodStatus = InaiPaymentMethodStatus.Canceled,
    var data: JSONObject = JSONObject()
)

data class InaiCardInfoResult(
    var status: InaiCardInfoStatus = InaiCardInfoStatus.Failed,
    var data: JSONObject = JSONObject()
)

data class InaiValidateFieldsResult(
    var status: InaiValidateFieldsStatus = InaiValidateFieldsStatus.Failed,
    var data: JSONObject = JSONObject()
)

interface InaiCheckoutDelegate {
    fun paymentFinished(result: InaiPaymentResult)
}

interface InaiPaymentMethodDelegate {
    fun paymentMethodSaved(result: InaiPaymentMethodResult)
}

interface InaiCardInfoDelegate {
    fun cardInfoFetched(result: InaiCardInfoResult)
}

interface InaiValidateFieldsDelegate {
    fun fieldsValidationFinished(result: InaiValidateFieldsResult)
}

/// Delegate for Credit Card Info Indenfication
interface InaiCardIdentificationDelegate {
    /// cardInfo: => { brand: "visa|mastercard", type: "credit|debit", issuerName: "<issuer_name>", category: "platinum", country: "USA|IND" }
    fun onCardIdentification(cardInfo: JSONObject)
}


class InaiCheckout()  {
    private lateinit var _config: InaiConfig
    private val sdkHostUrl = BuildConfig.InaiSdkHostUrl

    var cardIdentificationDelegate: InaiCardIdentificationDelegate? = null
    private lateinit var _checkoutFragment: InaiCheckoutFragment


    constructor(config: InaiConfig) : this() {
        if ( (config.orderId.isNullOrBlank()
                    && config.orderId.isNullOrBlank() )
            || config.token.isBlank()){
            //  Invalid config, reject and throw error
            throw Exception("Invalid config")
        }
        _config = config
    }

    fun presentCheckout(context: Context, delegate: Any) {
        //  Initiate the checkout dialog
        val checkoutFragment = InaiCheckoutFragment.newInstance(sdkHostUrl, _config, delegate, InaiViewMode.Payment)
        showCheckoutDialog(context, checkoutFragment)
    }

    fun addPaymentMethod(type: String, context: Context, delegate: Any) {
        //  Initiate the checkout dialog
        val extraArgs = "'$type'"
        val checkoutDialog = InaiCheckoutFragment.newInstance(
            sdkHostUrl,
            _config,
            delegate,
            InaiViewMode.AddPaymentMethod,
            extraArgs
        )
        showCheckoutDialog(context, checkoutDialog)
    }

    fun presentPayWithPaymentMethod(paymentMethodId: String,  context: Context, delegate: Any) {
        //  Initiate the checkout dialog
        val extraArgs = "'$paymentMethodId'"
        val checkoutFragment = InaiCheckoutFragment.newInstance(
            sdkHostUrl,
            _config,
            delegate,
            InaiViewMode.PayWithPaymentMethod,
            extraArgs
        )
        showCheckoutDialog(context, checkoutFragment)
    }

    fun makePayment(paymentMethodOption: String, paymentDetails: JSONObject, context: Context, delegate: Any) {
        val paymentDetailsJson = paymentDetails.toString()
        val extraArgs = "'${paymentMethodOption}',${paymentDetailsJson}"
        val checkoutFragment = InaiCheckoutFragment.newInstance(
            sdkHostUrl,
            _config,
            delegate,
            InaiViewMode.MakePayment,
            extraArgs
        )
        showCheckoutDialog(context, checkoutFragment)
    }

    fun getCardInfo(cardNumber: String, context: Context, delegate: Any) {
        //  Show the view as invisible..
        val extraArgs = "'$cardNumber'"
        val checkoutFragment = InaiCheckoutFragment.newInstance(
            sdkHostUrl,
            _config,
            delegate,
            InaiViewMode.GetCardInfo,
            extraArgs
        )
        showInvisibleCheckout(context, checkoutFragment)
    }

    fun validateFields(paymentMethodOption: String, paymentDetails: JSONObject, context: Context, delegate: Any) {
        val paymentDetailsJson = paymentDetails.toString()
        val extraArgs = "'${paymentMethodOption}',${paymentDetailsJson}"
        val checkoutFragment = InaiCheckoutFragment.newInstance(
            sdkHostUrl,
            _config,
            delegate,
            InaiViewMode.ValidateFields,
            extraArgs
        )
        showInvisibleCheckout(context, checkoutFragment)
    }

    private fun showCheckoutDialog(context: Context, checkoutFragment: InaiCheckoutFragment) {
        checkoutFragment.cardIdentificationDelegate = cardIdentificationDelegate
        this._checkoutFragment = checkoutFragment
        val fragmentActivity = context as FragmentActivity
        val checkoutDialog = InaiCheckoutDialog.newInstance(checkoutFragment)
        checkoutDialog.show(fragmentActivity.supportFragmentManager, "inai_checkout_30d9cb17_249c_4fb6")
    }

    private fun showInvisibleCheckout(context: Context, checkoutFragment: InaiCheckoutFragment) {
        checkoutFragment.cardIdentificationDelegate = cardIdentificationDelegate
        this._checkoutFragment = checkoutFragment
        val fragmentActivity = context as FragmentActivity
        val fragmentTransaction = fragmentActivity.supportFragmentManager.beginTransaction()
        val checkoutFragmentTransaction = fragmentTransaction.add(checkoutFragment, "inai_checkout_30d9cb17_249c_4fb6")
        checkoutFragmentTransaction.commit()
    }

    /// Hide inai widget without triggering a callback
    fun remove() {
        this._checkoutFragment.dismissWebView(true)
    }

    /// Disable Pay Button
    fun disableCTA(paymentMethods: Array<String>) {
        val paymentMethodsJSON = Json.encodeToString(paymentMethods)
        val jsString = "window.inai.disableCTA($paymentMethodsJSON);"
        this._checkoutFragment.evalJS(jsString)
    }

    /// Enable Pay Button
    fun enableCTA(paymentMethods: Array<String>) {
        val paymentMethodsJSON = Json.encodeToString(paymentMethods)
        val jsString = "window.inai.enableCTA($paymentMethodsJSON);"
        this._checkoutFragment.evalJS(jsString)
    }

    /// Update Pay Button
    fun updateCTA(ctaConfig: InaiCTAConfig) {
        val ctaConfigJson = Json.encodeToString(ctaConfig)
        val jsString = "window.inai.updateCTA($ctaConfigJson);"
        this._checkoutFragment.evalJS(jsString)
    }
}