package io.inai.android_sdk.helpers

import android.app.ActivityManager
import android.content.Context
import android.content.Context.ACTIVITY_SERVICE
import android.content.Context.BATTERY_SERVICE
import android.os.BatteryManager
import android.os.Build
import android.provider.Settings
import io.inai.android_sdk.InaiBaseConfig
import io.inai.android_sdk.InaiConfig
import io.inai.android_sdk.InaiConstants
import java.io.File

class InaiBaseUtils {

    fun setup(context: Context) {
        _context = context
    }
    @kotlinx.serialization.Serializable
    data class BreadcrumbsValues(val message: String, val category: String, val level: String, val timestamp: Long, val type: String, val data: BreadcrumbsData?)

    @kotlinx.serialization.Serializable
    data class BreadcrumbsData(val state:String, val screen: String)

    companion object {
        lateinit var _context: Context
        private val breadcrumbsValuesList = mutableListOf<BreadcrumbsValues>()

        fun getAvailableMemoryInfo(): Long {
            val availMemory = getMemInfo().availMem
            return availMemory
        }

        fun getTotalMemoryInfo(): Long {
            val totalMemory = getMemInfo().totalMem
            return totalMemory
        }

        fun isRunningOnLowMemory(): Boolean {
            return getMemInfo().lowMemory
        }

        fun getMemInfo(): ActivityManager.MemoryInfo {
            if(this::_context.isInitialized){
                val actManager = _context.getSystemService(ACTIVITY_SERVICE) as ActivityManager
                val memInfo = ActivityManager.MemoryInfo()
                actManager.getMemoryInfo(memInfo)
                return memInfo
            }
            return ActivityManager.MemoryInfo()
        }

        fun getBatteryPercentage(): Int {
            if(this::_context.isInitialized){
                val bm = _context.getSystemService(BATTERY_SERVICE) as BatteryManager
                return  bm.getIntProperty(BatteryManager.BATTERY_PROPERTY_CAPACITY)
            }
           return -1
        }

        fun getDeviceOrientation(): String {
            if (this::_context.isInitialized) {
                val orientation = _context.resources.configuration.orientation
                if (orientation == 1) {
                    return "Portrait"
                } else {
                    return "Landscape"
                }
            }
            return ""
        }

        fun getCurrentLocale(): String {
            if (this::_context.isInitialized) {
                val locale = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                    _context.resources.configuration.locales[0]
                } else {
                    _context.resources.configuration.locale
                }
                return locale.toString()
            }
            return ""
        }

        fun updateBreadCrumbsValues(breadcrumbsValues: BreadcrumbsValues){
            breadcrumbsValuesList.add(breadcrumbsValues)
        }

        fun getBreadCrumbsValues(): MutableList<BreadcrumbsValues>{
            return breadcrumbsValuesList
        }

        fun clearBreadCrumbsValues(){
            breadcrumbsValuesList.clear()
        }

        fun getCurrentTimestamp(): Long {
            return System.currentTimeMillis() / 1000
        }

        fun isDeveloperOptionEnabled(): Boolean {
            if(this::_context.isInitialized) {
                val isDevSettingEnabled = Settings.Global.getInt(
                    _context.contentResolver,
                    Settings.Global.DEVELOPMENT_SETTINGS_ENABLED,
                    if (Build.TYPE == "eng") 1 else 0
                ) !== 0
                return isDevSettingEnabled
            }
            return false
        }

        fun isDeviceRooted(): Boolean {
            val buildTags = Build.TAGS
            if (buildTags != null && buildTags.contains("test-keys")) {
                return true
            }
            return try {
                val file = File("/system/app/Superuser.apk")
                file.exists()
            } catch (e: Exception) {
                false
            }
        }

        fun saveInaiConfig(_config: InaiBaseConfig) {
            if(this::_context.isInitialized){
                val sharedPreference =  _context.getSharedPreferences(InaiConstants.INAI_SHARED_PREFERENCE,Context.MODE_PRIVATE)
                var editor = sharedPreference.edit()
                if(_config is InaiConfig){
                    editor.putString(InaiConstants.INAI_TOKEN,_config.token)
                    editor.putString(InaiConstants.INAI_ORDER_ID,_config.orderId)
                    if (!_config.countryCode.isNullOrEmpty())
                        editor.putString(InaiConstants.INAI_COUNTRY_CODE, _config.countryCode)
                }
                editor.commit()
            }
        }

       fun getValueFromSharedPrefs(key: String): String {
           if(this::_context.isInitialized){
               val sharedPreference =  _context.getSharedPreferences(InaiConstants.INAI_SHARED_PREFERENCE,Context.MODE_PRIVATE)
               if(sharedPreference.contains(key)){
                   return sharedPreference.getString(key,"")!!
               }
           }
           return ""
       }

        fun saveCrashLog(crashLog: String) {
            if(this::_context.isInitialized){
                val sharedPreference =  _context.getSharedPreferences(InaiConstants.INAI_SHARED_PREFERENCE,Context.MODE_PRIVATE)
                var editor = sharedPreference.edit()
                editor.putString(InaiConstants.INAI_CRASH_LOG,crashLog)
                editor.commit()
            }
        }

        fun clearSharedPref(){
            if(this::_context.isInitialized){
                val sharedPreference =  _context.getSharedPreferences(InaiConstants.INAI_SHARED_PREFERENCE,Context.MODE_PRIVATE)
                var editor = sharedPreference.edit()
                editor.clear()
                editor.commit()
            }
        }
    }


}