package io.inai.android_sdk

import android.content.Context
import androidx.fragment.app.FragmentActivity
import org.json.JSONObject

enum class InaiPayOutMethodStatus {
    Success, Failed, Canceled
}

data class InaiPayOutMethodResult(
    var status: InaiPayOutMethodStatus = InaiPayOutMethodStatus.Canceled,
    var data: JSONObject = JSONObject()
)

interface InaiPayoutMethodDelegate {
    fun payoutMethodAdded(result: InaiPayOutMethodResult)
}

class InaiPayouts() {

    private lateinit var _config: InaiPayoutsConfig
    private lateinit var _checkoutFragment: InaiCheckoutFragment
    private val sdkHostUrl = BuildConfig.InaiSdkHostUrl

    constructor(config: InaiPayoutsConfig) : this() {
        if (config.token.isBlank() || config.sessionToken.isBlank()) {
            //  Invalid config, reject and throw error
            throw Exception("Invalid config")
        }
        _config = config
    }

    fun addPayoutMethod(
        payoutMethodOption: String,
        payoutMethodDetails: JSONObject,
        context: Context,
        delegate: Any
    ) {
        //  Initiate the payout dialog
        val payoutDetailsJson = payoutMethodDetails.toString()
        val extraArgs = "'${payoutMethodOption}',${payoutDetailsJson}"
        val checkoutDialog = InaiCheckoutFragment.newInstance(
            sdkHostUrl,
            _config,
            delegate,
            InaiViewMode.AddPayoutMethod,
            extraArgs
        )
        showCheckoutDialog(context, checkoutDialog)
    }

    fun validateFields(payoutMethodOptions: String, payoutMethodDetails: JSONObject, context: Context, delegate: Any) {
        val payoutDetails = payoutMethodDetails.toString()
        val extraArgs = "'${payoutMethodOptions}',${payoutDetails}"
        val checkoutFragment = InaiCheckoutFragment.newInstance(
            sdkHostUrl,
            _config,
            delegate,
            InaiViewMode.ValidatePayoutFields,
            extraArgs
        )
        showInvisibleCheckout(context, checkoutFragment)
    }

    private fun showCheckoutDialog(context: Context, checkoutFragment: InaiCheckoutFragment) {
        this._checkoutFragment = checkoutFragment
        val fragmentActivity = context as FragmentActivity
        val checkoutDialog = InaiCheckoutDialog.newInstance(checkoutFragment, _config)
        checkoutDialog.show(fragmentActivity.supportFragmentManager, checkoutFragment.fragmentTag)
    }

    private fun showInvisibleCheckout(context: Context, checkoutFragment: InaiCheckoutFragment) {
        this._checkoutFragment = checkoutFragment
        val fragmentActivity = context as FragmentActivity
        val fragmentTransaction = fragmentActivity.supportFragmentManager.beginTransaction()
        val checkoutFragmentTransaction = fragmentTransaction.add(checkoutFragment, checkoutFragment.fragmentTag)
        checkoutFragmentTransaction.commit()
    }

}