package io.inai.android_sdk

import android.os.Build
import android.util.Log
import io.inai.android_sdk.expressPay.helpers.InaiNetworkRequestHandler
import io.inai.android_sdk.helpers.InaiBaseUtils
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.launch
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import org.json.JSONArray
import org.json.JSONObject

class InaiCrashlyticsHandler : Thread.UncaughtExceptionHandler {

    lateinit var defaultUncaughtExceptionHandler: Thread.UncaughtExceptionHandler
    private var coroutineScope = CoroutineScope(Job() + Dispatchers.IO)
    val urlWithParams = BuildConfig.InaiCrashlyticsURL
    val TAG = "InaiCrashlyticsHandler"

    companion object {
        @Volatile
        lateinit var inaiCrashlyticsHandler : InaiCrashlyticsHandler
        fun newInstance(defaultUncaughtExceptionHandler: Thread.UncaughtExceptionHandler): InaiCrashlyticsHandler {
            synchronized(this){
                if (!::inaiCrashlyticsHandler.isInitialized){
                    inaiCrashlyticsHandler = InaiCrashlyticsHandler()
                    inaiCrashlyticsHandler?.defaultUncaughtExceptionHandler = defaultUncaughtExceptionHandler
                }
                return inaiCrashlyticsHandler
            }
        }
    }

    val isProbablyRunningOnEmulator: Boolean by lazy {
        // Android SDK emulator
        return@lazy ((Build.MANUFACTURER == "Google" && Build.BRAND == "google" &&
                ((Build.FINGERPRINT.startsWith("google/sdk_gphone_")
                        && Build.FINGERPRINT.endsWith(":user/release-keys")
                        && Build.PRODUCT.startsWith("sdk_gphone_")
                        && Build.MODEL.startsWith("sdk_gphone_"))
                        //alternative
                        || (Build.FINGERPRINT.startsWith("google/sdk_gphone64_")
                        && (Build.FINGERPRINT.endsWith(":userdebug/dev-keys") || Build.FINGERPRINT.endsWith(":user/release-keys"))
                        && Build.PRODUCT.startsWith("sdk_gphone64_")
                        && Build.MODEL.startsWith("sdk_gphone64_"))))
                //
                || Build.FINGERPRINT.startsWith("generic")
                || Build.FINGERPRINT.startsWith("unknown")
                || Build.MODEL.contains("google_sdk")
                || Build.MODEL.contains("Emulator")
                || Build.MODEL.contains("Android SDK built for x86")
                //bluestacks
                || "QC_Reference_Phone" == Build.BOARD && !"Xiaomi".equals(Build.MANUFACTURER, ignoreCase = true)
                //bluestacks
                || Build.MANUFACTURER.contains("Genymotion")
                || Build.HOST.startsWith("Build")
                //MSI App Player
                || Build.BRAND.startsWith("generic") && Build.DEVICE.startsWith("generic")
                || Build.PRODUCT == "google_sdk")
    }

    override fun uncaughtException(thread: Thread?, ex: Throwable?) {

        val crashObj = JSONObject()
        val exceptionValueObj = JSONObject()
        val exceptionMechanismObj = JSONObject()
        val exceptionFramesObj = JSONObject()
        val exceptionFramesArray = JSONArray()
        val exceptionValuesArray = JSONArray()
        val exceptionObj = JSONObject()
        val exceptionTagObj = getExceptionTagObject()
        val deviceObj = JSONObject()

        exceptionMechanismObj.put("type", "generic")
        exceptionMechanismObj.put("handled", false)
        exceptionMechanismObj.put("synthetic", true)

        //frames value setting
        for (stacktrace in ex!!.stackTrace.reversed()) {
            var framesObj = JSONObject()
            framesObj.put("in_app", true)
            framesObj.put("function", stacktrace.methodName)
            framesObj.put("filename", stacktrace.fileName)
            framesObj.put("lineno", stacktrace.lineNumber)
            framesObj.put("context_line", stacktrace)
            exceptionFramesArray.put(framesObj)
        }
        exceptionFramesObj.put("frames", exceptionFramesArray)

        deviceObj.put("device",getDeviceDetailsObject())

        //exception value setting
        exceptionValueObj.put("type", ex.toString())
        exceptionValueObj.put("value", ex?.stackTraceToString())
        exceptionValueObj.put("thread_id", thread?.id)
        exceptionValueObj.put("mechanism", exceptionMechanismObj)
        exceptionValueObj.put("stacktrace", exceptionFramesObj)

        exceptionValuesArray.put(exceptionValueObj)
        exceptionObj.put("values", exceptionValuesArray)

        //entire crash object setting
        crashObj.put("timestamp", InaiBaseUtils.getCurrentTimestamp())
        crashObj.put("environment", BuildConfig.BUILD_TYPE)
        crashObj.put("culprit", ex?.cause)
        crashObj.put("message", ex?.message)
        crashObj.put("transaction", ex?.stackTrace?.component1()?.className)
        crashObj.put("tags", exceptionTagObj)
        crashObj.put("exception", exceptionObj)
        crashObj.put("contexts", deviceObj)
        crashObj.put("breadcrumbs", getBreadcrumbsArray())
        crashObj.put("user", getUserObject())

        //save crash log to shared preference
        InaiBaseUtils.saveCrashLog(crashObj.toString())

        val errorCallback: (String) -> Unit = { errorResponse ->
            Log.e(TAG, "Error while passing crash report: $errorResponse")
            defaultUncaughtExceptionHandler?.uncaughtException(thread, ex)
        }

        coroutineScope.launch(Dispatchers.IO) {
            InaiNetworkRequestHandler.postErrorLogToInaiWidget(
                urlWithParams,
                crashObj.toString(),
                errorCallback
            ) { response ->
                //clear shared preference on success
                InaiBaseUtils.clearSharedPref()
                defaultUncaughtExceptionHandler?.uncaughtException(thread, ex)
            }
        }
    }

    fun logEvent(event: String, value: String?) {

        val crashObj = JSONObject()
        val exceptionTagObj = getExceptionTagObject()
        val deviceObj = JSONObject()
        val exceptionValueObj = JSONObject()

        deviceObj.put("device",getDeviceDetailsObject())

        crashObj.put("timestamp", InaiBaseUtils.getCurrentTimestamp())
        crashObj.put("environment", BuildConfig.BUILD_TYPE)
        crashObj.put("culprit", "Event Log")
        crashObj.put("message", event)
        crashObj.put("tags", exceptionTagObj)
        crashObj.put("contexts", deviceObj)
        crashObj.put("breadcrumbs", getBreadcrumbsArray())
        crashObj.put("user", getUserObject())
        if(!value.isNullOrEmpty()){
            exceptionValueObj.put("type", event)
            exceptionValueObj.put("value", value)
            crashObj.put("exception", exceptionValueObj)
        }

        val errorCallback: (String) -> Unit = { errorResponse ->
            Log.e(TAG, "Error while logging Exception: $errorResponse")
        }

        coroutineScope.launch(Dispatchers.IO) {
            InaiNetworkRequestHandler.postErrorLogToInaiWidget(
                urlWithParams,
                crashObj.toString(),
                errorCallback
            ) { response ->

            }
        }
    }

    private fun getExceptionTagObject(): JSONObject {
        val exceptionTagObj = JSONObject()
        exceptionTagObj.put("sdkVersion", BuildConfig.InaiSdkVersion)
        exceptionTagObj.put("device", Build.MODEL)
        exceptionTagObj.put("device.brand", Build.BRAND)
        exceptionTagObj.put("device.family", Build.DEVICE)
        exceptionTagObj.put("os", Build.VERSION.RELEASE)
        exceptionTagObj.put("os.api.level", Build.VERSION.SDK_INT)

        exceptionTagObj.put("platform", "ANDROID")
        try {
            exceptionTagObj.put("os.version.name", Build.VERSION_CODES::class.java.fields[Build.VERSION.SDK_INT].name)
            exceptionTagObj.put("os.rooted", InaiBaseUtils.isDeviceRooted())
            exceptionTagObj.put("os.devOptionEnabled", InaiBaseUtils.isDeveloperOptionEnabled())

            if (!InaiBaseUtils.getValueFromSharedPrefs(InaiConstants.INAI_TOKEN).isNullOrEmpty())
                exceptionTagObj.put("merchantId", InaiBaseUtils.getValueFromSharedPrefs(InaiConstants.INAI_TOKEN))
            if (!InaiBaseUtils.getValueFromSharedPrefs(InaiConstants.INAI_ORDER_ID).isNullOrEmpty())
                exceptionTagObj.put("orderID", InaiBaseUtils.getValueFromSharedPrefs(InaiConstants.INAI_ORDER_ID))
            if (!InaiBaseUtils.getValueFromSharedPrefs(InaiConstants.INAI_COUNTRY_CODE).isNullOrEmpty())
                exceptionTagObj.put("countryCode", InaiBaseUtils.getValueFromSharedPrefs(InaiConstants.INAI_COUNTRY_CODE))
        }catch (e:Exception){

        }

        return exceptionTagObj
    }

    private fun getDeviceDetailsObject(): JSONObject{
        val deviceDetailsObj = JSONObject()
        try {
            val availMem = InaiBaseUtils.getAvailableMemoryInfo()
            val totalMem = InaiBaseUtils.getTotalMemoryInfo()
            deviceDetailsObj.put("memory_size", totalMem)
            deviceDetailsObj.put("free_memory", availMem)
            deviceDetailsObj.put("low_memory", InaiBaseUtils.isRunningOnLowMemory())
            deviceDetailsObj.put("orientation", InaiBaseUtils.getDeviceOrientation())
            deviceDetailsObj.put("locale", InaiBaseUtils.getCurrentLocale())
            if(InaiBaseUtils.getBatteryPercentage() > 0)
                deviceDetailsObj.put("battery_level", InaiBaseUtils.getBatteryPercentage())
        }catch (e:Exception){
            Log.d(TAG,"Exception while getting details ${e.message}")
        }
        deviceDetailsObj.put("manufacturer", Build.MANUFACTURER)
        deviceDetailsObj.put("model", Build.MODEL)
        deviceDetailsObj.put("brand", Build.BRAND)
        deviceDetailsObj.put("device", Build.DEVICE)
        deviceDetailsObj.put("product", Build.PRODUCT)
        deviceDetailsObj.put("simulator", isProbablyRunningOnEmulator)

        return deviceDetailsObj
    }

    private fun getBreadcrumbsArray(): JSONArray {
        val breadcrumbsArray = JSONArray()
        InaiBaseUtils.getBreadCrumbsValues().iterator().forEach {
            breadcrumbsArray.put(JSONObject(Json.encodeToString(it)))
        }
        return breadcrumbsArray
    }

    private fun getUserObject(): JSONObject {
        val userObj = JSONObject()
        userObj.put("ip_address", "{{auto}}")
        return userObj
    }

    fun uploadCrashLog(crashLog: String) {
        val errorCallback: (String) -> Unit = { errorResponse ->
            Log.e(TAG, "Error while upload crash report: $errorResponse")
        }

        coroutineScope.launch(Dispatchers.IO) {
            InaiNetworkRequestHandler.postErrorLogToInaiWidget(
                urlWithParams,
                crashLog,
                errorCallback
            ) { response ->
                //set empty string in shared preference crash log on success
                InaiBaseUtils.saveCrashLog("")
            }
        }

    }

}