package io.inai.android_sdk

import android.content.Context
import android.content.Intent
import android.content.pm.ActivityInfo
import androidx.fragment.app.FragmentActivity
import com.google.android.gms.common.api.ApiException
import com.google.android.gms.tasks.Task
import com.google.android.gms.wallet.*
import io.inai.android_sdk.helpers.InaiBaseUtils
import kotlinx.serialization.decodeFromString
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import org.json.JSONArray
import org.json.JSONObject

enum class InaiStatus {
    Success, Failed, Canceled
}

enum class InaiPaymentStatus {
    Success, Failed, Canceled
}

enum class InaiPaymentMethodStatus {
    Success, Failed, Canceled
}

enum class InaiInstallmentPlansInfoStatus {
    Success, Failed
}

enum class InaiExecuteStatus {
    Success, Failed, Canceled
}

enum class InaiCardInfoStatus {
    Success, Failed;

    companion object {
        fun valueOfOrElse(result: String): InaiCardInfoStatus {
            return if (values().map { it.name }.contains(result)) {
                valueOf(result)
            } else {
                Failed
            }
        }
    }
}

enum class InaiValidateFieldsStatus {
    Success, Failed;

    companion object {
        fun valueOfOrElse(result: String): InaiValidateFieldsStatus {
            return if (values().map { it.name }.contains(result)) {
                valueOf(result)
            } else {
                Failed
            }
        }
    }
}

data class InaiResult(
    var status: InaiStatus = InaiStatus.Canceled,
    var data: JSONObject = JSONObject("{code: \"PAYMENT_INTERRUPTED\", message: \"Payment was interrupted. Please verify the latest transaction status.\"}")
)

data class InaiPaymentResult(
    var status: InaiPaymentStatus = InaiPaymentStatus.Canceled,
    var data: JSONObject = JSONObject()
)

data class InaiPaymentMethodResult(
    var status: InaiPaymentMethodStatus = InaiPaymentMethodStatus.Canceled,
    var data: JSONObject = JSONObject()
)

data class InaiCardInfoResult(
    var status: InaiCardInfoStatus = InaiCardInfoStatus.Failed,
    var data: JSONObject = JSONObject()
)

data class InaiValidateFieldsResult(
    var status: InaiValidateFieldsStatus = InaiValidateFieldsStatus.Failed,
    var data: JSONObject = JSONObject()
)

data class InaiInstallmentPlansResult(
    var status: InaiInstallmentPlansInfoStatus = InaiInstallmentPlansInfoStatus.Failed,
    var data: JSONObject = JSONObject()
)

data class InaiExecuteResult(
    var status: InaiExecuteStatus = InaiExecuteStatus.Failed,
    var data: JSONObject = JSONObject()
)

interface InaiCheckoutDelegate {
    fun paymentFinished(result: InaiPaymentResult)
}

interface InaiPaymentMethodDelegate {
    fun paymentMethodSaved(result: InaiPaymentMethodResult)
}

interface InaiCardInfoDelegate {
    fun cardInfoFetched(result: InaiCardInfoResult)
}

interface InaiValidateFieldsDelegate {
    fun fieldsValidationFinished(result: InaiValidateFieldsResult)
}

/// Delegate for Credit Card Info Identification
interface InaiCardIdentificationDelegate {
    /// cardInfo: => { brand: "visa|mastercard", type: "credit|debit", issuerName: "<issuer_name>", category: "platinum", country: "USA|IND" }
    fun onCardIdentification(cardInfo: JSONObject)
}

interface InaiInstallmentPlansDelegate{
    fun installmentPlansFetched(installmentPlans: InaiInstallmentPlansResult)
}

interface InaiExecuteDelegate{
    fun executionFinished(executeResult: InaiExecuteResult)
}

public data class InaiGooglePayRequestData(
    var currencyCode: String = "",
    var countryCode: String = "",
    var merchantName: String = "",
    var merchantId: String = "",
    var productDescription: String = "",
    var orderAmount: Double = 0.0,
    var supportedNetworks: JSONArray = JSONArray(),
    var tokenization: JSONObject = JSONObject(),
    var canMakePayments: Boolean = false,
    var googlePayBaseConfiguration: JSONObject = JSONObject(),
    var baseCardPaymentMethod: JSONObject = JSONObject(),
    var paymentsClient: PaymentsClient? = null
)

const val CARD_METHOD_TYPE = "CARD"
const val TOTAL_PRICE_STATUS = "FINAL"

class InaiCheckout()  {
    private lateinit var _config: InaiConfig
    private val sdkHostUrl = BuildConfig.InaiSdkHostUrl

    private var cardIdentificationDelegate: InaiCardIdentificationDelegate? = null
    private lateinit var _checkoutFragment: InaiCheckoutFragment

    constructor(config: InaiConfig) : this() {
        inaiCrashlyticsHandler = InaiCrashlyticsHandler.newInstance(Thread.getDefaultUncaughtExceptionHandler())
        Thread.setDefaultUncaughtExceptionHandler(inaiCrashlyticsHandler)
        InaiBaseUtils.clearBreadCrumbsValues()
        if ( (config.orderId.isNullOrBlank()
                    && config.planId.isNullOrBlank() )
            || config.token.isNullOrBlank()){
            //  Invalid config, reject and throw error
            inaiCrashlyticsHandler.logEvent("Invalid config",Json.encodeToString(config))
            throw Exception("Invalid config")
        }
        InaiBaseUtils.updateBreadCrumbsValues(InaiBaseUtils.BreadcrumbsValues("Configuration", "Initialised", "info", InaiBaseUtils.getCurrentTimestamp(), "user", null))
        _config = config
    }

    fun activityResult(resultCode: Int, data: Intent) {
        this._checkoutFragment.activityResult(resultCode, data)
    }

    fun presentCheckout(context: Context, delegate: Any) {
        checkIfCrashLogAvailable(context)
        //  update breadcrumbs values
        InaiBaseUtils.updateBreadCrumbsValues(InaiBaseUtils.BreadcrumbsValues("Drop-in", "Initiated", "info", InaiBaseUtils.getCurrentTimestamp(), "user", null))
        //  Initiate the checkout dialog
        val checkoutFragment = InaiCheckoutFragment.newInstance(sdkHostUrl, _config, delegate, InaiViewMode.Payment)
        showCheckoutDialog(context, checkoutFragment)
    }

    fun addPaymentMethod(type: String, context: Context, delegate: Any) {
        checkIfCrashLogAvailable(context)
        //  update breadcrumbs values
        InaiBaseUtils.updateBreadCrumbsValues(InaiBaseUtils.BreadcrumbsValues("Add payment method", "Initiated", "info", InaiBaseUtils.getCurrentTimestamp(), "user", null))
        //  Initiate the checkout dialog
        val extraArgs = "'$type'"
        val checkoutDialog = InaiCheckoutFragment.newInstance(
            sdkHostUrl,
            _config,
            delegate,
            InaiViewMode.AddPaymentMethod,
            extraArgs
        )
        showCheckoutDialog(context, checkoutDialog)
    }

    fun presentPayWithPaymentMethod(paymentMethodId: String,  context: Context, delegate: Any) {
        checkIfCrashLogAvailable(context)
        //  update breadcrumbs values
        InaiBaseUtils.updateBreadCrumbsValues(InaiBaseUtils.BreadcrumbsValues("Pay with saved payment method", "Initiated", "info", InaiBaseUtils.getCurrentTimestamp(), "user", null))
        //  Initiate the checkout dialog
        val extraArgs = "'$paymentMethodId'"
        val checkoutFragment = InaiCheckoutFragment.newInstance(
            sdkHostUrl,
            _config,
            delegate,
            InaiViewMode.PayWithPaymentMethod,
            extraArgs
        )
        showCheckoutDialog(context, checkoutFragment)
    }

    fun makePayment(paymentMethodOption: String, paymentDetails: JSONObject, context: Context, delegate: Any) {
        checkIfCrashLogAvailable(context)
        //  update breadcrumbs values
        InaiBaseUtils.updateBreadCrumbsValues(InaiBaseUtils.BreadcrumbsValues("Headless payment", "Initiated", "info", InaiBaseUtils.getCurrentTimestamp(), "user", null))
        val paymentDetailsJson = paymentDetails.toString()
        val extraArgs = "'${paymentMethodOption}',${paymentDetailsJson}"
        val checkoutFragment = InaiCheckoutFragment.newInstance(
            sdkHostUrl,
            _config,
            delegate,
            InaiViewMode.MakePayment,
            extraArgs
        )
        showCheckoutDialog(context, checkoutFragment)
    }

    fun getCardInfo(cardNumber: String, context: Context, delegate: Any) {
        checkIfCrashLogAvailable(context)
        //  update breadcrumbs values
        InaiBaseUtils.updateBreadCrumbsValues(InaiBaseUtils.BreadcrumbsValues("Get card details", "Initiated", "info", InaiBaseUtils.getCurrentTimestamp(), "user", null))
        //  Show the view as invisible..
        val extraArgs = "'$cardNumber'"
        val checkoutFragment = InaiCheckoutFragment.newInstance(
            sdkHostUrl,
            _config,
            delegate,
            InaiViewMode.GetCardInfo,
            extraArgs
        )
        showInvisibleCheckout(context, checkoutFragment)
    }

    fun validateFields(paymentMethodOption: String, paymentDetails: JSONObject, context: Context, delegate: Any) {
        checkIfCrashLogAvailable(context)
        //  update breadcrumbs values
        InaiBaseUtils.updateBreadCrumbsValues(InaiBaseUtils.BreadcrumbsValues("Validate fields", "Initiated", "info", InaiBaseUtils.getCurrentTimestamp(), "user", null))
        val paymentDetailsJson = paymentDetails.toString()
        val extraArgs = "'${paymentMethodOption}',${paymentDetailsJson}"
        val checkoutFragment = InaiCheckoutFragment.newInstance(
            sdkHostUrl,
            _config,
            delegate,
            InaiViewMode.ValidateFields,
            extraArgs
        )
        showInvisibleCheckout(context, checkoutFragment)
    }

    fun getInstallmentPlans(paymentMethodOption: String, paymentDetails: JSONObject, context: Context, delegate: Any) {
        checkIfCrashLogAvailable(context)
        //  update breadcrumbs values
        InaiBaseUtils.updateBreadCrumbsValues(InaiBaseUtils.BreadcrumbsValues("Get installment plans", "Initiated", "info", InaiBaseUtils.getCurrentTimestamp(), "user", null))
        val paymentDetailsJson = paymentDetails.toString()
        val extraArgs = "'${paymentMethodOption}',${paymentDetailsJson}"
        val checkoutFragment = InaiCheckoutFragment.newInstance(
            sdkHostUrl,
            _config,
            delegate,
            InaiViewMode.InstallmentPlans,
            extraArgs
        )
        showInvisibleCheckout(context, checkoutFragment)
    }

    fun execute(payload: JSONObject,context: Context,delegate: Any){
        checkIfCrashLogAvailable(context)
        //  update breadcrumbs values
        InaiBaseUtils.updateBreadCrumbsValues(InaiBaseUtils.BreadcrumbsValues("Execute", "Initiated", "info", InaiBaseUtils.getCurrentTimestamp(), "user", null))
        val payloadJson = payload.toString()
        val extraArgs = "${payloadJson}"
        val checkoutFragment = InaiCheckoutFragment.newInstance(
            sdkHostUrl,
            _config,
            delegate,
            InaiViewMode.Execute,
            extraArgs
        )
        showInvisibleCheckout(context, checkoutFragment)
    }

    private fun showCheckoutDialog(context: Context, checkoutFragment: InaiCheckoutFragment) {

        checkoutFragment.cardIdentificationDelegate = cardIdentificationDelegate
        this._checkoutFragment = checkoutFragment
        val fragmentActivity = context as FragmentActivity
        //  Lock orientation to whatever the parent activity was set to.
        fragmentActivity.requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_LOCKED

        val checkoutDialog = InaiCheckoutDialog.newInstance(checkoutFragment)
        checkoutDialog.show(fragmentActivity.supportFragmentManager, checkoutFragment.fragmentTag)
    }

    private fun showInvisibleCheckout(context: Context, checkoutFragment: InaiCheckoutFragment) {

        checkoutFragment.cardIdentificationDelegate = cardIdentificationDelegate
        this._checkoutFragment = checkoutFragment
        val fragmentActivity = context as FragmentActivity
        //  Lock orientation to whatever the parent activity was set to.
        fragmentActivity.requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_LOCKED

        val fragmentTransaction = fragmentActivity.supportFragmentManager.beginTransaction()
        val checkoutFragmentTransaction = fragmentTransaction.add(checkoutFragment, checkoutFragment.fragmentTag)
        checkoutFragmentTransaction.commit()
    }

    /// Hide inai widget without triggering a callback
    fun remove() {
        this._checkoutFragment.dismissWebView(true)
    }

    /// Disable Pay Button
    fun disableCTA(paymentMethods: Array<String>) {
        val paymentMethodsJSON = Json.encodeToString(paymentMethods)
        val jsString = "window.inai.disableCTA($paymentMethodsJSON);"
        this._checkoutFragment.evalJS(jsString)
    }

    /// Enable Pay Button
    fun enableCTA(paymentMethods: Array<String>) {
        val paymentMethodsJSON = Json.encodeToString(paymentMethods)
        val jsString = "window.inai.enableCTA($paymentMethodsJSON);"
        this._checkoutFragment.evalJS(jsString)
    }

    /// Update Pay Button
    fun updateCTA(ctaConfig: InaiCTAConfig) {
        val ctaConfigJson = Json.encodeToString(ctaConfig)
        val jsString = "window.inai.updateCTA($ctaConfigJson);"
        this._checkoutFragment.evalJS(jsString)
    }

    private fun checkIfCrashLogAvailable(context: Context) {
        context?.let { InaiBaseUtils().setup(it) }
        if (!InaiBaseUtils.getValueFromSharedPrefs(InaiConstants.INAI_CRASH_LOG).isNullOrEmpty()) {
            val crashLog = InaiBaseUtils.getValueFromSharedPrefs(InaiConstants.INAI_CRASH_LOG)
            inaiCrashlyticsHandler.uploadCrashLog(crashLog)
        }
    }

    companion object {
        lateinit var inaiCrashlyticsHandler: InaiCrashlyticsHandler
        fun isCrashlyticsInitialized() = this@Companion::inaiCrashlyticsHandler.isInitialized

        private val googlePayBaseRequest = JSONObject().apply {
            put("apiVersion", 2)
            put("apiVersionMinor", 0)
        }

        @JvmStatic
        fun initGooglePay(paymentMethodJSON: JSONObject,
                          context: Context,
                          callback: (googlePayRequestData: InaiGooglePayRequestData?) -> Unit) {

            val rails = paymentMethodJSON.getJSONArray("payment_method_options")

            var googlePayRail:JSONObject? = null

            for (i in 0 until rails.length()) {
                val rail = rails.getJSONObject(i)
                val railCode = rail.getString("rail_code")
                if (railCode.equals("google_pay")) {
                    googlePayRail = rail
                }
            }

            //  Guard against no google pay rail case
            if (googlePayRail == null) {
                callback(null)
            } else {

                val googlePayRequestData = InaiGooglePayRequestData()

                val configs = googlePayRail.getJSONObject("configs")

                val supportedMethods = configs.getJSONArray("supported_methods")
                val allowedAuthMethods = JSONArray()
                //  eg : "PAN_ONLY" "CRYPTOGRAM_3DS"
                for (j in 0 until supportedMethods.length()) {
                    val supportedMethod = supportedMethods.getString(j)
                    allowedAuthMethods.put(supportedMethod)
                }

                val supportedNetworks = configs.getJSONArray("supported_networks")
                val allowedCardNetworks = JSONArray()
                //  eg : "AMEX" "DISCOVER" "JCB" "MASTERCARD" "VISA"
                for (j in 0 until supportedNetworks.length()) {
                    val allowedCardNetwork = supportedNetworks.getJSONObject(j)
                    allowedCardNetworks.put(allowedCardNetwork.getString("name"))
                }

                val cardPaymentMethods = JSONObject().apply {
                    val parameters = JSONObject().apply {
                        put("allowedAuthMethods", allowedAuthMethods)
                        put("allowedCardNetworks", allowedCardNetworks)
                        //  Added for fetching billing address
                        put("billingAddressRequired", true)
                        put("billingAddressParameters", JSONObject().apply {
                            put("format", "FULL")
                            put("phoneNumberRequired", true)
                        })
                    }
                    put("type", CARD_METHOD_TYPE)
                    put("parameters", parameters)
                }
                googlePayRequestData.baseCardPaymentMethod = cardPaymentMethods

                val requestJSON = googlePayBaseRequest.apply {
                    put("allowedPaymentMethods", JSONArray().put(cardPaymentMethods))
                }
                googlePayRequestData.googlePayBaseConfiguration = requestJSON

                googlePayRequestData.countryCode = configs.getString("country_code")
                googlePayRequestData.currencyCode = configs.getString("currency_code")
                googlePayRequestData.orderAmount = configs.getDouble("order_amount")
                googlePayRequestData.merchantId = configs.getString("merchant_id")
                //  Merchant name can be null, kotlin will parse this as "null"
                //  Assign merchant name only for non null value
                if (!configs.isNull("merchant_name")) {
                    googlePayRequestData.merchantName = configs.getString("merchant_name")
                }
                googlePayRequestData.tokenization = configs.getJSONObject("tokenization")
                val configEnvironment = configs.getString("environment")
                var environment = WalletConstants.ENVIRONMENT_PRODUCTION
                if(configEnvironment.equals("test",true)){
                     environment = WalletConstants.ENVIRONMENT_TEST
                }
                val walletOptions = Wallet.WalletOptions.Builder()
                    .setEnvironment(environment)
                    .build()

                val paymentsClient = Wallet.getPaymentsClient(context, walletOptions)
                val request = IsReadyToPayRequest.fromJson(requestJSON.toString())
                val task = paymentsClient.isReadyToPay(request)

                googlePayRequestData.paymentsClient = paymentsClient

                task.addOnCompleteListener { completedTask ->
                    try {
                        googlePayRequestData.canMakePayments =
                            completedTask.getResult(ApiException::class.java)
                        callback(googlePayRequestData)
                    } catch (exception: ApiException) {
                        callback(googlePayRequestData)
                    }
                }
            }
        }

        @JvmStatic
        fun launchGooglePayRequest(googlePayRequestData: InaiGooglePayRequestData): Task<PaymentData> {

            val cardPaymentMethod = googlePayRequestData.baseCardPaymentMethod.apply {
                put("type", "CARD")
                put("tokenizationSpecification", googlePayRequestData.tokenization)
            }

            val transactionInfo = JSONObject().apply {
                put("totalPrice", googlePayRequestData.orderAmount.toString())
                put("totalPriceStatus", TOTAL_PRICE_STATUS)
                put("currencyCode", googlePayRequestData.currencyCode)
                put("countryCode", googlePayRequestData.countryCode)
            }

            val merchantInfo = JSONObject().apply {
                put("merchantName", googlePayRequestData.merchantName)
                put("merchantId", googlePayRequestData.merchantId)
            }

            val paymentDataRequestJson =
                JSONObject(googlePayRequestData.googlePayBaseConfiguration.toString()).apply {
                    put("allowedPaymentMethods", JSONArray().put(cardPaymentMethod))
                    put("transactionInfo", transactionInfo)
                    put("merchantInfo", merchantInfo)
                }

            val paymentDataRequestJsonString = paymentDataRequestJson.toString()
            val paymentDataRequest = PaymentDataRequest.fromJson(paymentDataRequestJsonString)
            return googlePayRequestData.paymentsClient!!.loadPaymentData(paymentDataRequest)
        }


        /// Helper method to convert Payment Method Option data to ``InaiGooglePayRequestData``
        @JvmStatic
        fun getGooglePayRequestData(paymentData: PaymentData): JSONObject {

            val googlePaymentMethodData: JSONObject
            try {
                // Token will be null if PaymentDataRequest was not constructed using fromJson(String).
                val paymentInformation = paymentData.toJson()
                googlePaymentMethodData =
                    JSONObject(paymentInformation).getJSONObject("paymentMethodData")
            } catch (ex: Exception) {
                throw Exception("Invalid paymentData")
            }

            val googlePayRequestData = JSONObject()
            //  Extract info from googlePaymentMethodData
            //  Google Token details available on this link : https://developers.google.com/pay/api/android/guides/resources/payment-data-cryptography
            val tokenizationData = googlePaymentMethodData.getJSONObject("tokenizationData")
            val tokenString = tokenizationData.getString("token")
            val tokenObject = JSONObject(tokenString)
            googlePayRequestData.put("fields", JSONArray())
            googlePayRequestData.put("google_pay", tokenObject)
            return googlePayRequestData
        }

        /*Below method to return the InaiConfigStyles from the styles jsonString*/
        @JvmStatic
        fun getInaiConfigStyles(stylesData: String): InaiConfigStyles? {
            val json = Json { ignoreUnknownKeys = true }
            var inaiConfigStyles: InaiConfigStyles?=null
            try {
                inaiConfigStyles = json.decodeFromString<InaiConfigStyles>(stylesData)
            }catch (e: Exception){
                throw Exception("Invalid styles - $e")
            }
            return inaiConfigStyles
        }
    }
}