package io.inai.android_sdk.expressPay.helpers

import android.util.Base64
import io.inai.android_sdk.BuildConfig
import kotlinx.coroutines.*
import java.io.*
import java.net.HttpURLConnection
import java.net.URL
import java.nio.charset.StandardCharsets

object InaiNetworkRequestHandler {

    const val POST: String = "POST"
    private const val UNKNOWN_ERROR = "UNKNOWN_ERROR"

    //  We use a coroutine scope to launch network requests and switch between threads.
    //  The scope should always be cancelled if the operation is completed or the operation
    //  is no more necessary.
    private var coroutineScope: CoroutineScope? = null

    fun postErrorLogToSentry(
        url: String?,
        requestBody: String,
        errorCallback: (String) -> Unit,
        successCallback: (String) -> Unit
    ) {
        val errorHandler = CoroutineExceptionHandler { context, error ->
            //  Parse the error as a Result.Failure object and send it in the callback on the main thread.
            coroutineScope?.launch(Dispatchers.Main) {
                val errorResponse = Result.Failure(error.localizedMessage ?: UNKNOWN_ERROR)
                errorCallback(errorResponse.message)
            }
        }
        val mUrl = URL(url)
        val conn: HttpURLConnection = mUrl.openConnection() as HttpURLConnection
        conn.apply {
            readTimeout = 3000
            connectTimeout = 3000
            requestMethod = POST
            doInput = true
            doOutput = true
            setRequestProperty("Accept", "application/json")
            setRequestProperty("Content-Type", "application/json")
            setRequestProperty("Connection", "keep-alive")
            setRequestProperty(
                "X-Sentry-Auth",
                "Sentry sentry_version=7,sentry_secret=${BuildConfig.InaiSentrySecret},sentry_key=${BuildConfig.InaiSentryKey}"
            )
        }
        coroutineScope = CoroutineScope(SupervisorJob() + errorHandler)
        // Launch a child coroutine inside the parent scope on the Dispatchers.IO thread.
        coroutineScope?.launch(Dispatchers.IO) {
            val postData: ByteArray = requestBody.toByteArray(StandardCharsets.UTF_8)
            val base64data = Base64.encode(postData, Base64.NO_WRAP)
            try {
                val outputStream = DataOutputStream(conn.outputStream)
                outputStream.write(base64data)
                outputStream.flush()
            } catch (ex: Exception) {
                errorCallback(ex.message ?: "ERROR")
            }

            try {
                conn.connect()
                val responseCode: Int = conn.responseCode
                if (responseCode == 201 || responseCode == 200) {
                    val response = readResponse(conn)
                    successCallback(response)
                } else {
                    errorCallback(conn.responseMessage)
                }
            } catch (e: Exception) {
                errorCallback(e.message ?: "")
            }
        }

    }

    //  This cancels any running coroutine scopes. WHen a scope cancels any background work
    //  that was executing will be cancelled.
    fun cancelCoroutineScope() {
        coroutineScope?.cancel()
    }

    //  Reads the response from the connection object input stream
    private fun readResponse(conn: HttpURLConnection): String {
        val `in` =
            BufferedReader(InputStreamReader(conn.inputStream))
        var inputLine: String?
        val response = StringBuffer()
        while (`in`.readLine().also { inputLine = it } != null) {
            response.append(inputLine)
        }
        `in`.close()
        return response.toString()
    }


    //  A sealed class which contains two child classes - Success and Failure.
    //  This enables us to pass the success or failure result as a single Result
    //  instance. This result object can then be checked for a Success  instance
    //  or a Failure instance. This eliminates the need for separate callbacks
    //  for success and failure cases.
    sealed class Result {
        class Success(val message: String) : Result()
        class Failure(val message: String) : Result()
    }
}