package io.inai.android_sdk

import android.annotation.SuppressLint
import android.app.Activity
import android.content.Context
import android.content.Intent
import android.content.pm.ActivityInfo
import android.content.pm.PackageManager
import android.graphics.Bitmap
import android.net.Uri
import android.net.http.SslError
import android.os.Build
import android.os.Bundle
import android.os.Message
import android.util.Log
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.webkit.*
import android.widget.ProgressBar
import androidx.activity.result.ActivityResult
import androidx.activity.result.IntentSenderRequest
import androidx.activity.result.contract.ActivityResultContracts
import androidx.appcompat.app.AppCompatActivity
import androidx.fragment.app.Fragment
import com.google.android.gms.common.api.ApiException
import com.google.android.gms.common.api.CommonStatusCodes
import com.google.android.gms.common.api.ResolvableApiException
import com.google.android.gms.wallet.PaymentData
import io.inai.android_sdk.helpers.InaiBaseUtils
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.launch
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import org.json.JSONArray
import org.json.JSONObject


enum class InaiViewMode {
    Payment, AddPaymentMethod, PayWithPaymentMethod, MakePayment, GetCardInfo, ValidateFields,
    AddPayoutMethod, ValidatePayoutFields, CryptoEstimate,InstallmentPlans,Execute
}

class InaiCheckoutFragment : Fragment() {
    private lateinit var rootView: View
    private lateinit var webView: WebView
    private lateinit var _delegate: Any
    private lateinit var loadingIndicator: ProgressBar
    private lateinit var _config: InaiBaseConfig
    private var _url: String = ""
    private var inaiResult: InaiResult = InaiResult()
    private var _viewMode: InaiViewMode = InaiViewMode.Payment
    private var _extraArgs: String = ""
    private var _webViewLoaded: Boolean = false
    private var _webViewDismissed: Boolean = false
    private var _ignoreDelegateCallback: Boolean = false
    var fragmentTag: String = "inai_checkout_${java.util.UUID.randomUUID()}"
    var cardIdentificationDelegate: InaiCardIdentificationDelegate? = null
    private lateinit var transaction_id: String

    private val FCR = 4435
    private var _filePathCallback: ValueCallback<Array<Uri>>? = null
    private var coroutineScope = CoroutineScope(Job() + Dispatchers.IO)
    private var TAG = "InaiCheckoutFragment"
    companion object {

        fun newInstance(
            url: String,
            config: InaiBaseConfig,
            delegate: Any,
            viewMode: InaiViewMode,
            extraArgs: String = ""
        ): InaiCheckoutFragment {
            val checkoutFragment = InaiCheckoutFragment()
            checkoutFragment._url = url
            checkoutFragment._delegate = delegate
            checkoutFragment._config = config
            checkoutFragment._viewMode = viewMode
            checkoutFragment._extraArgs = extraArgs
            if(delegate is Fragment){
                Log.w(checkoutFragment.TAG,"'delegate' listener is implemented in Fragment, please implement 'delegate' listener in Activity")
            }
            return checkoutFragment
        }
    }

    private fun processActivityResult(resultCode: Int, data: Intent?) {
        var resultUriArray: MutableList<Uri> = mutableListOf()
        if (resultCode == Activity.RESULT_OK) {
            //  We have a valid result
            if (_filePathCallback != null) {
                data?.let{ data ->
                    data.clipData?.let { clipData ->
                        //  Multiple files uri
                        val itemCount = clipData.itemCount
                        for (i in 0 until itemCount) {
                            val clipDataItem = clipData.getItemAt(i)
                            val uri = clipDataItem.uri
                            resultUriArray.add(uri)
                        }
                    } ?: run {
                        data.dataString?.let { dataString ->
                            //  Single file uri
                            resultUriArray.add(Uri.parse(dataString))
                        }
                    }
                }
            }

            _filePathCallback!!.onReceiveValue(resultUriArray.toTypedArray())
            _filePathCallback = null
        }
    }

    private fun handleExtraArgs(){
        _extraArgs = if (_extraArgs.isNotEmpty()) {
            ",$_extraArgs,${getDeviceInfo()}"
        }else{
             ",${getDeviceInfo()}"
        }
    }

    private fun getDeviceInfo(): String{
        val info = JSONObject()
        val deviceInfo = JSONObject()
        deviceInfo.put("platform","ANDROID")
        deviceInfo.put("sdk_version",BuildConfig.InaiSdkVersion)
        info.put("deviceInfo", deviceInfo)
        return info.toString()
    }

    override fun onAttach(context: Context) {
        super.onAttach(context)
        InaiBaseUtils.updateBreadCrumbsValues(InaiBaseUtils.BreadcrumbsValues("", "ui.lifecycle", "info", InaiBaseUtils.getCurrentTimestamp(), "navigation", InaiBaseUtils.BreadcrumbsData("onAttach", TAG)))
        if (context is InaiCheckoutDelegate || context is InaiCardIdentificationDelegate || context is InaiPaymentMethodDelegate ||
            context is InaiInstallmentPlansDelegate || context is InaiExecuteDelegate || context is InaiValidateFieldsDelegate  || context is InaiCardInfoDelegate
        ) {
            if (!this::_delegate.isInitialized) {
                _delegate = (context as Any?)!!
            }
        }
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        InaiBaseUtils.updateBreadCrumbsValues(InaiBaseUtils.BreadcrumbsValues("", "ui.lifecycle", "info", InaiBaseUtils.getCurrentTimestamp(), "navigation", InaiBaseUtils.BreadcrumbsData("onCreate", TAG)))
    }

    @SuppressLint("SetJavaScriptEnabled")
    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View {
        context?.let { InaiBaseUtils().setup(it) }
        if(this::_config.isInitialized){
            InaiBaseUtils.saveInaiConfig(_config)
        }
        InaiBaseUtils.updateBreadCrumbsValues(InaiBaseUtils.BreadcrumbsValues("", "ui.lifecycle", "info", InaiBaseUtils.getCurrentTimestamp(), "navigation", InaiBaseUtils.BreadcrumbsData("onCreateView", TAG)))
        retainInstance = true
        //  Load the ui components
        if(savedInstanceState != null) {
            fragmentTag = savedInstanceState.getString(InaiConstants.FRAGMENT_TAG_KEY).toString()
            if(savedInstanceState.getString(InaiConstants.TRANSACTION_ID_KEY) != null){
                transaction_id = savedInstanceState.getString(InaiConstants.TRANSACTION_ID_KEY).toString()
            }
            _viewMode = savedInstanceState.getString(InaiConstants.VIEW_MODE_KEY)?.let { InaiViewMode.valueOf(it) }!!
            if(!this::rootView.isInitialized){
                InaiCrashlyticsHandler().logEvent("RootView is not initialized",null)
                rootView  = inflater.inflate(R.layout.inai_checkout_fragment, container, false)
                this.dismissWebView()
            }
        }else {
            rootView = inflater.inflate(R.layout.inai_checkout_fragment, container, false)
            loadingIndicator = rootView.findViewById(R.id.loading_indicator)
            webView = rootView.findViewById(R.id.webView)
            webView.settings.javaScriptEnabled = true
            webView.settings.domStorageEnabled = true
            webView.settings.javaScriptCanOpenWindowsAutomatically = true
            webView.settings.setSupportMultipleWindows(true)
            webView.settings.allowFileAccess = true
            webView.settings.allowContentAccess = true
            //  Register the js handler
            webView.addJavascriptInterface(this, "inaiHandler")
            webView.addJavascriptInterface(this, "inaiActionHandler")
            //Setting default data code & message
            inaiResult.data.put("code","PAYMENT_CANCELED")
            inaiResult.data.put("message","Payment Canceled by User")

            webView.webViewClient = object : WebViewClient() {

                override fun shouldOverrideUrlLoading(view: WebView, url: String): Boolean {
                    val uri: Uri = Uri.parse(url)
                    val queryParams = uri.queryParameterNames
                    val queryParamsJson = JSONObject()
                    for (item in queryParams) {
                        queryParamsJson.put(item, uri.getQueryParameter(item))
                    }
                    // condition to check if url does not start with http or https
                    if (!url.startsWith("https:") && !url.startsWith("http:")) {
                        //Allows to open UPI Apps on Loading upi URL
                        handleUPIUrl(url)
                        return true

                    } else if (
                        (queryParamsJson.has("transaction_id")
                                || queryParamsJson.has("transaction-id"))
                        && queryParamsJson.has("status")
                    ) {

                        var transactionId: String? = ""
                        if (queryParamsJson.has("transaction_id")) {
                            transactionId = queryParamsJson.get("transaction_id") as? String
                        } else if (queryParamsJson.has("transaction-id")) {
                            transactionId = queryParamsJson.get("transaction-id") as? String
                        }
                        val status = queryParamsJson.get("status") as? String

                        if (!status.isNullOrEmpty() && !transactionId.isNullOrEmpty()) {
                            //  We're looking for the transaction redirect url eg: https://payments.inai.io/v1/payment-status
                            //  ?transaction_id=e9a5fac1-dcc4-4cfc-a8ed-142bbdaa7ef1
                            //  &status=success
                            if ( status.equals("success", true)  || status.equals("pending", true)) {
                                inaiResult.status = InaiStatus.Success
                            } else {
                                inaiResult.status = InaiStatus.Failed
                            }
                            if (queryParamsJson.has("transaction-id")) {
                                queryParamsJson.put(
                                    "transaction_id",
                                    queryParamsJson.get("transaction-id")
                                )
                                queryParamsJson.remove("transaction-id")
                            }
                            inaiResult.data = queryParamsJson

                            //  Don't let the url load and finish the page
                            dismissWebView()
                            return true
                        }
                    }
                    //  Don't override other urls
                    return false
                }

                override fun onPageStarted(view: WebView?, url: String?, favicon: Bitmap?) {
                    loadingIndicator.visibility = View.VISIBLE
                }

                override fun onPageFinished(view: WebView?, url: String?) {
                    loadingIndicator.visibility = View.GONE
                    if (!_webViewLoaded) {
                        //  Call only once
                        //  Initialise the inai sdk with json data
                        webView.evaluateJavascript(
                            "window.webkit = { messageHandlers: { inaiHandler: window.inaiHandler, inaiActionHandler: window.inaiActionHandler} };",
                            null
                        )
                        //  Initialise the Inai SDK
                        val jsonString = Json.encodeToString(_config)
                        var initFn = ""

                        when (_viewMode) {
                            InaiViewMode.Payment -> {
                                initFn = "initPayment"
                            }
                            InaiViewMode.AddPaymentMethod -> {
                                initFn = "initAddPaymentMethod"
                            }
                            InaiViewMode.PayWithPaymentMethod -> {
                                initFn = "initPayWithPaymentMethod"
                            }
                            InaiViewMode.MakePayment -> {
                                initFn = "initMakePayment"
                            }
                            InaiViewMode.GetCardInfo -> {
                                initFn = "initGetCardInfo"
                            }
                            InaiViewMode.ValidateFields -> {
                                initFn = "initValidateFields"
                            }
                            InaiViewMode.AddPayoutMethod -> {
                                initFn = "initAddPayoutMethod"
                            }
                            InaiViewMode.ValidatePayoutFields -> {
                                initFn = "initValidatePayoutFields"
                            }
                            InaiViewMode.CryptoEstimate -> {
                                initFn = "initCryptoGetPurchaseEstimate"
                            }
                            InaiViewMode.InstallmentPlans ->{
                                initFn = "initGetInstallmentPlans"
                            }
                            InaiViewMode.Execute ->{
                                initFn = "initExecute"
                            }
                        }

                        handleExtraArgs()

                        var jsCode = "var jsArgs = $jsonString;"

                        if (cardIdentificationDelegate != null) {
                            //  Add onCardIdentification callback only if the delegate is present
                            jsCode += " jsArgs.onCardIdentification = (cardInfo) => { notifyApplication(convertToJSONString({cardInfo}));};"
                        }

                        jsCode += "$initFn(jsArgs$_extraArgs);"
                        webView.evaluateJavascript(jsCode, null)
                        _webViewLoaded = true
                    }
                }

                override fun onReceivedError(
                    view: WebView?,
                    request: WebResourceRequest?,
                    error: WebResourceError?
                ) {
                    loadingIndicator.visibility = View.GONE
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                        InaiCrashlyticsHandler().logEvent("Webview loading received Error ${error?.errorCode}","URL - ${request?.url} and reason - ${error?.description}")
                    }else{
                        InaiCrashlyticsHandler().logEvent("Webview loading received Error","URL- ${request?.url}")
                    }
                }

                override fun onReceivedSslError(
                    view: WebView?,
                    handler: SslErrorHandler?,
                    error: SslError?
                ) {
                    InaiCrashlyticsHandler().logEvent("Webview loading received SSL Error","Reason- ${error?.primaryError}")
                    super.onReceivedSslError(view, handler, error)
                }
            }
            webView.webChromeClient = object : WebChromeClient() {
                override fun onShowFileChooser(
                    webView: WebView,
                    filePathCallback: ValueCallback<Array<Uri>>,
                    fileChooserParams: FileChooserParams
                ): Boolean {
                    _filePathCallback = filePathCallback

                    val intent = fileChooserParams.createIntent()
                    //  Add multiple accept types as extra mime types
                    //  to enable multiple types of files to be selected
                    //  Without the following line we can only select the first type in the accept types list
                    intent.putExtra(Intent.EXTRA_MIME_TYPES, fileChooserParams.acceptTypes);
                    startActivityForResult(intent, FCR)
                    return true
                }

                override fun onJsAlert(
                    view: WebView?,
                    url: String?,
                    message: String?,
                    result: JsResult?
                ): Boolean {
                    return super.onJsAlert(view, url, message, result)
                }

                //  Function will be called whenever window.open() is called in js sdk.
                override fun onCreateWindow(
                    view: WebView?,
                    isDialog: Boolean,
                    isUserGesture: Boolean,
                    resultMsg: Message?
                ): Boolean {
                    //  Proceed only if resultMsg is not
                    resultMsg?.let {resultMessage ->
                        //  ResultMessage is cast as Webview.Transport object to intercept the URL.
                        val transport = resultMessage.obj as WebView.WebViewTransport
                        //  We create a new webview object and implement shouldOverrideUrlLoading. This
                        //  will intercept the the URL.
                        checkIfFragmentAttached {
                            val targetWebView = WebView(requireContext())
                            targetWebView.webViewClient = object : WebViewClient() {
                                override fun shouldOverrideUrlLoading(view: WebView, url: String): Boolean {
                                    //  We stop loading screen and handover the parsing od the URL to a browser.
                                    view.stopLoading()
                                    handleUPIUrl(url)
                                    return false
                                }
                            }
                            transport.webView = targetWebView;
                        }
                        resultMessage.sendToTarget();
                    }
                    return true
                }
            }

            //  Fire off the WebView
            webView.loadUrl(_url)
        }
        return rootView
    }

    override fun onStart() {
        super.onStart()
        InaiBaseUtils.updateBreadCrumbsValues(InaiBaseUtils.BreadcrumbsValues("Foreground", "ui.lifecycle", "info", InaiBaseUtils.getCurrentTimestamp(), "navigation", InaiBaseUtils.BreadcrumbsData("onStart", TAG)))
    }

    override fun onResume() {
        super.onResume()
        InaiBaseUtils.updateBreadCrumbsValues(InaiBaseUtils.BreadcrumbsValues("Foreground", "ui.lifecycle", "info", InaiBaseUtils.getCurrentTimestamp(), "navigation", InaiBaseUtils.BreadcrumbsData("onResume", TAG)))
    }

    //This function is to check whether the fragment requireContext() is not null
    fun checkIfFragmentAttached(operation: Context.() -> Unit) {
        if (isAdded && context != null) {
            operation(requireContext())
        }else{
            Log.d(TAG, "Fragment context not attached")
        }
    }

    @JavascriptInterface
    fun postMessage(message: String) {
        if (message.isNotEmpty()) {
            //  handle message from android
            try {
                val jsonObj = JSONObject(message)
                /*This action check is for inaiActionHandler message from js side*/
                if (jsonObj.has("actionKey") && jsonObj.has("actionPayload")) {
                    handleJsActions(jsonObj)
                    return
                }

                if (jsonObj.has("cardInfo")) {
                    activity?.runOnUiThread {
                        this.cardIdentificationDelegate?.onCardIdentification(jsonObj.getJSONObject("cardInfo"))
                    }
                    return
                }
                val inaiStatus = jsonObj.getString("inai_status")
                if (inaiStatus == "success") {
                    inaiResult.status = InaiStatus.Success
                } else {
                    inaiResult.status = InaiStatus.Failed
                }
                jsonObj.remove("inai_status")
                inaiResult.data = jsonObj
                dismissWebView()
            } catch (e: Exception) {
                Log.d(TAG, "postMessage exception - $e")
            }
        }
    }

    private fun handleJsActions(jsonObj: JSONObject) {
        var actionKey = jsonObj.getString("actionKey")

        when (actionKey) {
            "intentAppCheck" -> {
                performAppIntentCheck(jsonObj, actionKey)
            }
            "updateTransactionDetails" -> {
                setTransactionDetails(jsonObj)
            }
            "launchGooglePay" -> {
                handleGooglePay(jsonObj)
            }
        }
    }

    private fun handleGooglePay(jsonObj: JSONObject) {
        if (jsonObj.has("actionPayload")) {
            val paymentMethodJson = jsonObj.getJSONObject("actionPayload")
            checkIfFragmentAttached{
                InaiCheckout.initGooglePay(
                    paymentMethodJson,
                    requireActivity(),
                ) { googlePayRequestData ->
                    run {
                        if (googlePayRequestData != null) {
                            launchGooglePaymentSheet(googlePayRequestData)
                        } else {
                            InaiCrashlyticsHandler().logEvent("Google Pay request data Not Available",null)
                        }
                    }
                }
            }
        }
    }


    private fun setTransactionDetails(jsonObj: JSONObject) {
        if (jsonObj.has("actionPayload")) {
            val actionPayloadObj = jsonObj.getJSONObject("actionPayload")
            /* When the webview is closed by user this is what they will see in result data
               {"code": "PAYMENT_CANCELED",
                "message": "Payment Canceled by User",
                "transaction_id": "txn_123"
                } so added the below code */
            actionPayloadObj.put("code", "PAYMENT_CANCELED")
            actionPayloadObj.put("message", "Payment Canceled by User")
            inaiResult.data = actionPayloadObj
            if(actionPayloadObj.has("transaction_id")){
                transaction_id = actionPayloadObj.getString("transaction_id")
            }
        }
    }

    private fun performAppIntentCheck(jsonObj: JSONObject, actionKey: String) {
        val jsonArray = jsonObj.getJSONArray("actionPayload")
        val resultJsonArray = JSONArray()
        var hasPermission = false
        //To fetch list of installed UPI apps
        val upiAppList = getInstalledIntentApps(InaiConstants.DEFAULT_UPI_SCHEME)
        for (i in 0 until jsonArray.length()) {
            var item = jsonArray.getJSONObject(i)
            var packageName = item.getString("packageIdentifier")
            var appName = item.getString("appName")
            var url = item.getString("url")
            var rail = ""
            if(item.has("rail")){
                rail = item.getString("rail")
            }
            var isInstalled = isAppInstalled(packageName, upiAppList, url, rail)
            if (isInstalled) {
                hasPermission = true
            }
            var jsonObject = JSONObject()
            jsonObject.put("appName", appName)
            jsonObject.put("packageIdentifier", packageName)
            jsonObject.put("isInstalled", isInstalled)
            resultJsonArray.put(jsonObject)
        }
        var resultObj = JSONObject()
        resultObj.put("hasPermissions", hasPermission)
        resultObj.put("deeplinksApps", resultJsonArray)
        var notifyJsObj = JSONObject()
        notifyJsObj.put("notifyKey", actionKey)
        notifyJsObj.put("notifyPayload", resultObj)
        val resultString = notifyJsObj.toString()
        coroutineScope.launch(Dispatchers.Main){
            webView.evaluateJavascript("window.postMessage($resultString)", null)
        }
    }

    fun handleUPIUrl(url: String) {

        kotlin.runCatching {
            //To check if its paytm intent url
            if (url.startsWith(InaiConstants.PAYTM_URL_SCHEME)) {
                // modify intent url & set the paytm package to make paytm app redirection after payment completion
                val intentURL = url.replaceRange(0, 7, "upi")
                val intent = Intent(Intent.ACTION_VIEW, Uri.parse(intentURL))
                intent.setPackage(InaiConstants.PAYTM_PACKAGE_NAME)
                startActivity(intent)
            } else {
                startActivity(Intent(Intent.ACTION_VIEW, Uri.parse(url)))
            }
        }.getOrElse {
            val uri: Uri = Uri.parse(url)
            val queryParams = uri.queryParameterNames
            val queryParamsJson = JSONObject()
            val resultJson = JSONObject()
            for (item in queryParams) {
                queryParamsJson.put(item, uri.getQueryParameter(item))
            }
            if (
                (queryParamsJson.has("inai_transaction_id"))
            ) {

                var transactionId: String? = ""
                transactionId = queryParamsJson.get("inai_transaction_id") as? String

                if (!transactionId.isNullOrEmpty()) {
                    resultJson.put("transaction_id", transactionId)
                }
            }
            resultJson.put("code","APP_NOT_INSTALLED")
            resultJson.put("message", "Unable to open app to complete payment")
            inaiResult.data = resultJson
            inaiResult.status = InaiStatus.Failed

            dismissWebView()
        }
    }

    fun evalJS(jsStr: String) {
        webView.evaluateJavascript(jsStr, null)
    }

    fun activityResult(resultCode: Int, data: Intent) {
        processActivityResult(resultCode, data)
    }

    private fun isAppInstalled(
        packageName: String,
        upiAppList: ArrayList<String>,
        url: String,
        rail: String
    ): Boolean {
        //If rail code is UPI compare against the upiAppList for upi related apps check
        return if (rail.equals("upi", true)) {
            if (packageName.equals("android")) {
                //this check is for showing other upi apps
                upiAppList.size > 0
            } else {
                upiAppList.contains(packageName)
            }
        } else {
            //for other generic apps check
            val installedAppsList = getInstalledIntentApps(url)
            installedAppsList.contains(packageName)
        }
    }

    private fun getInstalledIntentApps(url: String): ArrayList<String> {
        val appList: ArrayList<String> = ArrayList()
        val uri = Uri.parse(url)
        val uriIntent = Intent()
        uriIntent.data = uri
        checkIfFragmentAttached{
            val packageManager: PackageManager =
                requireContext().packageManager
            val resolveInfoList =
                packageManager.queryIntentActivities(uriIntent, PackageManager.MATCH_DEFAULT_ONLY)
            if (resolveInfoList != null) {
                for (resolveInfo in resolveInfoList) {
                    appList.add(resolveInfo.activityInfo.packageName)
                }
            }
        }
        return appList
    }

    fun dismissWebView(ignoreDelegateCallback: Boolean = false) {
        //  CK - We need to run this code on UI thread since the webview will be on the UI thread
      activity?.runOnUiThread {
          //  Cleanup
          _webViewDismissed = true
          _ignoreDelegateCallback = ignoreDelegateCallback
          //  Call on main thread
          if(this::webView.isInitialized){
              webView.post { webView.stopLoading() }
          }

          //  dismiss the view
          checkIfFragmentAttached {
              requireActivity().supportFragmentManager
                  .findFragmentByTag(fragmentTag)?.let { fragment ->
                      requireActivity().supportFragmentManager.beginTransaction().remove(fragment)
                          .commitAllowingStateLoss()
                  }
          }
      }
    }

    override fun onPause() {
        super.onPause()
        InaiBaseUtils.updateBreadCrumbsValues(InaiBaseUtils.BreadcrumbsValues("Background", "ui.lifecycle", "info", InaiBaseUtils.getCurrentTimestamp(), "navigation", InaiBaseUtils.BreadcrumbsData("onPause", TAG)))
    }

    override fun onStop() {
        super.onStop()
        InaiBaseUtils.updateBreadCrumbsValues(InaiBaseUtils.BreadcrumbsValues("Background", "ui.lifecycle", "info", InaiBaseUtils.getCurrentTimestamp(), "navigation", InaiBaseUtils.BreadcrumbsData("onStop", TAG)))
    }

    override fun onDestroy() {
        super.onDestroy()
        InaiBaseUtils.updateBreadCrumbsValues(InaiBaseUtils.BreadcrumbsValues("", "ui.lifecycle", "info", InaiBaseUtils.getCurrentTimestamp(), "navigation", InaiBaseUtils.BreadcrumbsData("onDestroy", TAG)))
        if (_ignoreDelegateCallback) {
            //  Do not process any delegate callbacks
            return
        }

        //  View has been closed
        //  Notify the delegate
        if(this::_delegate.isInitialized) {
            when (_viewMode) {
                InaiViewMode.Payment -> {
                    val result = InaiPaymentResult()
                    result.data = inaiResult.data
                    if(!result.data.has("transaction_id") && this::transaction_id.isInitialized){
                        result.data.put("transaction_id", transaction_id)
                    }
                    result.status = InaiPaymentStatus.valueOf(inaiResult.status.toString())
                    (_delegate as InaiCheckoutDelegate).paymentFinished(result)
                }
                InaiViewMode.AddPaymentMethod -> {
                    val result = InaiPaymentMethodResult()
                    result.data = inaiResult.data
                    result.status = InaiPaymentMethodStatus.valueOf(inaiResult.status.toString())
                    (_delegate as InaiPaymentMethodDelegate).paymentMethodSaved(result)
                }
                InaiViewMode.PayWithPaymentMethod -> {
                    val result = InaiPaymentResult()
                    result.data = inaiResult.data
                    result.status = InaiPaymentStatus.valueOf(inaiResult.status.toString())
                    (_delegate as InaiCheckoutDelegate).paymentFinished(result)
                }
                InaiViewMode.MakePayment -> {
                    val result = InaiPaymentResult()
                    result.data = inaiResult.data
                    if(!result.data.has("transaction_id") && this::transaction_id.isInitialized){
                        result.data.put("transaction_id", transaction_id)
                    }
                    result.status = InaiPaymentStatus.valueOf(inaiResult.status.toString())
                    (_delegate as InaiCheckoutDelegate).paymentFinished(result)
                }
                InaiViewMode.GetCardInfo -> {
                    val result = InaiCardInfoResult()
                    result.data = inaiResult.data
                    result.status = InaiCardInfoStatus.valueOfOrElse(inaiResult.status.toString())
                    (_delegate as InaiCardInfoDelegate).cardInfoFetched(result)
                }
                InaiViewMode.ValidateFields -> {
                    val result = InaiValidateFieldsResult()
                    result.data = inaiResult.data
                    result.status = InaiValidateFieldsStatus.valueOfOrElse(inaiResult.status.toString())
                    (_delegate as InaiValidateFieldsDelegate).fieldsValidationFinished(result)
                }
                InaiViewMode.AddPayoutMethod -> {
                    val result = InaiPayOutMethodResult()
                    result.data = inaiResult.data
                    result.status = InaiPayOutMethodStatus.valueOf(inaiResult.status.toString())
                    (_delegate as InaiPayoutMethodDelegate).payoutMethodAdded(result)
                }
                InaiViewMode.ValidatePayoutFields -> {
                    val result = InaiValidateFieldsResult()
                    result.data = inaiResult.data
                    result.status = InaiValidateFieldsStatus.valueOfOrElse(inaiResult.status.toString())
                    (_delegate as InaiValidateFieldsDelegate).fieldsValidationFinished(result)
                }
                InaiViewMode.CryptoEstimate -> {
                    val result = InaiCrypto.InaiCryptoEstimateResult()
                    result.data = inaiResult.data
                    result.status = InaiCrypto.InaiCryptoEstimateStatus.valueOfOrElse(inaiResult.status.toString())
                    (_delegate as InaiCrypto.InaiCryptoEstimateDelegate).estimateReceived(result)
                }
                InaiViewMode.InstallmentPlans -> {
                    val result = InaiInstallmentPlansResult()
                    result.data = inaiResult.data
                    result.status = InaiInstallmentPlansInfoStatus.valueOf(inaiResult.status.toString())
                    (_delegate as InaiInstallmentPlansDelegate).installmentPlansFetched(result)
                }
                InaiViewMode.Execute -> {
                    val result = InaiExecuteResult()
                    result.data = inaiResult.data
                    result.status = InaiExecuteStatus.valueOf(inaiResult.status.toString())
                    (_delegate as InaiExecuteDelegate).executionFinished(result)
                }
            }
        }else{
            Log.d(TAG, "Delegate not initialized")
        }

        //  Unlock orientation to whatever was before.
        checkIfFragmentAttached {
            requireActivity().requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_USER
        }
    }

    override fun onDestroyView() {
        super.onDestroyView()
        InaiBaseUtils.updateBreadCrumbsValues(InaiBaseUtils.BreadcrumbsValues("", "ui.lifecycle", "info", InaiBaseUtils.getCurrentTimestamp(), "navigation", InaiBaseUtils.BreadcrumbsData("onDestroyView", TAG)))
    }

    override fun onSaveInstanceState(outState: Bundle) {
        super.onSaveInstanceState(outState)
        outState.putString(InaiConstants.FRAGMENT_TAG_KEY, fragmentTag)
        if(this::transaction_id.isInitialized){
            outState.putString(InaiConstants.TRANSACTION_ID_KEY, transaction_id)
        }
        outState.putString(InaiConstants.VIEW_MODE_KEY, _viewMode.name)
    }

    private fun launchGooglePaymentSheet(googlePayRequestData: InaiGooglePayRequestData) {
        val task = InaiCheckout.launchGooglePayRequest(googlePayRequestData)
        task.addOnCompleteListener { completedTask ->
            if (completedTask.isSuccessful) {
                completedTask.result.let(::handleGooglePaySuccess)
            } else {
                when (val exception = completedTask.exception) {
                    is ResolvableApiException -> {
                        resolvePaymentForResult.launch(
                            IntentSenderRequest.Builder(exception.resolution).build()
                        )
                    }
                    is ApiException -> {
                        val errorStr =
                            "ApiException Error code ${exception.statusCode} ${exception.message}"
                        Log.d(TAG, errorStr)
                    }
                    else -> {
                        val errorStr =
                            "Error ${CommonStatusCodes.INTERNAL_ERROR} Unexpected exception"
                        Log.d(TAG, errorStr)
                    }
                }
            }
        }
    }

    private val resolvePaymentForResult =
        registerForActivityResult(ActivityResultContracts.StartIntentSenderForResult()) { result: ActivityResult ->
            when (result.resultCode) {
                AppCompatActivity.RESULT_OK ->
                    result.data?.let { intent ->
                        PaymentData.getFromIntent(intent)?.let(::handleGooglePaySuccess)
                    }

                AppCompatActivity.RESULT_CANCELED -> {
                    // The user canceled the payment attempt
                    Log.d(TAG,"Google Pay sheet Canceled")
                }
            }
        }

    private fun handleGooglePaySuccess(paymentData: PaymentData) {
        //  Google Auth successful
        try {
            //  send the paymentData to the webview
            val googlePaymentMethodData = JSONObject(paymentData.toJson())
            coroutineScope.launch(Dispatchers.Main){
                var notifyJsObj = JSONObject()
                notifyJsObj.put("notifyKey", "launchGooglePay")
                notifyJsObj.put("notifyPayload", googlePaymentMethodData)
                val resultString = notifyJsObj.toString()
                coroutineScope.launch(Dispatchers.Main){
                    webView.evaluateJavascript("window.postMessage($resultString)", null)
                }
            }
        }catch (e: Exception) {
            Log.d(TAG, e.toString())
        }
    }
}