package io.inai.android_sdk

import android.annotation.SuppressLint
import android.app.Activity
import android.content.Intent
import android.graphics.Bitmap
import android.net.Uri
import android.os.Bundle
import android.os.Message
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.webkit.*
import android.widget.ProgressBar
import androidx.fragment.app.Fragment
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import org.json.JSONObject


enum class InaiViewMode {
    Payment, AddPaymentMethod, PayWithPaymentMethod, MakePayment, GetCardInfo, ValidateFields,
    AddPayoutMethod, ValidatePayoutFields, CryptoEstimate
}

class InaiCheckoutFragment : Fragment() {

    private lateinit var webView: WebView
    private lateinit var loadingIndicator: ProgressBar

    private lateinit var _delegate: Any
    private lateinit var _config: InaiBaseConfig
    private var _url: String = ""
    private var inaiResult: InaiResult = InaiResult()
    private var _viewMode: InaiViewMode = InaiViewMode.Payment
    private var _extraArgs: String = ""
    private var _webViewLoaded: Boolean = false
    private var _webViewDismissed: Boolean = false
    private var _ignoreDelegateCallback: Boolean = false
    val fragmentTag: String = "inai_checkout_${java.util.UUID.randomUUID()}"
    var cardIdentificationDelegate: InaiCardIdentificationDelegate? = null

    private val FCR = 4435
    private var _filePathCallback: ValueCallback<Array<Uri>>? = null

    companion object {
        fun newInstance(
            url: String,
            config: InaiBaseConfig,
            delegate: Any,
            viewMode: InaiViewMode,
            extraArgs: String = ""
        ): InaiCheckoutFragment {
            val checkoutFragment = InaiCheckoutFragment()
            checkoutFragment._url = url
            checkoutFragment._delegate = delegate
            checkoutFragment._config = config
            checkoutFragment._viewMode = viewMode
            checkoutFragment._extraArgs = extraArgs
            return checkoutFragment
        }
    }

    private fun processActivityResult(resultCode: Int, data: Intent?) {
        var resultUriArray: MutableList<Uri> = mutableListOf()
        if (resultCode == Activity.RESULT_OK) {
            //  We have a valid result
            if (_filePathCallback != null) {
                data?.let{ data ->
                    data.clipData?.let { clipData ->
                        //  Multiple files uri
                        val itemCount = clipData.itemCount
                        for (i in 0 until itemCount) {
                            val clipDataItem = clipData.getItemAt(i)
                            val uri = clipDataItem.uri
                            resultUriArray.add(uri)
                        }
                    } ?: run {
                        data.dataString?.let { dataString ->
                            //  Single file uri
                            resultUriArray.add(Uri.parse(dataString))
                        }
                    }
                }
            }

            _filePathCallback!!.onReceiveValue(resultUriArray.toTypedArray())
            _filePathCallback = null
        }
    }

    @SuppressLint("SetJavaScriptEnabled")
    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View {
        val rootView: View = inflater.inflate(R.layout.inai_checkout_fragment, container, false)

        //  Load the ui components
        loadingIndicator = rootView.findViewById(R.id.loading_indicator)
        webView = rootView.findViewById(R.id.webView)
        webView.settings.javaScriptEnabled = true
        webView.settings.domStorageEnabled = true
        webView.settings.javaScriptCanOpenWindowsAutomatically = true
        webView.settings.setSupportMultipleWindows(true)
        webView.settings.allowFileAccess = true
        webView.settings.allowContentAccess = true

        //  Register the js handler
        webView.addJavascriptInterface(this, "inaiHandler")
        webView.webViewClient = object : WebViewClient() {

            override fun shouldOverrideUrlLoading(view: WebView, url: String): Boolean {
                val uri: Uri = Uri.parse(url)
                val queryParams = uri.queryParameterNames
                val queryParamsJson = JSONObject()
                for (item in queryParams) {
                    queryParamsJson.put(item, uri.getQueryParameter(item))
                }

                if (
                    (queryParamsJson.has("transaction_id")
                            || queryParamsJson.has("transaction-id"))
                    && queryParamsJson.has("status")
                ) {

                    var transactionId: String? = ""
                    if (queryParamsJson.has("transaction_id")) {
                        transactionId = queryParamsJson.get("transaction_id") as? String
                    } else if (queryParamsJson.has("transaction-id")) {
                        transactionId = queryParamsJson.get("transaction-id") as? String
                    }
                    val status = queryParamsJson.get("status") as? String

                    if (!status.isNullOrEmpty() && !transactionId.isNullOrEmpty()) {
                        //  We're looking for the transaction redirect url eg: https://payments.inai.io/v1/payment-status
                        //  ?transaction_id=e9a5fac1-dcc4-4cfc-a8ed-142bbdaa7ef1
                        //  &status=success
                        if (status == "success") {
                            inaiResult.status = InaiStatus.Success
                        } else {
                            inaiResult.status = InaiStatus.Failed
                        }
                        if(queryParamsJson.has("transaction-id")){
                            queryParamsJson.put("transaction_id", queryParamsJson.get("transaction-id"))
                            queryParamsJson.remove("transaction-id")
                        }
                        inaiResult.data = queryParamsJson

                        //  Don't let the url load and finish the page
                        dismissWebView()
                        return true
                    }
                }
                //  Don't override other urls
                return false
            }

            override fun onPageStarted(view: WebView?, url: String?, favicon: Bitmap?) {
                loadingIndicator.visibility = View.VISIBLE
            }

            override fun onPageFinished(view: WebView?, url: String?) {
                loadingIndicator.visibility = View.GONE
                if (!_webViewLoaded) {
                    //  Call only once
                    //  Initialise the inai sdk with json data
                    webView.evaluateJavascript(
                        "window.webkit = { messageHandlers: { inaiHandler: window.inaiHandler} };",
                        null
                    )

                    //  Initialise the Inai SDK
                    val jsonString = Json.encodeToString(_config)
                    var initFn = ""

                    when (_viewMode) {
                        InaiViewMode.Payment -> {
                            initFn = "initPayment"
                        }
                        InaiViewMode.AddPaymentMethod -> {
                            initFn = "initAddPaymentMethod"
                        }
                        InaiViewMode.PayWithPaymentMethod -> {
                            initFn = "initPayWithPaymentMethod"
                        }
                        InaiViewMode.MakePayment -> {
                            initFn = "initMakePayment"
                        }
                        InaiViewMode.GetCardInfo -> {
                            initFn = "initGetCardInfo"
                        }
                        InaiViewMode.ValidateFields -> {
                            initFn = "initValidateFields"
                        }
                        InaiViewMode.AddPayoutMethod -> {
                            initFn = "initAddPayoutMethod"
                        }
                        InaiViewMode.ValidatePayoutFields -> {
                            initFn = "initValidatePayoutFields"
                        }
                        InaiViewMode.CryptoEstimate -> {
                            initFn = "initCryptoGetPurchaseEstimate"
                        }
                    }

                    if (_extraArgs.isNotEmpty()) {
                        _extraArgs = ",$_extraArgs"
                    }

                    var jsCode = "var jsArgs = $jsonString;"

                    if (cardIdentificationDelegate != null) {
                        //  Add onCardIdentification callback only if the delegate is present
                        jsCode += " jsArgs.onCardIdentification = (cardInfo) => { notifyApplication(convertToJSONString({cardInfo}));};"
                    }

                    jsCode += "$initFn(jsArgs$_extraArgs);"
                    webView.evaluateJavascript(jsCode, null)
                    _webViewLoaded = true
                }
            }

            override fun onReceivedError(
                view: WebView?,
                request: WebResourceRequest?,
                error: WebResourceError?
            ) {
                loadingIndicator.visibility = View.GONE
            }
        }
        webView.webChromeClient = object : WebChromeClient() {
            override fun onShowFileChooser(
                webView: WebView,
                filePathCallback: ValueCallback<Array<Uri>>,
                fileChooserParams: FileChooserParams
            ): Boolean {
                _filePathCallback = filePathCallback

                val intent = fileChooserParams.createIntent()
                //  Add multiple accept types as extra mime types
                //  to enable multiple types of files to be selected
                //  Without the following line we can only select the first type in the accept types list
                intent.putExtra(Intent.EXTRA_MIME_TYPES, fileChooserParams.acceptTypes);
                startActivityForResult(intent, FCR)
                return true
            }

            override fun onJsAlert(
                view: WebView?,
                url: String?,
                message: String?,
                result: JsResult?
            ): Boolean {
                return super.onJsAlert(view, url, message, result)
            }

            //  Function will be called whenever window.open() is called in js sdk.
            override fun onCreateWindow(
                view: WebView?,
                isDialog: Boolean,
                isUserGesture: Boolean,
                resultMsg: Message?
            ): Boolean {
                //  Proceed only if resultMsg is not
                resultMsg?.let {resultMessage ->
                    //  ResultMessage is cast as Webview.Transport object to intercept the URL.
                    val transport = resultMessage.obj as WebView.WebViewTransport
                    //  We create a new webview object and implement shouldOverrideUrlLoading. This
                    //  will intercept the the URL.
                    val targetWebView = WebView(requireContext())
                    targetWebView.webViewClient = object : WebViewClient() {
                        override fun shouldOverrideUrlLoading(view: WebView, url: String): Boolean {
                            //  We stop loading screen and handover the parsing od the URL to a browser.
                            view.stopLoading()
                            kotlin.runCatching {
                                startActivity(Intent(Intent.ACTION_VIEW, Uri.parse(url)))
                            }.getOrElse {
                                val uri: Uri = Uri.parse(url)
                                val queryParams = uri.queryParameterNames
                                val queryParamsJson = JSONObject()
                                val resultJson = JSONObject()
                                for (item in queryParams) {
                                    queryParamsJson.put(item, uri.getQueryParameter(item))
                                }
                                if (
                                    (queryParamsJson.has("inai_transaction_id"))
                                ) {

                                    var transactionId: String? = ""
                                    transactionId = queryParamsJson.get("inai_transaction_id") as? String

                                    if (!transactionId.isNullOrEmpty()) {
                                        resultJson.put("transaction_id", transactionId)
                                    }
                                }
                                resultJson.put("code","APP_NOT_INSTALLED")
                                resultJson.put("message", "Unable to open app to complete payment")
                                inaiResult.data = resultJson
                                inaiResult.status = InaiStatus.Failed

                                dismissWebView()
                            }
                            return false
                        }
                    }
                    transport.webView = targetWebView;
                    resultMessage.sendToTarget();
                }
                return true
            }
        }

        //  Fire off the WebView
        webView.loadUrl(_url)
        return rootView
    }

    @JavascriptInterface
    fun postMessage(message: String) {
        if (message.isNotEmpty()) {
            //  handle message from android
            val jsonObj = JSONObject(message)

            if (jsonObj.has("cardInfo")) {
                activity?.runOnUiThread {
                    this.cardIdentificationDelegate?.onCardIdentification(jsonObj.getJSONObject("cardInfo"))
                }
                return
            }

            val inaiStatus = jsonObj.getString("inai_status")
            if (inaiStatus == "success") {
                inaiResult.status = InaiStatus.Success
            } else {
                inaiResult.status = InaiStatus.Failed
            }

            jsonObj.remove("inai_status")
            inaiResult.data = jsonObj
            dismissWebView()
        }
    }

    fun evalJS(jsStr: String) {
        webView.evaluateJavascript(jsStr, null)
    }

    fun activityResult(resultCode: Int, data: Intent) {
        processActivityResult(resultCode, data)
    }

    fun dismissWebView(ignoreDelegateCallback: Boolean = false) {
        //  CK - We need to run this code on UI thread since the webview will be on the UI thread
      activity?.runOnUiThread {
          //  Cleanup
          _webViewDismissed = true
          _ignoreDelegateCallback = ignoreDelegateCallback
          //  Call on main thread
          webView.post { webView.stopLoading() }

          //  dismiss the view
          requireActivity().supportFragmentManager
              .findFragmentByTag(fragmentTag)?.let { fragment ->
                  requireActivity().supportFragmentManager.beginTransaction().remove(fragment)
                      .commitAllowingStateLoss()
              }
      }
    }

    override fun onDestroy() {
        super.onDestroy()

        if (_ignoreDelegateCallback) {
            //  Do not process any delegate callbacks
            return
        }

        //  View has been closed
        //  Notify the delegate
        when (_viewMode) {
            InaiViewMode.Payment -> {
                val result = InaiPaymentResult()
                result.data = inaiResult.data
                result.status = InaiPaymentStatus.valueOf(inaiResult.status.toString())
                (_delegate as InaiCheckoutDelegate).paymentFinished(result)
            }
            InaiViewMode.AddPaymentMethod -> {
                val result = InaiPaymentMethodResult()
                result.data = inaiResult.data
                result.status = InaiPaymentMethodStatus.valueOf(inaiResult.status.toString())
                (_delegate as InaiPaymentMethodDelegate).paymentMethodSaved(result)
            }
            InaiViewMode.PayWithPaymentMethod -> {
                val result = InaiPaymentResult()
                result.data = inaiResult.data
                result.status = InaiPaymentStatus.valueOf(inaiResult.status.toString())
                (_delegate as InaiCheckoutDelegate).paymentFinished(result)
            }
            InaiViewMode.MakePayment -> {
                val result = InaiPaymentResult()
                result.data = inaiResult.data
                result.status = InaiPaymentStatus.valueOf(inaiResult.status.toString())
                (_delegate as InaiCheckoutDelegate).paymentFinished(result)
            }
            InaiViewMode.GetCardInfo -> {
                val result = InaiCardInfoResult()
                result.data = inaiResult.data
                result.status = InaiCardInfoStatus.valueOfOrElse(inaiResult.status.toString())
                (_delegate as InaiCardInfoDelegate).cardInfoFetched(result)
            }
            InaiViewMode.ValidateFields -> {
                val result = InaiValidateFieldsResult()
                result.data = inaiResult.data
                result.status = InaiValidateFieldsStatus.valueOfOrElse(inaiResult.status.toString())
                (_delegate as InaiValidateFieldsDelegate).fieldsValidationFinished(result)
            }
            InaiViewMode.AddPayoutMethod -> {
                val result = InaiPayOutMethodResult()
                result.data = inaiResult.data
                result.status = InaiPayOutMethodStatus.valueOf(inaiResult.status.toString())
                (_delegate as InaiPayoutMethodDelegate).payoutMethodAdded(result)
            }
            InaiViewMode.ValidatePayoutFields -> {
                val result = InaiValidateFieldsResult()
                result.data = inaiResult.data
                result.status = InaiValidateFieldsStatus.valueOfOrElse(inaiResult.status.toString())
                (_delegate as InaiValidateFieldsDelegate).fieldsValidationFinished(result)
            }
            InaiViewMode.CryptoEstimate -> {
                val result = InaiCrypto.InaiCryptoEstimateResult()
                result.data = inaiResult.data
                result.status = InaiCrypto.InaiCryptoEstimateStatus.valueOfOrElse(inaiResult.status.toString())
                (_delegate as InaiCrypto.InaiCryptoEstimateDelegate).estimateReceived(result)
            }
        }
    }
}