package io.hellgate.android.sdk.service

import arrow.core.getOrElse
import arrow.core.raise.either
import arrow.core.raise.ensure
import arrow.fx.coroutines.closeable
import arrow.fx.coroutines.resourceScope
import io.hellgate.android.sdk.client.HttpClientError
import io.hellgate.android.sdk.client.basistheory.BasisTheoryClient
import io.hellgate.android.sdk.client.basistheory.btClient
import io.hellgate.android.sdk.client.hellgate.*
import io.hellgate.android.sdk.client.hellgate.HgClient
import io.hellgate.android.sdk.client.hellgate.SessionResponse
import io.hellgate.android.sdk.client.hellgate.hgClient
import io.hellgate.android.sdk.element.additionaldata.AdditionalDataTypes
import io.hellgate.android.sdk.model.CardData
import io.hellgate.android.sdk.model.TokenizeCardResponse

internal interface ITokenService {
    /**
     * Tokenize a card
     * @param sessionId The session id created with a x-api-key on the hg-backend
     * @param cardNumber The card number
     * @param year The year of the card 2 digits
     * @param month The month of the card 2 digits
     * @param cvc The cvc of the card
     */
    suspend fun tokenize(
        sessionId: String,
        cardData: CardData,
        additionalData: Map<AdditionalDataTypes, String>,
    ): TokenizeCardResponse
}

internal fun tokenService(
    hgUrl: String,
    hellgateClient: () -> HgClient = { hgClient(hgUrl) },
    basisClient: () -> BasisTheoryClient = { btClient() },
): ITokenService =
    object : ITokenService {
        override suspend fun tokenize(
            sessionId: String,
            cardData: CardData,
            additionalData: Map<AdditionalDataTypes, String>,
        ): TokenizeCardResponse =
            either {
                resourceScope {
                    val hgClient = closeable { hellgateClient() }
                    val info = hgClient.fetchSession(sessionId).bind()

                    ensure(info.nextAction == NextAction.TOKENIZE_CARD) { HttpClientError("Tokenization failed") }
                    ensure(info.data != null) { HttpClientError("Tokenization failed") }
                    ensure(info.data is SessionResponse.Data.BtApiKey) { HttpClientError("Tokenization failed") }

                    // TODO Think of goods ways to respond on a BT-Failure.
                    //  How can we handle this?
                    //  How can we inform the integrator that some more information is needed?
                    val btClient = closeable { basisClient() }
                    val response = btClient.tokenizeCard(info.data.btApiKey, cardData).bind()

                    val result = hgClient.completeTokenizeCard(sessionId, response.id, additionalData).bind()
                    ensure(result.status == "success") { HttpClientError("Tokenization failed") }
                    ensure(result.data != null) { HttpClientError("Tokenization failed") }
                    ensure(result.data is SessionResponse.Data.TokenId) { HttpClientError("Tokenization failed") }
                    TokenizeCardResponse.Success(result.data.tokenId)
                }
            }.getOrElse {
                TokenizeCardResponse.Failure(it.message, it.throwable)
            }
    }
