package io.hellgate.android.sdk

import androidx.annotation.MainThread
import arrow.core.getOrElse
import arrow.core.raise.either
import io.hellgate.android.sdk.Constants.HG_STAGING_URL
import io.hellgate.android.sdk.client.hellgate.*
import io.hellgate.android.sdk.client.hellgate.NextAction
import io.hellgate.android.sdk.handler.*
import io.hellgate.android.sdk.model.InvalidSessionState
import io.hellgate.android.sdk.service.tokenService
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock

internal object Constants {
    const val HG_STAGING_URL = "https://staging.hellgate.dev"
}

interface Hellgate {
    @MainThread
    suspend fun fetchSessionStatus(): SessionState

    @MainThread
    suspend fun cardHandler(): Result<CardHandler>
}

@MainThread
fun initHellgate(
    hgBaseUrl: String = HG_STAGING_URL,
    sessionId: String,
): Hellgate = internalHellgate(hgBaseUrl, sessionId)

internal fun internalHellgate(
    hgBaseUrl: String,
    sessionId: String,
    client: () -> HgClient = { hgClient(hgBaseUrl) },
) = object : Hellgate {
    private val mutex = Mutex()
    private var sessionState = SessionState.UNKNOWN

    override suspend fun fetchSessionStatus(): SessionState =
        mutex.withLock {
            either {
                val sessionInfo = client().use { it.fetchSession(sessionId) }.bind()
                when (sessionInfo.nextAction) {
                    // TODO session response is not really unambiguous, clear up the logic
                    NextAction.TOKENIZE_CARD -> SessionState.REQUIRE_TOKENIZATION
                    NextAction.WAIT -> SessionState.WAITING

                    NextAction.TDS_CHECK -> TODO()
                    NextAction.TDS_CHALLENGE -> TODO()
                    NextAction.TDS_DECOUPLED -> TODO()

                    null -> when (sessionInfo.status) {
                        "success" -> SessionState.COMPLETED
                        else -> SessionState.UNKNOWN
                        // TODO or maybe it was closed? but that would be a 404?
                    }
                }
            }.getOrElse { SessionState.UNKNOWN }
                .also { sessionState = it }
        }

    override suspend fun cardHandler(): Result<CardHandler> {
        fetchSessionStatus()
        return mutex.withLock {
            if (sessionState != SessionState.REQUIRE_TOKENIZATION) {
                Result.failure(InvalidSessionState.notTokenizeCard(sessionState.name))
            } else {
                Result.success(cardHandler(tokenService(hgBaseUrl), sessionId))
            }
        }
    }
}

enum class SessionState {
    REQUIRE_TOKENIZATION,
    WAITING,
    COMPLETED,
    UNKNOWN,
}
