package io.hellgate.android.sdk.client.hellgate

import arrow.core.Either
import arrow.core.raise.either
import io.hellgate.android.sdk.Mock
import io.hellgate.android.sdk.MockScenario
import io.hellgate.android.sdk.client.*
import io.hellgate.android.sdk.client.hellgate.Segments.COMPLETE_ACTION
import io.hellgate.android.sdk.client.hellgate.Segments.SESSIONS
import io.hellgate.android.sdk.client.hellgate.SessionCompleteTokenizeCard.*
import io.hellgate.android.sdk.client.hellgate.SessionCompleteTokenizeCard.AdditionalData.Companion.toDTO
import io.hellgate.android.sdk.element.additionaldata.AdditionalDataTypes
import io.ktor.client.HttpClient
import io.ktor.http.HttpMethod
import java.io.Closeable

internal object Segments {
    const val SESSIONS = "sessions"
    const val COMPLETE_ACTION = "complete-action"
}

internal interface HgClient : Closeable {
    suspend fun fetchSession(sessionId: String): Either<HttpClientError, SessionInformation>

    suspend fun completeTokenizeCard(
        sessionId: String,
        btToken: String,
        additionalData: Map<AdditionalDataTypes, String>
    ): Either<HttpClientError, SessionCompleteResponse>
}

internal fun hgClient(
    baseUrl: String,
    client: HttpClient = httpClient(),
) = object : HgClient {

    override fun close() {
        client.close()
    }

    override suspend fun fetchSession(sessionId: String): Either<HttpClientError, SessionInformation> =
        client.eitherRequest<SessionInformation, Unit>(
            HttpMethod.Get,
            baseUrl,
            listOf(SESSIONS, sessionId),
        )

    override suspend fun completeTokenizeCard(
        sessionId: String,
        btToken: String,
        additionalData: Map<AdditionalDataTypes, String>
    ): Either<HttpClientError, SessionCompleteResponse> = client.eitherRequest<SessionCompleteResponse, SessionCompleteTokenizeCard>(
        HttpMethod.Post,
        baseUrl,
        listOf(SESSIONS, sessionId, COMPLETE_ACTION),
        body = SessionCompleteTokenizeCard(
            result = Result(
                btToken,
                if (additionalData.isEmpty()) null else additionalData.toDTO()
            ),
        ),
    )
}

// TODO remove this when we have a real backend
internal fun hgSpyClient(
    baseUrl: String,
    client: HttpClient = httpClient(),
) = object : HgClient {

    override fun close() {
        client.close()
    }

    val hgClient = hgClient(baseUrl, client)

    override suspend fun fetchSession(sessionId: String): Either<HttpClientError, SessionInformation> = either {
        when (Mock.scenario) {
            is MockScenario.TokenizeCard -> hgClient.fetchSession(sessionId).bind()
            else -> {
                when (Mock.scenario.nextAction) {
                    NextAction.TOKENIZE_CARD -> hgClient.fetchSession(sessionId).bind()
                    NextAction.WAIT -> SessionInformation(SessionInformation.Data(""), NextAction.WAIT, null)
                    NextAction.TDS_CHECK -> SessionInformation(SessionInformation.Data(""), NextAction.TDS_CHECK, null)
                    null -> SessionInformation(SessionInformation.Data(""), null, "success")
                    else -> hgClient.fetchSession(sessionId).bind()
                }.also { Mock.scenario.progressScenario() }
            }
        }
    }

    override suspend fun completeTokenizeCard(
        sessionId: String,
        btToken: String,
        additionalData: Map<AdditionalDataTypes, String>
    ): Either<HttpClientError, SessionCompleteResponse> = client.eitherRequest<SessionCompleteResponse, SessionCompleteTokenizeCard>(
        HttpMethod.Post,
        baseUrl,
        listOf(SESSIONS, sessionId, COMPLETE_ACTION),
        body = SessionCompleteTokenizeCard(
            result = Result(
                btToken,
                if (additionalData.isEmpty()) null else additionalData.toDTO()
            ),
        ),
    )
}
