package io.hellgate.android.sdk.client.basistheory

import arrow.core.Either
import io.hellgate.android.sdk.client.*
import io.hellgate.android.sdk.model.CardData
import io.ktor.client.HttpClient
import io.ktor.http.HttpMethod
import io.ktor.http.headersOf
import java.io.Closeable

internal const val BT_URL = "https://api.basistheory.com"
internal const val BT_KEY_HEADER = "BT-API-KEY"
internal const val TOKENIZE_PATH = "tokenize"

internal interface BasisTheoryClient : Closeable {
    suspend fun tokenizeCard(
        btApiKey: String,
        cardData: CardData
    ): Either<HttpClientError, BtResponse>
}

internal fun btClient(
    baseUrl: String = BT_URL,
    client: HttpClient = httpClient(),
) = object : BasisTheoryClient {

    override fun close() {
        client.close()
    }

    override suspend fun tokenizeCard(
        btApiKey: String,
        cardData: CardData
    ): Either<HttpClientError, BtResponse> =
        client.eitherRequest<BtResponse, BtTokenizeRequest>(
            method = HttpMethod.Post,
            baseUrl = baseUrl,
            pathSegments = listOf(TOKENIZE_PATH),
            additionalHeaders = headersOf(BT_KEY_HEADER, btApiKey),
            body = BtTokenizeRequest(
                BtTokenizeRequest.Data.fromCardData(cardData)
            ),
        )
}
