package io.hellgate.android.sdk

import androidx.annotation.MainThread
import arrow.core.getOrElse
import arrow.core.raise.either
import io.hellgate.android.sdk.Constants.HG_STAGING_URL
import io.hellgate.android.sdk.client.hellgate.NextAction
import io.hellgate.android.sdk.client.hellgate.hgSpyClient
import io.hellgate.android.sdk.handler.*
import io.hellgate.android.sdk.model.InvalidSessionState
import io.hellgate.android.sdk.service.tokenService
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock

internal object Constants {
    const val HG_STAGING_URL = "https://staging.hellgate.dev"
}

interface Hellgate {
    @MainThread
    suspend fun fetchSessionStatus(): SessionState

    @MainThread
    suspend fun cardHandler(): Result<CardHandler>

    @MainThread
    suspend fun tdsHandler(): Result<TDSHandler>
}

fun initHellgate(
    hgBaseUrl: String = HG_STAGING_URL,
    sessionId: String
) = object : Hellgate {

    private val mutex = Mutex()
    var sessionState = SessionState.WAITING

    override suspend fun fetchSessionStatus(): SessionState = mutex.withLock {
        either {
            val sessionInfo = hgSpyClient(hgBaseUrl).use { it.fetchSession(sessionId) }.bind()
            when (sessionInfo.nextAction) {
                // TODO session response is not really unambiguous, clear up the logic
                NextAction.TOKENIZE_CARD -> SessionState.REQUIRE_TOKENIZATION
                NextAction.TDS_CHECK -> SessionState.REQUIRE_3DS
                NextAction.WAIT -> SessionState.WAITING
                null -> when (sessionInfo.status) {
                    "success" -> SessionState.COMPLETED
                    else -> SessionState.UNKNOWN
                    // TODO or maybe it was closed? but that would be a 404?
                }

                NextAction.TDS_CHALLENGE -> TODO()
                NextAction.TDS_DECOUPLED -> TODO()
            }
        }.getOrElse { SessionState.UNKNOWN }
    }

    override suspend fun cardHandler(): Result<CardHandler> =
        mutex.withLock {
            if (sessionState == SessionState.REQUIRE_TOKENIZATION) {
                Result.failure(InvalidSessionState.notTokenizeCard(sessionState.name))
            } else {
                Result.success(initCardHandler(tokenService(hgBaseUrl), sessionId))
            }
        }

    override suspend fun tdsHandler(): Result<TDSHandler> =
        mutex.withLock {
            if (sessionState == SessionState.REQUIRE_3DS) {
                Result.failure(InvalidSessionState.notTDSToComplete(sessionState.name))
            } else {
                Result.success(initTDSHandler())
            }
        }
}

enum class SessionState {
    CREATED,
    REQUIRE_TOKENIZATION,
    REQUIRE_3DS,
    WAITING,
    COMPLETED,
    UNKNOWN
}
