/*
 * Copyright (c) 2024 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.webserver;

import java.net.InetAddress;
import java.net.SocketOption;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.ServiceLoader;
import java.util.function.Consumer;
import java.util.function.Supplier;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Errors;
import io.helidon.common.Generated;
import io.helidon.common.HelidonServiceLoader;
import io.helidon.common.config.Config;
import io.helidon.common.context.Context;
import io.helidon.common.socket.SocketOptions;
import io.helidon.common.tls.Tls;
import io.helidon.common.tls.TlsConfig;
import io.helidon.http.RequestedUriDiscoveryContext;
import io.helidon.http.encoding.ContentEncodingContext;
import io.helidon.http.encoding.ContentEncodingContextConfig;
import io.helidon.http.media.MediaContext;
import io.helidon.http.media.MediaContextConfig;
import io.helidon.webserver.http.DirectHandlers;
import io.helidon.webserver.http.HttpRouting;
import io.helidon.webserver.spi.ProtocolConfig;
import io.helidon.webserver.spi.ProtocolConfigProvider;
import io.helidon.webserver.spi.ServerConnectionSelector;

/**
 * Configuration of a server listener (server socket).
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.codegen.BuilderCodegen", trigger = "io.helidon.webserver.ListenerConfigBlueprint")
public interface ListenerConfig extends ListenerConfigBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static ListenerConfig.Builder builder() {
        return new ListenerConfig.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static ListenerConfig.Builder builder(ListenerConfig instance) {
        return ListenerConfig.builder().from(instance);
    }

    /**
     * Create a new instance from configuration.
     *
     * @param config used to configure the new instance
     * @return a new instance configured from configuration
     */
    static ListenerConfig create(Config config) {
        return ListenerConfig.builder().config(config).buildPrototype();
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static ListenerConfig create() {
        return ListenerConfig.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link ListenerConfig}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends ListenerConfig.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends ListenerConfig> implements Prototype.ConfiguredBuilder<BUILDER, PROTOTYPE> {

        private final List<io.helidon.common.Builder<?, ? extends Routing>> routings = new ArrayList<>();
        private final List<ProtocolConfig> protocols = new ArrayList<>();
        private final List<ServerConnectionSelector> connectionSelectors = new ArrayList<>();
        private final Map<SocketOption<?>, Object> listenerSocketOptions = new LinkedHashMap<>();
        private boolean enableProxyProtocol = false;
        private boolean isConnectionSelectorsMutated;
        private boolean isProtocolsMutated;
        private boolean isRoutingsMutated;
        private boolean protocolsDiscoverServices = true;
        private HttpRouting.Builder routing;
        private Config config;
        private ConnectionConfig connectionConfig;
        private ContentEncodingContext contentEncoding;
        private Context listenerContext;
        private DirectHandlers directHandlers;
        private Duration idleConnectionPeriod = Duration.parse("PT2M");
        private Duration idleConnectionTimeout = Duration.parse("PT5M");
        private Duration shutdownGracePeriod = Duration.parse("PT0.5S");
        private InetAddress address;
        private int backlog = 1024;
        private int maxConcurrentRequests = -1;
        private int maxInMemoryEntity = 131072;
        private int maxTcpConnections = -1;
        private int port = 0;
        private int writeBufferSize = 512;
        private int writeQueueLength = 0;
        private Integer receiveBufferSize;
        private long maxPayloadSize = -1;
        private MediaContext mediaContext;
        private RequestedUriDiscoveryContext requestedUriDiscoveryContext;
        private SocketOptions connectionOptions;
        private String host = "0.0.0.0";
        private String name = "@default";
        private Tls tls;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance. This method disables automatic service discovery.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(ListenerConfig prototype) {
            if (!isProtocolsMutated) {
                protocols.clear();
            }
            addProtocols(prototype.protocols());
            protocolsDiscoverServices = false;
            routing(prototype.routing());
            if (!isRoutingsMutated) {
                routings.clear();
            }
            addRoutings(prototype.routings());
            name(prototype.name());
            host(prototype.host());
            address(prototype.address());
            port(prototype.port());
            backlog(prototype.backlog());
            maxPayloadSize(prototype.maxPayloadSize());
            receiveBufferSize(prototype.receiveBufferSize());
            writeQueueLength(prototype.writeQueueLength());
            writeBufferSize(prototype.writeBufferSize());
            shutdownGracePeriod(prototype.shutdownGracePeriod());
            connectionConfig(prototype.connectionConfig());
            tls(prototype.tls());
            contentEncoding(prototype.contentEncoding());
            mediaContext(prototype.mediaContext());
            connectionOptions(prototype.connectionOptions());
            maxTcpConnections(prototype.maxTcpConnections());
            maxConcurrentRequests(prototype.maxConcurrentRequests());
            idleConnectionTimeout(prototype.idleConnectionTimeout());
            idleConnectionPeriod(prototype.idleConnectionPeriod());
            maxInMemoryEntity(prototype.maxInMemoryEntity());
            addListenerSocketOptions(prototype.listenerSocketOptions());
            if (!isConnectionSelectorsMutated) {
                connectionSelectors.clear();
            }
            addConnectionSelectors(prototype.connectionSelectors());
            directHandlers(prototype.directHandlers());
            listenerContext(prototype.listenerContext());
            enableProxyProtocol(prototype.enableProxyProtocol());
            requestedUriDiscoveryContext(prototype.requestedUriDiscoveryContext());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(ListenerConfig.BuilderBase<?, ?> builder) {
            if (isProtocolsMutated) {
                if (builder.isProtocolsMutated) {
                    addProtocols(builder.protocols);
                }
            } else {
                protocols.clear();
                addProtocols(builder.protocols);
            }
            protocolsDiscoverServices = builder.protocolsDiscoverServices;
            builder.routing().ifPresent(this::routing);
            if (isRoutingsMutated) {
                if (builder.isRoutingsMutated) {
                    addRoutings(builder.routings);
                }
            } else {
                routings.clear();
                addRoutings(builder.routings);
            }
            name(builder.name());
            host(builder.host());
            builder.address().ifPresent(this::address);
            port(builder.port());
            backlog(builder.backlog());
            maxPayloadSize(builder.maxPayloadSize());
            builder.receiveBufferSize().ifPresent(this::receiveBufferSize);
            writeQueueLength(builder.writeQueueLength());
            writeBufferSize(builder.writeBufferSize());
            shutdownGracePeriod(builder.shutdownGracePeriod());
            builder.connectionConfig().ifPresent(this::connectionConfig);
            builder.tls().ifPresent(this::tls);
            builder.contentEncoding().ifPresent(this::contentEncoding);
            builder.mediaContext().ifPresent(this::mediaContext);
            builder.connectionOptions().ifPresent(this::connectionOptions);
            maxTcpConnections(builder.maxTcpConnections());
            maxConcurrentRequests(builder.maxConcurrentRequests());
            idleConnectionTimeout(builder.idleConnectionTimeout());
            idleConnectionPeriod(builder.idleConnectionPeriod());
            maxInMemoryEntity(builder.maxInMemoryEntity());
            addListenerSocketOptions(builder.listenerSocketOptions);
            if (isConnectionSelectorsMutated) {
                if (builder.isConnectionSelectorsMutated) {
                    addConnectionSelectors(builder.connectionSelectors);
                }
            } else {
                connectionSelectors.clear();
                addConnectionSelectors(builder.connectionSelectors);
            }
            builder.directHandlers().ifPresent(this::directHandlers);
            builder.listenerContext().ifPresent(this::listenerContext);
            enableProxyProtocol(builder.enableProxyProtocol());
            builder.requestedUriDiscoveryContext().ifPresent(this::requestedUriDiscoveryContext);
            return self();
        }

        /**
         * Customize HTTP routing of this listener.
         *
         * @param builderConsumer consumer of HTTP Routing builder
         * @return updated builder instance
         */
        public BUILDER routing(Consumer<HttpRouting.Builder> builderConsumer) {
            WebServerConfigSupport.ListenerCustomMethods.routing(this, builderConsumer);
            return self();
        }

        /**
         * Update builder from configuration (node of this type).
         * If a value is present in configuration, it would override currently configured values.
         *
         * @param config configuration instance used to obtain values to update this builder
         * @return updated builder instance
         */
        @Override
        public BUILDER config(Config config) {
            Objects.requireNonNull(config);
            this.config = config;
            config.get("name").as(String.class).ifPresent(this::name);
            config.get("host").as(String.class).ifPresent(this::host);
            config.get("port").as(Integer.class).ifPresent(this::port);
            config.get("backlog").as(Integer.class).ifPresent(this::backlog);
            config.get("max-payload-size").as(Long.class).ifPresent(this::maxPayloadSize);
            config.get("receive-buffer-size").as(Integer.class).ifPresent(this::receiveBufferSize);
            config.get("write-queue-length").as(Integer.class).ifPresent(this::writeQueueLength);
            config.get("write-buffer-size").as(Integer.class).ifPresent(this::writeBufferSize);
            config.get("shutdown-grace-period").as(Duration.class).ifPresent(this::shutdownGracePeriod);
            config.get("connection-config").map(ConnectionConfig::create).ifPresent(this::connectionConfig);
            config.get("tls").map(TlsConfig::create).ifPresent(this::tls);
            config.get("content-encoding").map(ContentEncodingContextConfig::create).ifPresent(this::contentEncoding);
            config.get("media-context").map(MediaContextConfig::create).ifPresent(this::mediaContext);
            config.get("connection-options").map(SocketOptions::create).ifPresent(this::connectionOptions);
            config.get("max-tcp-connections").as(Integer.class).ifPresent(this::maxTcpConnections);
            config.get("max-concurrent-requests").as(Integer.class).ifPresent(this::maxConcurrentRequests);
            config.get("idle-connection-timeout").as(Duration.class).ifPresent(this::idleConnectionTimeout);
            config.get("idle-connection-period").as(Duration.class).ifPresent(this::idleConnectionPeriod);
            config.get("max-in-memory-entity").as(Integer.class).ifPresent(this::maxInMemoryEntity);
            config.get("requested-uri-discovery").map(RequestedUriDiscoveryContext::create).ifPresent(this::requestedUriDiscoveryContext);
            return self();
        }

        /**
         * Configuration of protocols. This may be either protocol selectors, or protocol upgraders from HTTP/1.1.
         * As the order is not important (providers are ordered by weight by default), we can use a configuration as an object,
         * such as:
         * <pre>
         * protocols:
         *   providers:
         *     http_1_1:
         *       max-prologue-length: 8192
         *     http_2:
         *       max-frame-size: 4096
         *     websocket:
         *       ....
         * </pre>
         *
         * @param discoverServices whether to discover implementations through service loader
         * @return updated builder instance
         * @see #protocols()
         */
        public BUILDER protocolsDiscoverServices(boolean discoverServices) {
            this.protocolsDiscoverServices = discoverServices;
            return self();
        }

        /**
         * Configuration of protocols. This may be either protocol selectors, or protocol upgraders from HTTP/1.1.
         * As the order is not important (providers are ordered by weight by default), we can use a configuration as an object,
         * such as:
         * <pre>
         * protocols:
         *   providers:
         *     http_1_1:
         *       max-prologue-length: 8192
         *     http_2:
         *       max-frame-size: 4096
         *     websocket:
         *       ....
         * </pre>
         *
         * @param protocols all defined protocol configurations, loaded from service loader by default
         * @return updated builder instance
         * @see #protocols()
         */
        public BUILDER protocols(List<? extends ProtocolConfig> protocols) {
            Objects.requireNonNull(protocols);
            isProtocolsMutated = true;
            this.protocols.clear();
            this.protocols.addAll(protocols);
            return self();
        }

        /**
         * Configuration of protocols. This may be either protocol selectors, or protocol upgraders from HTTP/1.1.
         * As the order is not important (providers are ordered by weight by default), we can use a configuration as an object,
         * such as:
         * <pre>
         * protocols:
         *   providers:
         *     http_1_1:
         *       max-prologue-length: 8192
         *     http_2:
         *       max-frame-size: 4096
         *     websocket:
         *       ....
         * </pre>
         *
         * @param protocols all defined protocol configurations, loaded from service loader by default
         * @return updated builder instance
         * @see #protocols()
         */
        public BUILDER addProtocols(List<? extends ProtocolConfig> protocols) {
            Objects.requireNonNull(protocols);
            isProtocolsMutated = true;
            this.protocols.addAll(protocols);
            return self();
        }

        /**
         * Configuration of protocols. This may be either protocol selectors, or protocol upgraders from HTTP/1.1.
         * As the order is not important (providers are ordered by weight by default), we can use a configuration as an object,
         * such as:
         * <pre>
         * protocols:
         *   providers:
         *     http_1_1:
         *       max-prologue-length: 8192
         *     http_2:
         *       max-frame-size: 4096
         *     websocket:
         *       ....
         * </pre>
         *
         * @param protocol all defined protocol configurations, loaded from service loader by default
         * @return updated builder instance
         * @see #protocols()
         */
        public BUILDER addProtocol(ProtocolConfig protocol) {
            Objects.requireNonNull(protocol);
            this.protocols.add(protocol);
            isProtocolsMutated = true;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #routing()
         */
        public BUILDER clearRouting() {
            this.routing = null;
            return self();
        }

        /**
         * Http routing. This will always be added to the resulting {@link io.helidon.webserver.Router}, if defined,
         *  overriding any HTTP routing already present.
         * If a custom listener has routing defined, it will be used, otherwise routing defined on web server will be used.
         *
         * @param routing HTTP Routing for this listener/server
         * @return updated builder instance
         * @see #routing()
         */
        public BUILDER routing(HttpRouting.Builder routing) {
            Objects.requireNonNull(routing);
            this.routing = routing;
            return self();
        }

        /**
         * List of all routings (possibly for multiple protocols). This allows adding non-http protocols as well,
         * as opposed to {@link #routing()}
         *
         * @param routings router for this listener/server
         * @return updated builder instance
         * @see #routings()
         */
        public BUILDER routings(List<io.helidon.common.Builder<?, ? extends Routing>> routings) {
            Objects.requireNonNull(routings);
            isRoutingsMutated = true;
            this.routings.clear();
            this.routings.addAll(routings);
            return self();
        }

        /**
         * List of all routings (possibly for multiple protocols). This allows adding non-http protocols as well,
         * as opposed to {@link #routing()}
         *
         * @param routings router for this listener/server
         * @return updated builder instance
         * @see #routings()
         */
        public BUILDER addRoutings(List<io.helidon.common.Builder<?, ? extends Routing>> routings) {
            Objects.requireNonNull(routings);
            isRoutingsMutated = true;
            this.routings.addAll(routings);
            return self();
        }

        /**
         * List of all routings (possibly for multiple protocols). This allows adding non-http protocols as well,
         * as opposed to {@link #routing()}
         *
         * @param routing router for this listener/server
         * @return updated builder instance
         * @see #routings()
         */
        public BUILDER addRouting(io.helidon.common.Builder<?, ? extends Routing> routing) {
            Objects.requireNonNull(routing);
            this.routings.add(routing);
            isRoutingsMutated = true;
            return self();
        }

        /**
         * Name of this socket. Defaults to {@code @default}.
         * Must be defined if more than one socket is needed.
         *
         * @param name name of the socket
         * @return updated builder instance
         * @see #name()
         */
        public BUILDER name(String name) {
            Objects.requireNonNull(name);
            this.name = name;
            return self();
        }

        /**
         * Host of the default socket. Defaults to all host addresses ({@code 0.0.0.0}).
         *
         * @param host host address to listen on (for the default socket)
         * @return updated builder instance
         * @see #host()
         */
        public BUILDER host(String host) {
            Objects.requireNonNull(host);
            this.host = host;
            return self();
        }

        /**
         * Address to use. If both this and {@link #host()} is configured, this will be used.
         *
         * @param address address to use
         * @return updated builder instance
         * @see #address()
         */
        public BUILDER address(InetAddress address) {
            Objects.requireNonNull(address);
            this.address = address;
            return self();
        }

        /**
         * Port of the default socket.
         * If configured to {@code 0} (the default), server starts on a random port.
         *
         * @param port port to listen on (for the default socket)
         * @return updated builder instance
         * @see #port()
         */
        public BUILDER port(int port) {
            this.port = port;
            return self();
        }

        /**
         * Accept backlog.
         *
         * @param backlog backlog
         * @return updated builder instance
         * @see #backlog()
         */
        public BUILDER backlog(int backlog) {
            this.backlog = backlog;
            return self();
        }

        /**
         * Maximal number of bytes an entity may have.
         * If {@link io.helidon.http.HeaderNames#CONTENT_LENGTH} is used, this is checked immediately,
         * if {@link io.helidon.http.HeaderValues#TRANSFER_ENCODING_CHUNKED} is used, we will fail when the
         * number of bytes read would exceed the max payload size.
         * Defaults to unlimited ({@code -1}).
         *
         * @param maxPayloadSize maximal number of bytes of entity
         * @return updated builder instance
         * @see #maxPayloadSize()
         */
        public BUILDER maxPayloadSize(long maxPayloadSize) {
            this.maxPayloadSize = maxPayloadSize;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #receiveBufferSize()
         */
        public BUILDER clearReceiveBufferSize() {
            this.receiveBufferSize = null;
            return self();
        }

        /**
         * Listener receive buffer size.
         *
         * @param receiveBufferSize buffer size in bytes
         * @return updated builder instance
         * @see #receiveBufferSize()
         */
        public BUILDER receiveBufferSize(int receiveBufferSize) {
            Objects.requireNonNull(receiveBufferSize);
            this.receiveBufferSize = receiveBufferSize;
            return self();
        }

        /**
         * Number of buffers queued for write operations.
         *
         * @param writeQueueLength maximal number of queued writes, defaults to 0
         * @return updated builder instance
         * @see #writeQueueLength()
         */
        public BUILDER writeQueueLength(int writeQueueLength) {
            this.writeQueueLength = writeQueueLength;
            return self();
        }

        /**
         * Initial buffer size in bytes of {@link java.io.BufferedOutputStream} created internally to
         * write data to a socket connection. Default is {@code 512}.
         *
         * @param writeBufferSize initial buffer size used for writing
         * @return updated builder instance
         * @see #writeBufferSize()
         */
        public BUILDER writeBufferSize(int writeBufferSize) {
            this.writeBufferSize = writeBufferSize;
            return self();
        }

        /**
         * Grace period in ISO 8601 duration format to allow running tasks to complete before listener's shutdown.
         * Default is {@code 500} milliseconds.
         * <p>Configuration file values example: {@code PT0.5S}, {@code PT2S}.
         *
         * @param shutdownGracePeriod grace period
         * @return updated builder instance
         * @see #shutdownGracePeriod()
         */
        public BUILDER shutdownGracePeriod(Duration shutdownGracePeriod) {
            Objects.requireNonNull(shutdownGracePeriod);
            this.shutdownGracePeriod = shutdownGracePeriod;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #connectionConfig()
         */
        public BUILDER clearConnectionConfig() {
            this.connectionConfig = null;
            return self();
        }

        /**
         * Configuration of a connection (established from client against our server).
         *
         * @param connectionConfig connection configuration
         * @return updated builder instance
         * @see #connectionConfig()
         */
        public BUILDER connectionConfig(ConnectionConfig connectionConfig) {
            Objects.requireNonNull(connectionConfig);
            this.connectionConfig = connectionConfig;
            return self();
        }

        /**
         * Configuration of a connection (established from client against our server).
         *
         * @param consumer connection configuration
         * @return updated builder instance
         * @see #connectionConfig()
         */
        public BUILDER connectionConfig(Consumer<ConnectionConfig.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = ConnectionConfig.builder();
            consumer.accept(builder);
            this.connectionConfig(builder.build());
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #tls()
         */
        public BUILDER clearTls() {
            this.tls = null;
            return self();
        }

        /**
         * Listener TLS configuration.
         *
         * @param tls tls of this configuration
         * @return updated builder instance
         * @see #tls()
         */
        public BUILDER tls(Tls tls) {
            Objects.requireNonNull(tls);
            this.tls = tls;
            return self();
        }

        /**
         * Listener TLS configuration.
         *
         * @param tlsConfig tls of this configuration
         * @return updated builder instance
         * @see #tls()
         */
        public BUILDER tls(TlsConfig tlsConfig) {
            Objects.requireNonNull(tlsConfig);
            this.tls = Tls.create(tlsConfig);
            return self();
        }

        /**
         * Listener TLS configuration.
         *
         * @param consumer tls of this configuration
         * @return updated builder instance
         * @see #tls()
         */
        public BUILDER tls(Consumer<TlsConfig.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = TlsConfig.builder();
            consumer.accept(builder);
            this.tls(builder.build());
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #contentEncoding()
         */
        public BUILDER clearContentEncoding() {
            this.contentEncoding = null;
            return self();
        }

        /**
         * Configure the listener specific {@link io.helidon.http.encoding.ContentEncodingContext}.
         * This method discards all previously registered ContentEncodingContext.
         * If no content encoding context is registered, content encoding context of the webserver would be used.
         *
         * @param contentEncoding content encoding context
         * @return updated builder instance
         * @see #contentEncoding()
         */
        public BUILDER contentEncoding(ContentEncodingContext contentEncoding) {
            Objects.requireNonNull(contentEncoding);
            this.contentEncoding = contentEncoding;
            return self();
        }

        /**
         * Configure the listener specific {@link io.helidon.http.encoding.ContentEncodingContext}.
         * This method discards all previously registered ContentEncodingContext.
         * If no content encoding context is registered, content encoding context of the webserver would be used.
         *
         * @param contentEncodingConfig content encoding context
         * @return updated builder instance
         * @see #contentEncoding()
         */
        public BUILDER contentEncoding(ContentEncodingContextConfig contentEncodingConfig) {
            Objects.requireNonNull(contentEncodingConfig);
            this.contentEncoding = ContentEncodingContext.create(contentEncodingConfig);
            return self();
        }

        /**
         * Configure the listener specific {@link io.helidon.http.encoding.ContentEncodingContext}.
         * This method discards all previously registered ContentEncodingContext.
         * If no content encoding context is registered, content encoding context of the webserver would be used.
         *
         * @param consumer content encoding context
         * @return updated builder instance
         * @see #contentEncoding()
         */
        public BUILDER contentEncoding(Consumer<ContentEncodingContextConfig.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = ContentEncodingContextConfig.builder();
            consumer.accept(builder);
            this.contentEncoding(builder.build());
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #mediaContext()
         */
        public BUILDER clearMediaContext() {
            this.mediaContext = null;
            return self();
        }

        /**
         * Configure the listener specific {@link io.helidon.http.media.MediaContext}.
         * This method discards all previously registered MediaContext.
         * If no media context is registered, media context of the webserver would be used.
         *
         * @param mediaContext media context
         * @return updated builder instance
         * @see #mediaContext()
         */
        public BUILDER mediaContext(MediaContext mediaContext) {
            Objects.requireNonNull(mediaContext);
            this.mediaContext = mediaContext;
            return self();
        }

        /**
         * Configure the listener specific {@link io.helidon.http.media.MediaContext}.
         * This method discards all previously registered MediaContext.
         * If no media context is registered, media context of the webserver would be used.
         *
         * @param mediaContextConfig media context
         * @return updated builder instance
         * @see #mediaContext()
         */
        public BUILDER mediaContext(MediaContextConfig mediaContextConfig) {
            Objects.requireNonNull(mediaContextConfig);
            this.mediaContext = MediaContext.create(mediaContextConfig);
            return self();
        }

        /**
         * Configure the listener specific {@link io.helidon.http.media.MediaContext}.
         * This method discards all previously registered MediaContext.
         * If no media context is registered, media context of the webserver would be used.
         *
         * @param consumer media context
         * @return updated builder instance
         * @see #mediaContext()
         */
        public BUILDER mediaContext(Consumer<MediaContextConfig.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = MediaContextConfig.builder();
            consumer.accept(builder);
            this.mediaContext(builder.build());
            return self();
        }

        /**
         * Options for connections accepted by this listener.
         * This is not used to setup server connection.
         *
         * @param connectionOptions socket options
         * @return updated builder instance
         * @see #connectionOptions()
         */
        public BUILDER connectionOptions(SocketOptions connectionOptions) {
            Objects.requireNonNull(connectionOptions);
            this.connectionOptions = connectionOptions;
            return self();
        }

        /**
         * Options for connections accepted by this listener.
         * This is not used to setup server connection.
         *
         * @param consumer consumer of builder for
         *                 socket options
         * @return updated builder instance
         * @see #connectionOptions()
         */
        public BUILDER connectionOptions(Consumer<SocketOptions.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = SocketOptions.builder();
            consumer.accept(builder);
            this.connectionOptions(builder.build());
            return self();
        }

        /**
         * Options for connections accepted by this listener.
         * This is not used to setup server connection.
         *
         * @param supplier supplier of
         *                 socket options
         * @return updated builder instance
         * @see #connectionOptions()
         */
        public BUILDER connectionOptions(Supplier<? extends SocketOptions> supplier) {
            Objects.requireNonNull(supplier);
            this.connectionOptions(supplier.get());
            return self();
        }

        /**
         * Limits the number of connections that can be opened at a single point in time.
         * Defaults to {@code -1}, meaning "unlimited" - what the system allows.
         *
         * @param maxTcpConnections number of TCP connections that can be opened to this listener, regardless of protocol
         * @return updated builder instance
         * @see #maxTcpConnections()
         */
        public BUILDER maxTcpConnections(int maxTcpConnections) {
            this.maxTcpConnections = maxTcpConnections;
            return self();
        }

        /**
         * Limits the number of requests that can be executed at the same time (the number of active virtual threads of requests).
         * Defaults to {@code -1}, meaning "unlimited" - what the system allows.
         * Also make sure that this number is higher than the expected time it takes to handle a single request in your application,
         * as otherwise you may stop in-progress requests.
         *
         * @param maxConcurrentRequests number of requests that can be processed on this listener, regardless of protocol
         * @return updated builder instance
         * @see #maxConcurrentRequests()
         */
        public BUILDER maxConcurrentRequests(int maxConcurrentRequests) {
            this.maxConcurrentRequests = maxConcurrentRequests;
            return self();
        }

        /**
         * How long should we wait before closing a connection that has no traffic on it.
         * Defaults to {@code PT5M} (5 minutes). Note that the timestamp is refreshed max. once per second, so this setting
         * would be useless if configured for shorter periods of time (also not a very good support for connection keep alive,
         * if the connections are killed so soon anyway).
         *
         * @param idleConnectionTimeout timeout of idle connections
         * @return updated builder instance
         * @see #idleConnectionTimeout()
         */
        public BUILDER idleConnectionTimeout(Duration idleConnectionTimeout) {
            Objects.requireNonNull(idleConnectionTimeout);
            this.idleConnectionTimeout = idleConnectionTimeout;
            return self();
        }

        /**
         * How often should we check for {@link #idleConnectionTimeout()}.
         * Defaults to {@code PT2M} (2 minutes).
         *
         * @param idleConnectionPeriod period of checking for idle connections
         * @return updated builder instance
         * @see #idleConnectionPeriod()
         */
        public BUILDER idleConnectionPeriod(Duration idleConnectionPeriod) {
            Objects.requireNonNull(idleConnectionPeriod);
            this.idleConnectionPeriod = idleConnectionPeriod;
            return self();
        }

        /**
         * If the entity is expected to be smaller that this number of bytes, it would be buffered in memory to optimize
         * performance when writing it.
         * If bigger, streaming will be used.
         * <p>
         * Note that for some entity types we cannot use streaming, as they are already fully in memory (String, byte[]), for such
         * cases, this option is ignored.
         * <p>
         * Default is 128Kb.
         *
         * @param maxInMemoryEntity maximal number of bytes to buffer in memory for supported writers
         * @return updated builder instance
         * @see #maxInMemoryEntity()
         */
        public BUILDER maxInMemoryEntity(int maxInMemoryEntity) {
            this.maxInMemoryEntity = maxInMemoryEntity;
            return self();
        }

        /**
         * This method replaces all values with the new ones.
         *
         * @param listenerSocketOptions custom socket options
         * @return updated builder instance
         * @see #listenerSocketOptions()
         */
        public BUILDER listenerSocketOptions(Map<SocketOption<?>, ?> listenerSocketOptions) {
            Objects.requireNonNull(listenerSocketOptions);
            this.listenerSocketOptions.clear();
            this.listenerSocketOptions.putAll(listenerSocketOptions);
            return self();
        }

        /**
         * This method keeps existing values, then puts all new values into the map.
         *
         * @param listenerSocketOptions custom socket options
         * @return updated builder instance
         * @see #listenerSocketOptions()
         */
        public BUILDER addListenerSocketOptions(Map<SocketOption<?>, ?> listenerSocketOptions) {
            Objects.requireNonNull(listenerSocketOptions);
            this.listenerSocketOptions.putAll(listenerSocketOptions);
            return self();
        }

        /**
         * This method adds a new value to the map, or replaces it if the key already exists.
         *
         * @param key key to add or replace
         * @param listenerSocketOption new value for the key
         * @param <TYPE> Type to correctly map key and value
         * @return updated builder instance
         * @see #listenerSocketOptions()
         */
        public <TYPE extends Object> BUILDER putListenerSocketOption(SocketOption<TYPE> key, TYPE listenerSocketOption) {
            Objects.requireNonNull(key);
            Objects.requireNonNull(listenerSocketOption);
            this.listenerSocketOptions.put(key, listenerSocketOption);
            return self();
        }

        /**
         * Explicitly defined connection selectors to be used with this socket.
         * This list is augmented with the result of {@link #protocols()}, but custom selectors are always used first.
         *
         * @param connectionSelectors connection selectors to be used for this socket
         * @return updated builder instance
         * @see #connectionSelectors()
         */
        public BUILDER connectionSelectors(List<? extends ServerConnectionSelector> connectionSelectors) {
            Objects.requireNonNull(connectionSelectors);
            isConnectionSelectorsMutated = true;
            this.connectionSelectors.clear();
            this.connectionSelectors.addAll(connectionSelectors);
            return self();
        }

        /**
         * Explicitly defined connection selectors to be used with this socket.
         * This list is augmented with the result of {@link #protocols()}, but custom selectors are always used first.
         *
         * @param connectionSelectors connection selectors to be used for this socket
         * @return updated builder instance
         * @see #connectionSelectors()
         */
        public BUILDER addConnectionSelectors(List<? extends ServerConnectionSelector> connectionSelectors) {
            Objects.requireNonNull(connectionSelectors);
            isConnectionSelectorsMutated = true;
            this.connectionSelectors.addAll(connectionSelectors);
            return self();
        }

        /**
         * Explicitly defined connection selectors to be used with this socket.
         * This list is augmented with the result of {@link #protocols()}, but custom selectors are always used first.
         *
         * @param connectionSelector connection selectors to be used for this socket
         * @return updated builder instance
         * @see #connectionSelectors()
         */
        public BUILDER addConnectionSelector(ServerConnectionSelector connectionSelector) {
            Objects.requireNonNull(connectionSelector);
            this.connectionSelectors.add(connectionSelector);
            isConnectionSelectorsMutated = true;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #directHandlers()
         */
        public BUILDER clearDirectHandlers() {
            this.directHandlers = null;
            return self();
        }

        /**
         * Direct handlers specific for this listener.
         * A direct handler takes care of problems that happen before (or outside of) routing, such as bad request.
         *
         * @param directHandlers direct handlers
         * @return updated builder instance
         * @see #directHandlers()
         */
        public BUILDER directHandlers(DirectHandlers directHandlers) {
            Objects.requireNonNull(directHandlers);
            this.directHandlers = directHandlers;
            return self();
        }

        /**
         * Direct handlers specific for this listener.
         * A direct handler takes care of problems that happen before (or outside of) routing, such as bad request.
         *
         * @param consumer direct handlers
         * @return updated builder instance
         * @see #directHandlers()
         */
        public BUILDER directHandlers(Consumer<DirectHandlers.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = DirectHandlers.builder();
            consumer.accept(builder);
            this.directHandlers(builder.build());
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #listenerContext()
         */
        public BUILDER clearListenerContext() {
            this.listenerContext = null;
            return self();
        }

        /**
         * Listener scoped context to be used as a parent for webserver request contexts (if used).
         * If an explicit context is used, you need to take care of correctly configuring its parent.
         * It is expected that the parent of this context is the WebServer context. You should also configure explicit
         * WebServer context when using this method
         *
         * @param listenerContext listener context
         * @return updated builder instance
         * @see #listenerContext()
         */
        public BUILDER listenerContext(Context listenerContext) {
            Objects.requireNonNull(listenerContext);
            this.listenerContext = listenerContext;
            return self();
        }

        /**
         * Listener scoped context to be used as a parent for webserver request contexts (if used).
         * If an explicit context is used, you need to take care of correctly configuring its parent.
         * It is expected that the parent of this context is the WebServer context. You should also configure explicit
         * WebServer context when using this method
         *
         * @param consumer listener context
         * @return updated builder instance
         * @see WebServerConfig#serverContext()
         * @see #listenerContext()
         */
        public BUILDER listenerContext(Consumer<Context.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = Context.builder();
            consumer.accept(builder);
            this.listenerContext(builder.build());
            return self();
        }

        /**
         * Enable proxy protocol support for this socket. This protocol is supported by
         * some load balancers/reverse proxies as a means to convey client information that
         * would otherwise be lost. If enabled, the proxy protocol header must be present
         * on every new connection established with your server. For more information,
         * see <a href="https://www.haproxy.org/download/1.8/doc/proxy-protocol.txt">
         * the specification</a>. Default is {@code false}.
         *
         * @param enableProxyProtocol proxy support status
         * @return updated builder instance
         * @see #enableProxyProtocol()
         */
        public BUILDER enableProxyProtocol(boolean enableProxyProtocol) {
            this.enableProxyProtocol = enableProxyProtocol;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #requestedUriDiscoveryContext()
         */
        public BUILDER clearRequestedUriDiscoveryContext() {
            this.requestedUriDiscoveryContext = null;
            return self();
        }

        /**
         * Requested URI discovery context.
         *
         * @param requestedUriDiscoveryContext discovery context
         * @return updated builder instance
         * @see #requestedUriDiscoveryContext()
         */
        public BUILDER requestedUriDiscoveryContext(RequestedUriDiscoveryContext requestedUriDiscoveryContext) {
            Objects.requireNonNull(requestedUriDiscoveryContext);
            this.requestedUriDiscoveryContext = requestedUriDiscoveryContext;
            return self();
        }

        /**
         * Requested URI discovery context.
         *
         * @param consumer discovery context
         * @return updated builder instance
         * @see #requestedUriDiscoveryContext()
         */
        public BUILDER requestedUriDiscoveryContext(Consumer<RequestedUriDiscoveryContext.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = RequestedUriDiscoveryContext.builder();
            consumer.accept(builder);
            this.requestedUriDiscoveryContext(builder.build());
            return self();
        }

        /**
         * Configuration of protocols. This may be either protocol selectors, or protocol upgraders from HTTP/1.1.
         * As the order is not important (providers are ordered by weight by default), we can use a configuration as an object,
         * such as:
         * <pre>
         * protocols:
         *   providers:
         *     http_1_1:
         *       max-prologue-length: 8192
         *     http_2:
         *       max-frame-size: 4096
         *     websocket:
         *       ....
         * </pre>
         *
         * @return the protocols
         */
        public List<ProtocolConfig> protocols() {
            return protocols;
        }

        /**
         * Http routing. This will always be added to the resulting {@link io.helidon.webserver.Router}, if defined,
         *  overriding any HTTP routing already present.
         * If a custom listener has routing defined, it will be used, otherwise routing defined on web server will be used.
         *
         * @return the routing
         */
        public Optional<HttpRouting.Builder> routing() {
            return Optional.ofNullable(routing);
        }

        /**
         * List of all routings (possibly for multiple protocols). This allows adding non-http protocols as well,
         * as opposed to {@link #routing()}
         *
         * @return the routings
         */
        public List<io.helidon.common.Builder<?, ? extends Routing>> routings() {
            return routings;
        }

        /**
         * Name of this socket. Defaults to {@code @default}.
         * Must be defined if more than one socket is needed.
         *
         * @return the name
         */
        public String name() {
            return name;
        }

        /**
         * Host of the default socket. Defaults to all host addresses ({@code 0.0.0.0}).
         *
         * @return the host
         */
        public String host() {
            return host;
        }

        /**
         * Address to use. If both this and {@link #host()} is configured, this will be used.
         *
         * @return the address
         */
        public Optional<InetAddress> address() {
            return Optional.ofNullable(address);
        }

        /**
         * Port of the default socket.
         * If configured to {@code 0} (the default), server starts on a random port.
         *
         * @return the port
         */
        public int port() {
            return port;
        }

        /**
         * Accept backlog.
         *
         * @return the backlog
         */
        public int backlog() {
            return backlog;
        }

        /**
         * Maximal number of bytes an entity may have.
         * If {@link io.helidon.http.HeaderNames#CONTENT_LENGTH} is used, this is checked immediately,
         * if {@link io.helidon.http.HeaderValues#TRANSFER_ENCODING_CHUNKED} is used, we will fail when the
         * number of bytes read would exceed the max payload size.
         * Defaults to unlimited ({@code -1}).
         *
         * @return the max payload size
         */
        public long maxPayloadSize() {
            return maxPayloadSize;
        }

        /**
         * Listener receive buffer size.
         *
         * @return the receive buffer size
         */
        public Optional<Integer> receiveBufferSize() {
            return Optional.ofNullable(receiveBufferSize);
        }

        /**
         * Number of buffers queued for write operations.
         *
         * @return the write queue length
         */
        public int writeQueueLength() {
            return writeQueueLength;
        }

        /**
         * Initial buffer size in bytes of {@link java.io.BufferedOutputStream} created internally to
         * write data to a socket connection. Default is {@code 512}.
         *
         * @return the write buffer size
         */
        public int writeBufferSize() {
            return writeBufferSize;
        }

        /**
         * Grace period in ISO 8601 duration format to allow running tasks to complete before listener's shutdown.
         * Default is {@code 500} milliseconds.
         * <p>Configuration file values example: {@code PT0.5S}, {@code PT2S}.
         *
         * @return the shutdown grace period
         */
        public Duration shutdownGracePeriod() {
            return shutdownGracePeriod;
        }

        /**
         * Configuration of a connection (established from client against our server).
         *
         * @return the connection config
         */
        public Optional<ConnectionConfig> connectionConfig() {
            return Optional.ofNullable(connectionConfig);
        }

        /**
         * Listener TLS configuration.
         *
         * @return the tls
         */
        public Optional<Tls> tls() {
            return Optional.ofNullable(tls);
        }

        /**
         * Configure the listener specific {@link io.helidon.http.encoding.ContentEncodingContext}.
         * This method discards all previously registered ContentEncodingContext.
         * If no content encoding context is registered, content encoding context of the webserver would be used.
         *
         * @return the content encoding
         */
        public Optional<ContentEncodingContext> contentEncoding() {
            return Optional.ofNullable(contentEncoding);
        }

        /**
         * Configure the listener specific {@link io.helidon.http.media.MediaContext}.
         * This method discards all previously registered MediaContext.
         * If no media context is registered, media context of the webserver would be used.
         *
         * @return the media context
         */
        public Optional<MediaContext> mediaContext() {
            return Optional.ofNullable(mediaContext);
        }

        /**
         * Options for connections accepted by this listener.
         * This is not used to setup server connection.
         *
         * @return the connection options
         */
        public Optional<SocketOptions> connectionOptions() {
            return Optional.ofNullable(connectionOptions);
        }

        /**
         * Limits the number of connections that can be opened at a single point in time.
         * Defaults to {@code -1}, meaning "unlimited" - what the system allows.
         *
         * @return the max tcp connections
         */
        public int maxTcpConnections() {
            return maxTcpConnections;
        }

        /**
         * Limits the number of requests that can be executed at the same time (the number of active virtual threads of requests).
         * Defaults to {@code -1}, meaning "unlimited" - what the system allows.
         * Also make sure that this number is higher than the expected time it takes to handle a single request in your application,
         * as otherwise you may stop in-progress requests.
         *
         * @return the max concurrent requests
         */
        public int maxConcurrentRequests() {
            return maxConcurrentRequests;
        }

        /**
         * How long should we wait before closing a connection that has no traffic on it.
         * Defaults to {@code PT5M} (5 minutes). Note that the timestamp is refreshed max. once per second, so this setting
         * would be useless if configured for shorter periods of time (also not a very good support for connection keep alive,
         * if the connections are killed so soon anyway).
         *
         * @return the idle connection timeout
         */
        public Duration idleConnectionTimeout() {
            return idleConnectionTimeout;
        }

        /**
         * How often should we check for {@link #idleConnectionTimeout()}.
         * Defaults to {@code PT2M} (2 minutes).
         *
         * @return the idle connection period
         */
        public Duration idleConnectionPeriod() {
            return idleConnectionPeriod;
        }

        /**
         * If the entity is expected to be smaller that this number of bytes, it would be buffered in memory to optimize
         * performance when writing it.
         * If bigger, streaming will be used.
         * <p>
         * Note that for some entity types we cannot use streaming, as they are already fully in memory (String, byte[]), for such
         * cases, this option is ignored.
         * <p>
         * Default is 128Kb.
         *
         * @return the max in memory entity
         */
        public int maxInMemoryEntity() {
            return maxInMemoryEntity;
        }

        /**
         * Server listener socket options.
         * Unless configured through builder, {@code SO_REUSEADDR} is set to {@code true},
         * and {@code SO_RCVBUF} is set to {@code 4096}.
         *
         * @return the listener socket options
         */
        public Map<SocketOption<?>, Object> listenerSocketOptions() {
            return listenerSocketOptions;
        }

        /**
         * Explicitly defined connection selectors to be used with this socket.
         * This list is augmented with the result of {@link #protocols()}, but custom selectors are always used first.
         *
         * @return the connection selectors
         */
        public List<ServerConnectionSelector> connectionSelectors() {
            return connectionSelectors;
        }

        /**
         * Direct handlers specific for this listener.
         * A direct handler takes care of problems that happen before (or outside of) routing, such as bad request.
         *
         * @return the direct handlers
         */
        public Optional<DirectHandlers> directHandlers() {
            return Optional.ofNullable(directHandlers);
        }

        /**
         * Listener scoped context to be used as a parent for webserver request contexts (if used).
         * If an explicit context is used, you need to take care of correctly configuring its parent.
         * It is expected that the parent of this context is the WebServer context. You should also configure explicit
         * WebServer context when using this method
         *
         * @return the listener context
         * @see WebServerConfig#serverContext()
         * @see #listenerContext()
         */
        public Optional<Context> listenerContext() {
            return Optional.ofNullable(listenerContext);
        }

        /**
         * Enable proxy protocol support for this socket. This protocol is supported by
         * some load balancers/reverse proxies as a means to convey client information that
         * would otherwise be lost. If enabled, the proxy protocol header must be present
         * on every new connection established with your server. For more information,
         * see <a href="https://www.haproxy.org/download/1.8/doc/proxy-protocol.txt">
         * the specification</a>. Default is {@code false}.
         *
         * @return the enable proxy protocol
         */
        public boolean enableProxyProtocol() {
            return enableProxyProtocol;
        }

        /**
         * Requested URI discovery context.
         *
         * @return the requested uri discovery context
         */
        public Optional<RequestedUriDiscoveryContext> requestedUriDiscoveryContext() {
            return Optional.ofNullable(requestedUriDiscoveryContext);
        }

        /**
         * If this instance was configured, this would be the config instance used.
         *
         * @return config node used to configure this builder, or empty if not configured
         */
        public Optional<Config> config() {
            return Optional.ofNullable(config);
        }

        @Override
        public String toString() {
            return "ListenerConfigBuilder{"
                    + "protocols=" + protocols + ","
                    + "routing=" + routing + ","
                    + "routings=" + routings + ","
                    + "name=" + name + ","
                    + "host=" + host + ","
                    + "address=" + address + ","
                    + "port=" + port + ","
                    + "backlog=" + backlog + ","
                    + "maxPayloadSize=" + maxPayloadSize + ","
                    + "receiveBufferSize=" + receiveBufferSize + ","
                    + "writeQueueLength=" + writeQueueLength + ","
                    + "writeBufferSize=" + writeBufferSize + ","
                    + "shutdownGracePeriod=" + shutdownGracePeriod + ","
                    + "connectionConfig=" + connectionConfig + ","
                    + "tls=" + tls + ","
                    + "contentEncoding=" + contentEncoding + ","
                    + "mediaContext=" + mediaContext + ","
                    + "connectionOptions=" + connectionOptions + ","
                    + "maxTcpConnections=" + maxTcpConnections + ","
                    + "maxConcurrentRequests=" + maxConcurrentRequests + ","
                    + "idleConnectionTimeout=" + idleConnectionTimeout + ","
                    + "idleConnectionPeriod=" + idleConnectionPeriod + ","
                    + "maxInMemoryEntity=" + maxInMemoryEntity + ","
                    + "listenerSocketOptions=" + listenerSocketOptions + ","
                    + "connectionSelectors=" + connectionSelectors + ","
                    + "directHandlers=" + directHandlers + ","
                    + "listenerContext=" + listenerContext + ","
                    + "enableProxyProtocol=" + enableProxyProtocol + ","
                    + "requestedUriDiscoveryContext=" + requestedUriDiscoveryContext
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        @SuppressWarnings("unchecked")
        protected void preBuildPrototype() {
            var config = this.config == null ? Config.empty() : this.config;
            {
                var serviceLoader = HelidonServiceLoader.create(ServiceLoader.load(ProtocolConfigProvider.class));
                this.addProtocols(discoverServices(config, "protocols", serviceLoader, ProtocolConfigProvider.class, ProtocolConfig.class, protocolsDiscoverServices, protocols));
            }
            new WebServerConfigSupport.ListenerConfigDecorator().decorate(this);
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
            Errors.Collector collector = Errors.collector();
            if (address == null) {
                collector.fatal(getClass(), "Property \"address\" must not be null, but not set");
            }
            if (connectionOptions == null) {
                collector.fatal(getClass(), "Property \"connection-options\" must not be null, but not set");
            }
            collector.collect().checkValid();
        }

        /**
         * Http routing. This will always be added to the resulting {@link io.helidon.webserver.Router}, if defined,
         *  overriding any HTTP routing already present.
         * If a custom listener has routing defined, it will be used, otherwise routing defined on web server will be used.
         *
         * @param routing HTTP Routing for this listener/server
         * @return updated builder instance
         * @see #routing()
         */
        BUILDER routing(Optional<? extends HttpRouting.Builder> routing) {
            Objects.requireNonNull(routing);
            this.routing = routing.map(io.helidon.webserver.http.HttpRouting.Builder.class::cast).orElse(this.routing);
            return self();
        }

        /**
         * Listener receive buffer size.
         *
         * @param receiveBufferSize buffer size in bytes
         * @return updated builder instance
         * @see #receiveBufferSize()
         */
        BUILDER receiveBufferSize(Optional<Integer> receiveBufferSize) {
            Objects.requireNonNull(receiveBufferSize);
            this.receiveBufferSize = receiveBufferSize.map(java.lang.Integer.class::cast).orElse(this.receiveBufferSize);
            return self();
        }

        /**
         * Configuration of a connection (established from client against our server).
         *
         * @param connectionConfig connection configuration
         * @return updated builder instance
         * @see #connectionConfig()
         */
        BUILDER connectionConfig(Optional<? extends ConnectionConfig> connectionConfig) {
            Objects.requireNonNull(connectionConfig);
            this.connectionConfig = connectionConfig.map(ConnectionConfig.class::cast).orElse(this.connectionConfig);
            return self();
        }

        /**
         * Listener TLS configuration.
         *
         * @param tls tls of this configuration
         * @return updated builder instance
         * @see #tls()
         */
        BUILDER tls(Optional<? extends Tls> tls) {
            Objects.requireNonNull(tls);
            this.tls = tls.map(io.helidon.common.tls.Tls.class::cast).orElse(this.tls);
            return self();
        }

        /**
         * Configure the listener specific {@link io.helidon.http.encoding.ContentEncodingContext}.
         * This method discards all previously registered ContentEncodingContext.
         * If no content encoding context is registered, content encoding context of the webserver would be used.
         *
         * @param contentEncoding content encoding context
         * @return updated builder instance
         * @see #contentEncoding()
         */
        BUILDER contentEncoding(Optional<? extends ContentEncodingContext> contentEncoding) {
            Objects.requireNonNull(contentEncoding);
            this.contentEncoding = contentEncoding.map(io.helidon.http.encoding.ContentEncodingContext.class::cast).orElse(this.contentEncoding);
            return self();
        }

        /**
         * Configure the listener specific {@link io.helidon.http.media.MediaContext}.
         * This method discards all previously registered MediaContext.
         * If no media context is registered, media context of the webserver would be used.
         *
         * @param mediaContext media context
         * @return updated builder instance
         * @see #mediaContext()
         */
        BUILDER mediaContext(Optional<? extends MediaContext> mediaContext) {
            Objects.requireNonNull(mediaContext);
            this.mediaContext = mediaContext.map(io.helidon.http.media.MediaContext.class::cast).orElse(this.mediaContext);
            return self();
        }

        /**
         * Direct handlers specific for this listener.
         * A direct handler takes care of problems that happen before (or outside of) routing, such as bad request.
         *
         * @param directHandlers direct handlers
         * @return updated builder instance
         * @see #directHandlers()
         */
        BUILDER directHandlers(Optional<? extends DirectHandlers> directHandlers) {
            Objects.requireNonNull(directHandlers);
            this.directHandlers = directHandlers.map(io.helidon.webserver.http.DirectHandlers.class::cast).orElse(this.directHandlers);
            return self();
        }

        /**
         * Listener scoped context to be used as a parent for webserver request contexts (if used).
         * If an explicit context is used, you need to take care of correctly configuring its parent.
         * It is expected that the parent of this context is the WebServer context. You should also configure explicit
         * WebServer context when using this method
         *
         * @param listenerContext listener context
         * @return updated builder instance
         * @see #listenerContext()
         */
        BUILDER listenerContext(Optional<? extends Context> listenerContext) {
            Objects.requireNonNull(listenerContext);
            this.listenerContext = listenerContext.map(io.helidon.common.context.Context.class::cast).orElse(this.listenerContext);
            return self();
        }

        /**
         * Requested URI discovery context.
         *
         * @param requestedUriDiscoveryContext discovery context
         * @return updated builder instance
         * @see #requestedUriDiscoveryContext()
         */
        BUILDER requestedUriDiscoveryContext(Optional<? extends RequestedUriDiscoveryContext> requestedUriDiscoveryContext) {
            Objects.requireNonNull(requestedUriDiscoveryContext);
            this.requestedUriDiscoveryContext = requestedUriDiscoveryContext.map(io.helidon.http.RequestedUriDiscoveryContext.class::cast).orElse(this.requestedUriDiscoveryContext);
            return self();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class ListenerConfigImpl implements ListenerConfig {

            private final boolean enableProxyProtocol;
            private final Duration idleConnectionPeriod;
            private final Duration idleConnectionTimeout;
            private final Duration shutdownGracePeriod;
            private final InetAddress address;
            private final int backlog;
            private final int maxConcurrentRequests;
            private final int maxInMemoryEntity;
            private final int maxTcpConnections;
            private final int port;
            private final int writeBufferSize;
            private final int writeQueueLength;
            private final List<io.helidon.common.Builder<?, ? extends Routing>> routings;
            private final List<ProtocolConfig> protocols;
            private final List<ServerConnectionSelector> connectionSelectors;
            private final long maxPayloadSize;
            private final Map<SocketOption<?>, Object> listenerSocketOptions;
            private final Optional<ConnectionConfig> connectionConfig;
            private final Optional<Context> listenerContext;
            private final Optional<Tls> tls;
            private final Optional<RequestedUriDiscoveryContext> requestedUriDiscoveryContext;
            private final Optional<ContentEncodingContext> contentEncoding;
            private final Optional<MediaContext> mediaContext;
            private final Optional<DirectHandlers> directHandlers;
            private final Optional<HttpRouting.Builder> routing;
            private final Optional<Integer> receiveBufferSize;
            private final SocketOptions connectionOptions;
            private final String host;
            private final String name;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected ListenerConfigImpl(ListenerConfig.BuilderBase<?, ?> builder) {
                this.protocols = List.copyOf(builder.protocols());
                this.routing = builder.routing();
                this.routings = List.copyOf(builder.routings());
                this.name = builder.name();
                this.host = builder.host();
                this.address = builder.address().get();
                this.port = builder.port();
                this.backlog = builder.backlog();
                this.maxPayloadSize = builder.maxPayloadSize();
                this.receiveBufferSize = builder.receiveBufferSize();
                this.writeQueueLength = builder.writeQueueLength();
                this.writeBufferSize = builder.writeBufferSize();
                this.shutdownGracePeriod = builder.shutdownGracePeriod();
                this.connectionConfig = builder.connectionConfig();
                this.tls = builder.tls();
                this.contentEncoding = builder.contentEncoding();
                this.mediaContext = builder.mediaContext();
                this.connectionOptions = builder.connectionOptions().get();
                this.maxTcpConnections = builder.maxTcpConnections();
                this.maxConcurrentRequests = builder.maxConcurrentRequests();
                this.idleConnectionTimeout = builder.idleConnectionTimeout();
                this.idleConnectionPeriod = builder.idleConnectionPeriod();
                this.maxInMemoryEntity = builder.maxInMemoryEntity();
                this.listenerSocketOptions = Collections.unmodifiableMap(new LinkedHashMap<>(builder.listenerSocketOptions()));
                this.connectionSelectors = List.copyOf(builder.connectionSelectors());
                this.directHandlers = builder.directHandlers();
                this.listenerContext = builder.listenerContext();
                this.enableProxyProtocol = builder.enableProxyProtocol();
                this.requestedUriDiscoveryContext = builder.requestedUriDiscoveryContext();
            }

            @Override
            public List<ProtocolConfig> protocols() {
                return protocols;
            }

            @Override
            public Optional<HttpRouting.Builder> routing() {
                return routing;
            }

            @Override
            public List<io.helidon.common.Builder<?, ? extends Routing>> routings() {
                return routings;
            }

            @Override
            public String name() {
                return name;
            }

            @Override
            public String host() {
                return host;
            }

            @Override
            public InetAddress address() {
                return address;
            }

            @Override
            public int port() {
                return port;
            }

            @Override
            public int backlog() {
                return backlog;
            }

            @Override
            public long maxPayloadSize() {
                return maxPayloadSize;
            }

            @Override
            public Optional<Integer> receiveBufferSize() {
                return receiveBufferSize;
            }

            @Override
            public int writeQueueLength() {
                return writeQueueLength;
            }

            @Override
            public int writeBufferSize() {
                return writeBufferSize;
            }

            @Override
            public Duration shutdownGracePeriod() {
                return shutdownGracePeriod;
            }

            @Override
            public Optional<ConnectionConfig> connectionConfig() {
                return connectionConfig;
            }

            @Override
            public Optional<Tls> tls() {
                return tls;
            }

            @Override
            public Optional<ContentEncodingContext> contentEncoding() {
                return contentEncoding;
            }

            @Override
            public Optional<MediaContext> mediaContext() {
                return mediaContext;
            }

            @Override
            public SocketOptions connectionOptions() {
                return connectionOptions;
            }

            @Override
            public int maxTcpConnections() {
                return maxTcpConnections;
            }

            @Override
            public int maxConcurrentRequests() {
                return maxConcurrentRequests;
            }

            @Override
            public Duration idleConnectionTimeout() {
                return idleConnectionTimeout;
            }

            @Override
            public Duration idleConnectionPeriod() {
                return idleConnectionPeriod;
            }

            @Override
            public int maxInMemoryEntity() {
                return maxInMemoryEntity;
            }

            @Override
            public Map<SocketOption<?>, Object> listenerSocketOptions() {
                return listenerSocketOptions;
            }

            @Override
            public List<ServerConnectionSelector> connectionSelectors() {
                return connectionSelectors;
            }

            @Override
            public Optional<DirectHandlers> directHandlers() {
                return directHandlers;
            }

            @Override
            public Optional<Context> listenerContext() {
                return listenerContext;
            }

            @Override
            public boolean enableProxyProtocol() {
                return enableProxyProtocol;
            }

            @Override
            public Optional<RequestedUriDiscoveryContext> requestedUriDiscoveryContext() {
                return requestedUriDiscoveryContext;
            }

            @Override
            public String toString() {
                return "ListenerConfig{"
                        + "protocols=" + protocols + ","
                        + "routing=" + routing + ","
                        + "routings=" + routings + ","
                        + "name=" + name + ","
                        + "host=" + host + ","
                        + "address=" + address + ","
                        + "port=" + port + ","
                        + "backlog=" + backlog + ","
                        + "maxPayloadSize=" + maxPayloadSize + ","
                        + "receiveBufferSize=" + receiveBufferSize + ","
                        + "writeQueueLength=" + writeQueueLength + ","
                        + "writeBufferSize=" + writeBufferSize + ","
                        + "shutdownGracePeriod=" + shutdownGracePeriod + ","
                        + "connectionConfig=" + connectionConfig + ","
                        + "tls=" + tls + ","
                        + "contentEncoding=" + contentEncoding + ","
                        + "mediaContext=" + mediaContext + ","
                        + "connectionOptions=" + connectionOptions + ","
                        + "maxTcpConnections=" + maxTcpConnections + ","
                        + "maxConcurrentRequests=" + maxConcurrentRequests + ","
                        + "idleConnectionTimeout=" + idleConnectionTimeout + ","
                        + "idleConnectionPeriod=" + idleConnectionPeriod + ","
                        + "maxInMemoryEntity=" + maxInMemoryEntity + ","
                        + "listenerSocketOptions=" + listenerSocketOptions + ","
                        + "connectionSelectors=" + connectionSelectors + ","
                        + "directHandlers=" + directHandlers + ","
                        + "listenerContext=" + listenerContext + ","
                        + "enableProxyProtocol=" + enableProxyProtocol + ","
                        + "requestedUriDiscoveryContext=" + requestedUriDiscoveryContext
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof ListenerConfig other)) {
                    return false;
                }
                return Objects.equals(protocols, other.protocols())
                    && Objects.equals(routing, other.routing())
                    && Objects.equals(routings, other.routings())
                    && Objects.equals(name, other.name())
                    && Objects.equals(host, other.host())
                    && Objects.equals(address, other.address())
                    && port == other.port()
                    && backlog == other.backlog()
                    && maxPayloadSize == other.maxPayloadSize()
                    && Objects.equals(receiveBufferSize, other.receiveBufferSize())
                    && writeQueueLength == other.writeQueueLength()
                    && writeBufferSize == other.writeBufferSize()
                    && Objects.equals(shutdownGracePeriod, other.shutdownGracePeriod())
                    && Objects.equals(connectionConfig, other.connectionConfig())
                    && Objects.equals(tls, other.tls())
                    && Objects.equals(contentEncoding, other.contentEncoding())
                    && Objects.equals(mediaContext, other.mediaContext())
                    && Objects.equals(connectionOptions, other.connectionOptions())
                    && maxTcpConnections == other.maxTcpConnections()
                    && maxConcurrentRequests == other.maxConcurrentRequests()
                    && Objects.equals(idleConnectionTimeout, other.idleConnectionTimeout())
                    && Objects.equals(idleConnectionPeriod, other.idleConnectionPeriod())
                    && maxInMemoryEntity == other.maxInMemoryEntity()
                    && Objects.equals(listenerSocketOptions, other.listenerSocketOptions())
                    && Objects.equals(connectionSelectors, other.connectionSelectors())
                    && Objects.equals(directHandlers, other.directHandlers())
                    && Objects.equals(listenerContext, other.listenerContext())
                    && enableProxyProtocol == other.enableProxyProtocol()
                    && Objects.equals(requestedUriDiscoveryContext, other.requestedUriDiscoveryContext());
            }

            @Override
            public int hashCode() {
                return Objects.hash(protocols, routing, routings, name, host, address, port, backlog, maxPayloadSize, receiveBufferSize, writeQueueLength, writeBufferSize, shutdownGracePeriod, connectionConfig, tls, contentEncoding, mediaContext, connectionOptions, maxTcpConnections, maxConcurrentRequests, idleConnectionTimeout, idleConnectionPeriod, maxInMemoryEntity, listenerSocketOptions, connectionSelectors, directHandlers, listenerContext, enableProxyProtocol, requestedUriDiscoveryContext);
            }

        }

    }

    /**
     * Fluent API builder for {@link ListenerConfig}.
     */
    class Builder extends ListenerConfig.BuilderBase<ListenerConfig.Builder, ListenerConfig> implements io.helidon.common.Builder<ListenerConfig.Builder, ListenerConfig> {

        private Builder() {
        }

        @Override
        public ListenerConfig buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new ListenerConfigImpl(this);
        }

        @Override
        public ListenerConfig build() {
            return buildPrototype();
        }

    }

}
