/*
 * Copyright (c) 2024 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.webserver;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.ServiceLoader;
import java.util.function.Consumer;
import java.util.function.Supplier;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Generated;
import io.helidon.common.HelidonServiceLoader;
import io.helidon.common.config.Config;
import io.helidon.common.context.Context;
import io.helidon.webserver.http.HttpRouting;
import io.helidon.webserver.spi.ServerFeature;
import io.helidon.webserver.spi.ServerFeatureProvider;

/**
 * WebServer configuration bean.
 * See {@link WebServer#create(java.util.function.Consumer)}.
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.codegen.BuilderCodegen", trigger = "io.helidon.webserver.WebServerConfigBlueprint")
public interface WebServerConfig extends WebServerConfigBlueprint, Prototype.Api, ListenerConfig {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static WebServerConfig.Builder builder() {
        return new WebServerConfig.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static WebServerConfig.Builder builder(WebServerConfig instance) {
        return WebServerConfig.builder().from(instance);
    }

    /**
     * Create a new instance from configuration.
     *
     * @param config used to configure the new instance
     * @return a new instance configured from configuration
     */
    static WebServerConfig create(Config config) {
        return WebServerConfig.builder().config(config).buildPrototype();
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static WebServerConfig create() {
        return WebServerConfig.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link WebServer}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends WebServerConfig.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends WebServerConfig> extends ListenerConfig.BuilderBase<BUILDER, PROTOTYPE> implements Prototype.ConfiguredBuilder<BUILDER, PROTOTYPE> {

        private final List<ServerFeature> features = new ArrayList<>();
        private final Map<String, ListenerConfig> sockets = new LinkedHashMap<>();
        private final Map<String, List<io.helidon.common.Builder<?, ? extends Routing>>> namedRoutings = new LinkedHashMap<>();
        private boolean featuresDiscoverServices = true;
        private boolean isFeaturesMutated;
        private boolean shutdownHook = true;
        private Config config;
        private Context serverContext;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(WebServerConfig prototype) {
            super.from(prototype);
            shutdownHook(prototype.shutdownHook());
            addSockets(prototype.sockets());
            addNamedRoutings(prototype.namedRoutings());
            if (!isFeaturesMutated) {
                features.clear();
            }
            addFeatures(prototype.features());
            serverContext(prototype.serverContext());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(WebServerConfig.BuilderBase<?, ?> builder) {
            super.from(builder);
            shutdownHook(builder.shutdownHook());
            addSockets(builder.sockets);
            addNamedRoutings(builder.namedRoutings);
            if (isFeaturesMutated) {
                if (builder.isFeaturesMutated) {
                    addFeatures(builder.features);
                }
            } else {
                features.clear();
                addFeatures(builder.features);
            }
            builder.serverContext().ifPresent(this::serverContext);
            return self();
        }

        /**
         * Add Http routing for an additional socket.
         *
         * @param socket   name of the socket
         * @param consumer HTTP Routing for the given socket name
         * @return updated builder instance
         */
        public BUILDER routing(String socket, Consumer<HttpRouting.Builder> consumer) {
            WebServerConfigSupport.CustomMethods.routing(this, socket, consumer);
            return self();
        }

        /**
         * Add Http routing for an additional socket.
         *
         * @param socket  name of the socket
         * @param routing HTTP Routing for the given socket name
         * @return updated builder instance
         */
        public BUILDER routing(String socket, HttpRouting.Builder routing) {
            WebServerConfigSupport.CustomMethods.routing(this, socket, routing);
            return self();
        }

        /**
         * Update builder from configuration (node of this type).
         * If a value is present in configuration, it would override currently configured values.
         *
         * @param config configuration instance used to obtain values to update this builder
         * @return updated builder instance
         */
        @Override
        public BUILDER config(Config config) {
            Objects.requireNonNull(config);
            this.config = config;
            super.config(config);
            config.get("shutdown-hook").as(Boolean.class).ifPresent(this::shutdownHook);
            config.get("sockets").asNodeList().ifPresent(nodes -> nodes.forEach(node -> sockets.put(node.get("name").asString().orElse(node.name()), node.map(ListenerConfig::create).get())));
            return self();
        }

        /**
         * When true the webserver registers a shutdown hook with the JVM Runtime.
         * <p>
         * Defaults to true. Set this to false such that a shutdown hook is not registered.
         *
         * @param shutdownHook whether to register a shutdown hook
         * @return updated builder instance
         * @see #shutdownHook()
         */
        public BUILDER shutdownHook(boolean shutdownHook) {
            this.shutdownHook = shutdownHook;
            return self();
        }

        /**
         * This method replaces all values with the new ones.
         *
         * @param sockets map of listener configurations, except for the default one
         * @return updated builder instance
         * @see #sockets()
         */
        public BUILDER sockets(Map<? extends String, ? extends ListenerConfig> sockets) {
            Objects.requireNonNull(sockets);
            this.sockets.clear();
            this.sockets.putAll(sockets);
            return self();
        }

        /**
         * This method keeps existing values, then puts all new values into the map.
         *
         * @param sockets map of listener configurations, except for the default one
         * @return updated builder instance
         * @see #sockets()
         */
        public BUILDER addSockets(Map<? extends String, ? extends ListenerConfig> sockets) {
            Objects.requireNonNull(sockets);
            this.sockets.putAll(sockets);
            return self();
        }

        /**
         * This method adds a new value to the map, or replaces it if the key already exists.
         *
         * @param key key to add or replace
         * @param socket new value for the key
         * @return updated builder instance
         * @see #sockets()
         */
        public BUILDER putSocket(String key, ListenerConfig socket) {
            Objects.requireNonNull(key);
            Objects.requireNonNull(socket);
            this.sockets.put(key, socket);
            return self();
        }

        /**
         * This method adds a new value to the map, or replaces it if the key already exists.
         *
         * @param key key to add or replace
         * @param consumer builder consumer to create new value for the key
         * @return updated builder instance
         * @see #sockets()
         */
        public BUILDER putSocket(String key, Consumer<ListenerConfig.Builder> consumer) {
            Objects.requireNonNull(key);
            Objects.requireNonNull(consumer);
            var builder = ListenerConfig.builder();
            consumer.accept(builder);
            this.putSocket(key, builder.build());
            return self();
        }

        /**
         * Server features allow customization of the server, listeners, or routings.
         *
         * @param discoverServices whether to discover implementations through service loader
         * @return updated builder instance
         * @see #features()
         */
        public BUILDER featuresDiscoverServices(boolean discoverServices) {
            this.featuresDiscoverServices = discoverServices;
            return self();
        }

        /**
         * Server features allow customization of the server, listeners, or routings.
         *
         * @param features server features
         * @return updated builder instance
         * @see #features()
         */
        public BUILDER features(List<? extends ServerFeature> features) {
            Objects.requireNonNull(features);
            isFeaturesMutated = true;
            this.features.clear();
            this.features.addAll(features);
            return self();
        }

        /**
         * Server features allow customization of the server, listeners, or routings.
         *
         * @param features server features
         * @return updated builder instance
         * @see #features()
         */
        public BUILDER addFeatures(List<? extends ServerFeature> features) {
            Objects.requireNonNull(features);
            isFeaturesMutated = true;
            this.features.addAll(features);
            return self();
        }

        /**
         * Server features allow customization of the server, listeners, or routings.
         *
         * @param feature server features
         * @return updated builder instance
         * @see #features()
         */
        public BUILDER addFeature(ServerFeature feature) {
            Objects.requireNonNull(feature);
            this.features.add(feature);
            isFeaturesMutated = true;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #serverContext()
         */
        public BUILDER clearServerContext() {
            this.serverContext = null;
            return self();
        }

        /**
         * Context for the WebServer, if none defined, a new one will be created with global context as the root.
         *
         * @param serverContext server context
         * @return updated builder instance
         * @see #serverContext()
         */
        public BUILDER serverContext(Context serverContext) {
            Objects.requireNonNull(serverContext);
            this.serverContext = serverContext;
            return self();
        }

        /**
         * Context for the WebServer, if none defined, a new one will be created with global context as the root.
         *
         * @param consumer server context
         * @return updated builder instance
         * @see #serverContext()
         */
        public BUILDER serverContext(Consumer<Context.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = Context.builder();
            consumer.accept(builder);
            this.serverContext(builder.build());
            return self();
        }

        /**
         * When true the webserver registers a shutdown hook with the JVM Runtime.
         * <p>
         * Defaults to true. Set this to false such that a shutdown hook is not registered.
         *
         * @return the shutdown hook
         */
        public boolean shutdownHook() {
            return shutdownHook;
        }

        /**
         * Socket configurations.
         * Note that socket named {@value WebServer#DEFAULT_SOCKET_NAME} cannot be used,
         * configure the values on the server directly.
         *
         * @return the sockets
         */
        public Map<String, ListenerConfig> sockets() {
            return sockets;
        }

        /**
         * Routing for additional sockets.
         * Note that socket named {@value WebServer#DEFAULT_SOCKET_NAME} cannot be used,
         * configure the routing on the server directly.
         *
         * @return the named routings
         */
        public Map<String, List<io.helidon.common.Builder<?, ? extends Routing>>> namedRoutings() {
            return namedRoutings;
        }

        /**
         * Server features allow customization of the server, listeners, or routings.
         *
         * @return the features
         */
        public List<ServerFeature> features() {
            return features;
        }

        /**
         * Context for the WebServer, if none defined, a new one will be created with global context as the root.
         *
         * @return the server context
         */
        public Optional<Context> serverContext() {
            return Optional.ofNullable(serverContext);
        }

        /**
         * If this instance was configured, this would be the config instance used.
         *
         * @return config node used to configure this builder, or empty if not configured
         */
        public Optional<Config> config() {
            return Optional.ofNullable(config);
        }

        @Override
        public String toString() {
            return "WebServerConfigBuilder{"
                    + "shutdownHook=" + shutdownHook + ","
                    + "sockets=" + sockets + ","
                    + "namedRoutings=" + namedRoutings + ","
                    + "features=" + features + ","
                    + "serverContext=" + serverContext
                    + "};"
                    + super.toString();
        }

        /**
         * Handles providers and decorators.
         */
        @SuppressWarnings("unchecked")
        protected void preBuildPrototype() {
            super.preBuildPrototype();
            var config = this.config == null ? Config.empty() : this.config;
            {
                var serviceLoader = HelidonServiceLoader.create(ServiceLoader.load(ServerFeatureProvider.class));
                this.addFeatures(discoverServices(config, "features", serviceLoader, ServerFeatureProvider.class, ServerFeature.class, featuresDiscoverServices, features));
            }
            new WebServerConfigSupport.ServerConfigDecorator().decorate(this);
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
            super.validatePrototype();
        }

        /**
         * This method replaces all values with the new ones.
         *
         * @param namedRoutings map of routing
         * @return updated builder instance
         * @see #namedRoutings()
         */
        BUILDER namedRoutings(Map<? extends String, List<io.helidon.common.Builder<?, ? extends Routing>>> namedRoutings) {
            Objects.requireNonNull(namedRoutings);
            this.namedRoutings.clear();
            this.namedRoutings.putAll(namedRoutings);
            return self();
        }

        /**
         * This method keeps existing values, then puts all new values into the map.
         *
         * @param namedRoutings map of routing
         * @return updated builder instance
         * @see #namedRoutings()
         */
        BUILDER addNamedRoutings(Map<? extends String, List<io.helidon.common.Builder<?, ? extends Routing>>> namedRoutings) {
            Objects.requireNonNull(namedRoutings);
            this.namedRoutings.putAll(namedRoutings);
            return self();
        }

        /**
         * This method adds a new value to the map value, or creates a new value.
         *
         * @param key key to add to
         * @param namedRouting additional value for the key
         * @return updated builder instance
         * @see #namedRoutings()
         */
        BUILDER addNamedRouting(String key, io.helidon.common.Builder<?, ? extends Routing> namedRouting) {
            Objects.requireNonNull(key);
            Objects.requireNonNull(namedRouting);
            this.namedRoutings.compute(key, (k, v) -> {
                v = v == null ? new ArrayList<>() : new ArrayList<>(v);
                v.add(namedRouting);
                return v;
            });
            return self();
        }

        /**
         * This method adds a new value to the map value, or creates a new value.
         *
         * @param key key to add to
         * @param namedRoutings additional values for the key
         * @return updated builder instance
         * @see #namedRoutings()
         */
        BUILDER addNamedRoutings(String key, List<io.helidon.common.Builder<?, ? extends Routing>> namedRoutings) {
            Objects.requireNonNull(key);
            Objects.requireNonNull(namedRoutings);
            this.namedRoutings.compute(key, (k, v) -> {
                v = v == null ? new ArrayList<>() : new ArrayList<>(v);
                v.addAll(namedRoutings);
                return v;
            });
            return self();
        }

        /**
         * This method adds a new value to the map, or replaces it if the key already exists.
         *
         * @param key key to add or replace
         * @param namedRouting new value for the key
         * @return updated builder instance
         * @see #namedRoutings()
         */
        BUILDER putNamedRouting(String key, List<io.helidon.common.Builder<?, ? extends Routing>> namedRouting) {
            Objects.requireNonNull(key);
            Objects.requireNonNull(namedRouting);
            this.namedRoutings.put(key, List.copyOf(namedRouting));
            return self();
        }

        /**
         * Context for the WebServer, if none defined, a new one will be created with global context as the root.
         *
         * @param serverContext server context
         * @return updated builder instance
         * @see #serverContext()
         */
        BUILDER serverContext(Optional<? extends Context> serverContext) {
            Objects.requireNonNull(serverContext);
            this.serverContext = serverContext.map(io.helidon.common.context.Context.class::cast).orElse(this.serverContext);
            return self();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class WebServerConfigImpl extends ListenerConfigImpl implements WebServerConfig, Supplier<WebServer> {

            private final boolean shutdownHook;
            private final List<ServerFeature> features;
            private final Map<String, ListenerConfig> sockets;
            private final Map<String, List<io.helidon.common.Builder<?, ? extends Routing>>> namedRoutings;
            private final Optional<Context> serverContext;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected WebServerConfigImpl(WebServerConfig.BuilderBase<?, ?> builder) {
                super(builder);
                this.shutdownHook = builder.shutdownHook();
                this.sockets = Collections.unmodifiableMap(new LinkedHashMap<>(builder.sockets()));
                this.namedRoutings = Collections.unmodifiableMap(new LinkedHashMap<>(builder.namedRoutings()));
                this.features = List.copyOf(builder.features());
                this.serverContext = builder.serverContext();
            }

            @Override
            public WebServer build() {
                return WebServer.create(this);
            }

            @Override
            public WebServer get() {
                return build();
            }

            @Override
            public boolean shutdownHook() {
                return shutdownHook;
            }

            @Override
            public Map<String, ListenerConfig> sockets() {
                return sockets;
            }

            @Override
            public Map<String, List<io.helidon.common.Builder<?, ? extends Routing>>> namedRoutings() {
                return namedRoutings;
            }

            @Override
            public List<ServerFeature> features() {
                return features;
            }

            @Override
            public Optional<Context> serverContext() {
                return serverContext;
            }

            @Override
            public String toString() {
                return "WebServerConfig{"
                        + "shutdownHook=" + shutdownHook + ","
                        + "sockets=" + sockets + ","
                        + "namedRoutings=" + namedRoutings + ","
                        + "features=" + features + ","
                        + "serverContext=" + serverContext
                        + "};"
                        + super.toString();
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof WebServerConfig other)) {
                    return false;
                }
                return super.equals(other)
                            && shutdownHook == other.shutdownHook()
                    && Objects.equals(sockets, other.sockets())
                    && Objects.equals(namedRoutings, other.namedRoutings())
                    && Objects.equals(features, other.features())
                    && Objects.equals(serverContext, other.serverContext());
            }

            @Override
            public int hashCode() {
                return 31 * super.hashCode() + Objects.hash(shutdownHook, sockets, namedRoutings, features, serverContext);
            }

        }

    }

    /**
     * Fluent API builder for {@link WebServer}.
     */
    class Builder extends WebServerConfig.BuilderBase<WebServerConfig.Builder, WebServerConfig> implements io.helidon.common.Builder<WebServerConfig.Builder, WebServer> {

        private Builder() {
        }

        @Override
        public WebServerConfig buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new WebServerConfigImpl(this);
        }

        @Override
        public WebServer build() {
            return WebServer.create(this.buildPrototype());
        }

    }

}
