/*
 * Copyright (c) 2023 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.webserver;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Supplier;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Generated;
import io.helidon.common.config.Config;
import io.helidon.common.context.Context;

/**
 * WebServer configuration bean.
 * See {@link WebServer#create(java.util.function.Consumer)}.
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.processor.BlueprintProcessor", trigger = "io.helidon.webserver.WebServerConfigBlueprint")
public interface WebServerConfig extends WebServerConfigBlueprint, Prototype.Api, ListenerConfig {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static WebServerConfig.Builder builder() {
        return new WebServerConfig.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static WebServerConfig.Builder builder(WebServerConfig instance) {
        return WebServerConfig.builder().from(instance);
    }

    /**
     * Create a new instance from configuration.
     *
     * @param config used to configure the new instance
     * @return a new instance configured from configuration
     */
    static WebServerConfig create(Config config) {
        return WebServerConfig.builder().config(config).buildPrototype();
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static WebServerConfig create() {
        return WebServerConfig.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link WebServer}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends WebServerConfig.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends WebServerConfig> extends ListenerConfig.BuilderBase<BUILDER, PROTOTYPE> implements Prototype.ConfiguredBuilder<BUILDER, PROTOTYPE> {

        private final Map<String, ListenerConfig> sockets = new LinkedHashMap<>();
        private boolean shutdownHook = true;
        private Config config;
        private Context serverContext;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(WebServerConfig prototype) {
            super.from(prototype);
            shutdownHook(prototype.shutdownHook());
            addSockets(prototype.sockets());
            serverContext(prototype.serverContext());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(WebServerConfig.BuilderBase<?, ?> builder) {
            super.from(builder);
            shutdownHook(builder.shutdownHook());
            addSockets(builder.sockets());
            builder.serverContext().ifPresent(this::serverContext);
            return self();
        }

        /**
         * Update builder from configuration (node of this type).
         * If a value is present in configuration, it would override currently configured values.
         *
         * @param config configuration instance used to obtain values to update this builder
         * @return updated builder instance
         */
        @Override
        public BUILDER config(Config config) {
            Objects.requireNonNull(config);
            this.config = config;
            super.config(config);
            config.get("shutdown-hook").as(Boolean.class).ifPresent(this::shutdownHook);
            config.get("sockets").asNodeList().ifPresent(nodes -> nodes.forEach(node -> sockets.put(node.get("name").asString().orElse(node.name()), node.map(ListenerConfig::create).get())));
            return self();
        }

        /**
         * When true the webserver registers a shutdown hook with the JVM Runtime.
         * <p>
         * Defaults to true. Set this to false such that a shutdown hook is not registered.
         *
         * @param shutdownHook whether to register a shutdown hook
         * @return updated builder instance
         * @see #shutdownHook()
         */
        public BUILDER shutdownHook(boolean shutdownHook) {
            this.shutdownHook = shutdownHook;
            return self();
        }

        /**
         * This method replaces all values with the new ones.
         *
         * @param sockets map of listener configurations, except for the default one
         * @return updated builder instance
         * @see #sockets()
         */
        public BUILDER sockets(Map<? extends String, ? extends ListenerConfig> sockets) {
            Objects.requireNonNull(sockets);
            this.sockets.clear();
            this.sockets.putAll(sockets);
            return self();
        }

        /**
         * This method keeps existing values, then puts all new values into the map.
         *
         * @param sockets map of listener configurations, except for the default one
         * @return updated builder instance
         * @see #sockets()
         */
        public BUILDER addSockets(Map<? extends String, ? extends ListenerConfig> sockets) {
            Objects.requireNonNull(sockets);
            this.sockets.putAll(sockets);
            return self();
        }

        /**
         * This method adds a new value to the map, or replaces it if the key already exists.
         *
         * @param key key to add or replace
         * @param socket new value for the key
         * @return updated builder instance
         * @see #sockets()
         */
        public BUILDER putSocket(String key, ListenerConfig socket) {
            Objects.requireNonNull(key);
            Objects.requireNonNull(socket);
            this.sockets.put(key, socket);
            return self();
        }

        /**
         * This method adds a new value to the map, or replaces it if the key already exists.
         *
         * @param key key to add or replace
         * @param consumer builder consumer to create new value for the key
         * @return updated builder instance
         * @see #sockets()
         */
        public BUILDER putSocket(String key, Consumer<ListenerConfig.Builder> consumer) {
            Objects.requireNonNull(key);
            Objects.requireNonNull(consumer);
            var builder = ListenerConfig.builder();
            consumer.accept(builder);
            this.putSocket(key, builder.build());
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #serverContext()
         */
        public BUILDER clearServerContext() {
            this.serverContext = null;
            return self();
        }

        /**
         * Context for the WebServer, if none defined, a new one will be created with global context as the root.
         *
         * @param serverContext server context
         * @return updated builder instance
         * @see #serverContext()
         */
        public BUILDER serverContext(Context serverContext) {
            Objects.requireNonNull(serverContext);
            this.serverContext = serverContext;
            return self();
        }

        /**
         * Context for the WebServer, if none defined, a new one will be created with global context as the root.
         *
         * @param consumer server context
         * @return updated builder instance
         * @see #serverContext()
         */
        public BUILDER serverContext(Consumer<Context.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = Context.builder();
            consumer.accept(builder);
            this.serverContext(builder.build());
            return self();
        }

        /**
         * When true the webserver registers a shutdown hook with the JVM Runtime.
         * <p>
         * Defaults to true. Set this to false such that a shutdown hook is not registered.
         *
         * @return the shutdown hook
         */
        public boolean shutdownHook() {
            return shutdownHook;
        }

        /**
         * Socket configurations.
         * Note that socket named {@value WebServer#DEFAULT_SOCKET_NAME} cannot be used,
         * configure the values on the server directly.
         *
         * @return the sockets
         */
        public Map<String, ListenerConfig> sockets() {
            return sockets;
        }

        /**
         * Context for the WebServer, if none defined, a new one will be created with global context as the root.
         *
         * @return the server context
         */
        public Optional<Context> serverContext() {
            return Optional.ofNullable(serverContext);
        }

        /**
         * If this instance was configured, this would be the config instance used.
         *
         * @return config node used to configure this builder, or empty if not configured
         */
        public Optional<Config> config() {
            return Optional.ofNullable(config);
        }

        @Override
        public String toString() {
            return "WebServerConfigBuilder{"
                    + "shutdownHook=" + shutdownHook + ","
                    + "sockets=" + sockets + ","
                    + "serverContext=" + serverContext
                    + "};"
                    + super.toString();
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
            super.preBuildPrototype();
            new WebServerConfigBlueprint.ServerConfigDecorator().decorate(this);
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
            super.validatePrototype();
        }

        /**
         * Context for the WebServer, if none defined, a new one will be created with global context as the root.
         *
         * @param serverContext server context
         * @return updated builder instance
         * @see #serverContext()
         */
        BUILDER serverContext(Optional<? extends Context> serverContext) {
            Objects.requireNonNull(serverContext);
            this.serverContext = serverContext.orElse(null);
            return self();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class WebServerConfigImpl extends ListenerConfigImpl implements WebServerConfig, Supplier<WebServer> {

            private final boolean shutdownHook;
            private final Map<String, ListenerConfig> sockets;
            private final Optional<Context> serverContext;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected WebServerConfigImpl(WebServerConfig.BuilderBase<?, ?> builder) {
                super(builder);
                this.shutdownHook = builder.shutdownHook();
                this.sockets = Collections.unmodifiableMap(new LinkedHashMap<>(builder.sockets()));
                this.serverContext = builder.serverContext();
            }

            @Override
            public WebServer build() {
                return WebServer.create(this);
            }

            @Override
            public WebServer get() {
                return build();
            }

            @Override
            public boolean shutdownHook() {
                return shutdownHook;
            }

            @Override
            public Map<String, ListenerConfig> sockets() {
                return sockets;
            }

            @Override
            public Optional<Context> serverContext() {
                return serverContext;
            }

            @Override
            public String toString() {
                return "WebServerConfig{"
                        + "shutdownHook=" + shutdownHook + ","
                        + "sockets=" + sockets + ","
                        + "serverContext=" + serverContext
                        + "};"
                        + super.toString();
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof WebServerConfig other)) {
                    return false;
                }
                return super.equals(other)
                        && shutdownHook == other.shutdownHook()
                        && Objects.equals(sockets, other.sockets())
                        && Objects.equals(serverContext, other.serverContext());
            }

            @Override
            public int hashCode() {
                return 31 * super.hashCode() + Objects.hash(shutdownHook, sockets, serverContext);
            }

        }

    }

    /**
     * Fluent API builder for {@link WebServer}.
     */
    class Builder extends WebServerConfig.BuilderBase<WebServerConfig.Builder, WebServerConfig> implements io.helidon.common.Builder<WebServerConfig.Builder, WebServer> {

        private Builder() {
        }

        @Override
        public WebServerConfig buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new WebServerConfigImpl(this);
        }

        @Override
        public WebServer build() {
            return WebServer.create(this.buildPrototype());
        }

    }

}
