/*
 * Copyright (c) 2023 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.webserver;

import java.net.SocketOption;
import java.time.Duration;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Generated;

/**
 * Configuration of a server connection (for each connection created by clients).
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.processor.BlueprintProcessor", trigger = "io.helidon.webserver.ConnectionConfigBlueprint")
public interface ConnectionConfig extends ConnectionConfigBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static ConnectionConfig.Builder builder() {
        return new ConnectionConfig.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static ConnectionConfig.Builder builder(ConnectionConfig instance) {
        return ConnectionConfig.builder().from(instance);
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static ConnectionConfig create() {
        return ConnectionConfig.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link ConnectionConfig}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends ConnectionConfig.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends ConnectionConfig> implements Prototype.Builder<BUILDER, PROTOTYPE> {

        private final Map<SocketOption<?>, Object> socketOptions = new LinkedHashMap<>();
        private boolean keepAlive = true;
        private boolean reuseAddress = true;
        private boolean tcpNoDelay = false;
        private Duration connectTimeout = Duration.parse("PT10S");
        private Duration readTimeout = Duration.parse("PT30S");
        private int receiveBufferSize = 32768;
        private int sendBufferSize = 32768;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(ConnectionConfig prototype) {
            readTimeout(prototype.readTimeout());
            connectTimeout(prototype.connectTimeout());
            sendBufferSize(prototype.sendBufferSize());
            receiveBufferSize(prototype.receiveBufferSize());
            keepAlive(prototype.keepAlive());
            reuseAddress(prototype.reuseAddress());
            tcpNoDelay(prototype.tcpNoDelay());
            addSocketOptions(prototype.socketOptions());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(ConnectionConfig.BuilderBase<?, ?> builder) {
            readTimeout(builder.readTimeout());
            connectTimeout(builder.connectTimeout());
            sendBufferSize(builder.sendBufferSize());
            receiveBufferSize(builder.receiveBufferSize());
            keepAlive(builder.keepAlive());
            reuseAddress(builder.reuseAddress());
            tcpNoDelay(builder.tcpNoDelay());
            addSocketOptions(builder.socketOptions());
            return self();
        }

        /**
         * Read timeout.
         * Default is {@value #DEFAULT_READ_TIMEOUT_DURATION}
         *
         * @param readTimeout read timeout
         * @return updated builder instance
         * @see #readTimeout()
         */
        public BUILDER readTimeout(Duration readTimeout) {
            Objects.requireNonNull(readTimeout);
            this.readTimeout = readTimeout;
            return self();
        }

        /**
         * Connect timeout.
         * Default is {@value #DEFAULT_CONNECT_TIMEOUT_DURATION}.
         *
         * @param connectTimeout connect timeout
         * @return updated builder instance
         * @see #connectTimeout()
         */
        public BUILDER connectTimeout(Duration connectTimeout) {
            Objects.requireNonNull(connectTimeout);
            this.connectTimeout = connectTimeout;
            return self();
        }

        /**
         * Socket send buffer size.
         * Default is {@value #DEFAULT_SO_BUFFER_SIZE}.
         *
         * @param sendBufferSize buffer size, in bytes
         * @return updated builder instance
         * @see #sendBufferSize()
         */
        public BUILDER sendBufferSize(int sendBufferSize) {
            this.sendBufferSize = sendBufferSize;
            return self();
        }

        /**
         * Socket receive buffer size.
         * Default is {@value #DEFAULT_SO_BUFFER_SIZE}.
         *
         * @param receiveBufferSize buffer size, in bytes
         * @return updated builder instance
         * @see #receiveBufferSize()
         */
        public BUILDER receiveBufferSize(int receiveBufferSize) {
            this.receiveBufferSize = receiveBufferSize;
            return self();
        }

        /**
         * Configure socket keep alive.
         * Default is {@code true}.
         *
         * @param keepAlive keep alive
         * @return updated builder instance
         * @see #keepAlive()
         */
        public BUILDER keepAlive(boolean keepAlive) {
            this.keepAlive = keepAlive;
            return self();
        }

        /**
         * Socket reuse address.
         * Default is {@code true}.
         *
         * @param reuseAddress whether to reuse address
         * @return updated builder instance
         * @see #reuseAddress()
         */
        public BUILDER reuseAddress(boolean reuseAddress) {
            this.reuseAddress = reuseAddress;
            return self();
        }

        /**
         * This option may improve performance on some systems.
         * Default is {@code false}.
         *
         * @param tcpNoDelay whether to use TCP_NODELAY, defaults to {@code false}
         * @return updated builder instance
         * @see #tcpNoDelay()
         */
        public BUILDER tcpNoDelay(boolean tcpNoDelay) {
            this.tcpNoDelay = tcpNoDelay;
            return self();
        }

        /**
         * This method replaces all values with the new ones.
         *
         * @param socketOptions socket options
         * @return updated builder instance
         * @see #socketOptions()
         */
        public BUILDER socketOptions(Map<SocketOption<?>, ?> socketOptions) {
            Objects.requireNonNull(socketOptions);
            this.socketOptions.clear();
            this.socketOptions.putAll(socketOptions);
            return self();
        }

        /**
         * This method keeps existing values, then puts all new values into the map.
         *
         * @param socketOptions socket options
         * @return updated builder instance
         * @see #socketOptions()
         */
        public BUILDER addSocketOptions(Map<SocketOption<?>, ?> socketOptions) {
            Objects.requireNonNull(socketOptions);
            this.socketOptions.putAll(socketOptions);
            return self();
        }

        /**
         * This method adds a new value to the map, or replaces it if the key already exists.
         *
         * @param key key to add or replace
         * @param socketOption new value for the key
         * @return updated builder instance
         * @see #socketOptions()
         */
        public BUILDER putSocketOption(SocketOption<?> key, Object socketOption) {
            Objects.requireNonNull(key);
            Objects.requireNonNull(socketOption);
            this.socketOptions.put(key, socketOption);
            return self();
        }

        /**
         * Read timeout.
         * Default is {@value #DEFAULT_READ_TIMEOUT_DURATION}
         *
         * @return the read timeout
         */
        public Duration readTimeout() {
            return readTimeout;
        }

        /**
         * Connect timeout.
         * Default is {@value #DEFAULT_CONNECT_TIMEOUT_DURATION}.
         *
         * @return the connect timeout
         */
        public Duration connectTimeout() {
            return connectTimeout;
        }

        /**
         * Socket send buffer size.
         * Default is {@value #DEFAULT_SO_BUFFER_SIZE}.
         *
         * @return the send buffer size
         */
        public int sendBufferSize() {
            return sendBufferSize;
        }

        /**
         * Socket receive buffer size.
         * Default is {@value #DEFAULT_SO_BUFFER_SIZE}.
         *
         * @return the receive buffer size
         */
        public int receiveBufferSize() {
            return receiveBufferSize;
        }

        /**
         * Configure socket keep alive.
         * Default is {@code true}.
         *
         * @return the keep alive
         */
        public boolean keepAlive() {
            return keepAlive;
        }

        /**
         * Socket reuse address.
         * Default is {@code true}.
         *
         * @return the reuse address
         */
        public boolean reuseAddress() {
            return reuseAddress;
        }

        /**
         * This option may improve performance on some systems.
         * Default is {@code false}.
         *
         * @return the tcp no delay
         */
        public boolean tcpNoDelay() {
            return tcpNoDelay;
        }

        /**
         * Set an arbitrary socket option. A mapping of a socket option to its value.
         * Socket options may be system specific. Most commonly supported socket options are available as methods directly.
         *
         * @return the socket options
         */
        public Map<SocketOption<?>, Object> socketOptions() {
            return socketOptions;
        }

        @Override
        public String toString() {
            return "ConnectionConfigBuilder{"
                    + "readTimeout=" + readTimeout + ","
                    + "connectTimeout=" + connectTimeout + ","
                    + "sendBufferSize=" + sendBufferSize + ","
                    + "receiveBufferSize=" + receiveBufferSize + ","
                    + "keepAlive=" + keepAlive + ","
                    + "reuseAddress=" + reuseAddress + ","
                    + "tcpNoDelay=" + tcpNoDelay + ","
                    + "socketOptions=" + socketOptions
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class ConnectionConfigImpl implements ConnectionConfig {

            private final boolean keepAlive;
            private final boolean reuseAddress;
            private final boolean tcpNoDelay;
            private final Duration connectTimeout;
            private final Duration readTimeout;
            private final int receiveBufferSize;
            private final int sendBufferSize;
            private final Map<SocketOption<?>, Object> socketOptions;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected ConnectionConfigImpl(ConnectionConfig.BuilderBase<?, ?> builder) {
                this.readTimeout = builder.readTimeout();
                this.connectTimeout = builder.connectTimeout();
                this.sendBufferSize = builder.sendBufferSize();
                this.receiveBufferSize = builder.receiveBufferSize();
                this.keepAlive = builder.keepAlive();
                this.reuseAddress = builder.reuseAddress();
                this.tcpNoDelay = builder.tcpNoDelay();
                this.socketOptions = Collections.unmodifiableMap(new LinkedHashMap<>(builder.socketOptions()));
            }

            @Override
            public Duration readTimeout() {
                return readTimeout;
            }

            @Override
            public Duration connectTimeout() {
                return connectTimeout;
            }

            @Override
            public int sendBufferSize() {
                return sendBufferSize;
            }

            @Override
            public int receiveBufferSize() {
                return receiveBufferSize;
            }

            @Override
            public boolean keepAlive() {
                return keepAlive;
            }

            @Override
            public boolean reuseAddress() {
                return reuseAddress;
            }

            @Override
            public boolean tcpNoDelay() {
                return tcpNoDelay;
            }

            @Override
            public Map<SocketOption<?>, Object> socketOptions() {
                return socketOptions;
            }

            @Override
            public String toString() {
                return "ConnectionConfig{"
                        + "readTimeout=" + readTimeout + ","
                        + "connectTimeout=" + connectTimeout + ","
                        + "sendBufferSize=" + sendBufferSize + ","
                        + "receiveBufferSize=" + receiveBufferSize + ","
                        + "keepAlive=" + keepAlive + ","
                        + "reuseAddress=" + reuseAddress + ","
                        + "tcpNoDelay=" + tcpNoDelay + ","
                        + "socketOptions=" + socketOptions
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof ConnectionConfig other)) {
                    return false;
                }
                return Objects.equals(readTimeout, other.readTimeout())
                        && Objects.equals(connectTimeout, other.connectTimeout())
                        && sendBufferSize == other.sendBufferSize()
                        && receiveBufferSize == other.receiveBufferSize()
                        && keepAlive == other.keepAlive()
                        && reuseAddress == other.reuseAddress()
                        && tcpNoDelay == other.tcpNoDelay()
                        && Objects.equals(socketOptions, other.socketOptions());
            }

            @Override
            public int hashCode() {
                return Objects.hash(readTimeout, connectTimeout, sendBufferSize, receiveBufferSize, keepAlive, reuseAddress, tcpNoDelay, socketOptions);
            }

        }

    }

    /**
     * Fluent API builder for {@link ConnectionConfig}.
     */
    class Builder extends ConnectionConfig.BuilderBase<ConnectionConfig.Builder, ConnectionConfig> implements io.helidon.common.Builder<ConnectionConfig.Builder, ConnectionConfig> {

        private Builder() {
        }

        @Override
        public ConnectionConfig buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new ConnectionConfigImpl(this);
        }

        @Override
        public ConnectionConfig build() {
            return buildPrototype();
        }

    }

}
