/*
 * Copyright (c) 2023 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.webserver.websocket;

import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Generated;
import io.helidon.common.config.Config;

/**
 * WebSocket protocol configuration.
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.processor.BlueprintProcessor", trigger = "io.helidon.webserver.websocket.WsConfigBlueprint")
public interface WsConfig extends WsConfigBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static WsConfig.Builder builder() {
        return new WsConfig.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static WsConfig.Builder builder(WsConfig instance) {
        return WsConfig.builder().from(instance);
    }

    /**
     * Create a new instance from configuration.
     *
     * @param config used to configure the new instance
     * @return a new instance configured from configuration
     */
    static WsConfig create(Config config) {
        return WsConfig.builder().config(config).buildPrototype();
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static WsConfig create() {
        return WsConfig.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link WsConfig}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends WsConfig.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends WsConfig> implements Prototype.ConfiguredBuilder<BUILDER, PROTOTYPE> {

        private final Set<String> origins = new LinkedHashSet<>();
        private Config config;
        private int maxFrameLength = 1048576;
        private String name = "websocket";

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(WsConfig prototype) {
            addOrigins(prototype.origins());
            name(prototype.name());
            maxFrameLength(prototype.maxFrameLength());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(WsConfig.BuilderBase<?, ?> builder) {
            addOrigins(builder.origins());
            name(builder.name());
            maxFrameLength(builder.maxFrameLength());
            return self();
        }

        /**
         * Update builder from configuration (node of this type).
         * If a value is present in configuration, it would override currently configured values.
         *
         * @param config configuration instance used to obtain values to update this builder
         * @return updated builder instance
         */
        @Override
        public BUILDER config(Config config) {
            Objects.requireNonNull(config);
            this.config = config;
            config.get("origins").asList(String.class).map(java.util.Set::copyOf).ifPresent(this::origins);
            config.get("name").as(String.class).ifPresent(this::name);
            config.get("max-frame-length").as(Integer.class).ifPresent(this::maxFrameLength);
            return self();
        }

        /**
         * WebSocket origins.
         *
         * @param origins origins
         * @return updated builder instance
         * @see #origins()
         */
        public BUILDER origins(Set<? extends String> origins) {
            Objects.requireNonNull(origins);
            this.origins.clear();
            this.origins.addAll(origins);
            return self();
        }

        /**
         * WebSocket origins.
         *
         * @param origins origins
         * @return updated builder instance
         * @see #origins()
         */
        public BUILDER addOrigins(Set<? extends String> origins) {
            Objects.requireNonNull(origins);
            this.origins.addAll(origins);
            return self();
        }

        /**
         * WebSocket origins.
         *
         * @param origin origins
         * @return updated builder instance
         * @see #origins()
         */
        public BUILDER addOrigin(String origin) {
            Objects.requireNonNull(origin);
            this.origins.add(origin);
            return self();
        }

        /**
         * Name of this configuration.
         *
         * @param name configuration name
         * @return updated builder instance
         * @see #name()
         */
        public BUILDER name(String name) {
            Objects.requireNonNull(name);
            this.name = name;
            return self();
        }

        /**
         * Max WebSocket frame size supported by the server on a read operation.
         * Default is 1 MB.
         *
         * @param maxFrameLength max frame size to read
         * @return updated builder instance
         * @see #maxFrameLength()
         */
        public BUILDER maxFrameLength(int maxFrameLength) {
            this.maxFrameLength = maxFrameLength;
            return self();
        }

        /**
         * WebSocket origins.
         *
         * @return the origins
         */
        public Set<String> origins() {
            return origins;
        }

        /**
         * Name of this configuration.
         *
         * @return the name
         */
        public String name() {
            return name;
        }

        /**
         * Max WebSocket frame size supported by the server on a read operation.
         * Default is 1 MB.
         *
         * @return the max frame length
         */
        public int maxFrameLength() {
            return maxFrameLength;
        }

        /**
         * If this instance was configured, this would be the config instance used.
         *
         * @return config node used to configure this builder, or empty if not configured
         */
        public Optional<Config> config() {
            return Optional.ofNullable(config);
        }

        @Override
        public String toString() {
            return "WsConfigBuilder{"
                    + "origins=" + origins + ","
                    + "name=" + name + ","
                    + "maxFrameLength=" + maxFrameLength
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class WsConfigImpl implements WsConfig {

            private final int maxFrameLength;
            private final Set<String> origins;
            private final String name;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected WsConfigImpl(WsConfig.BuilderBase<?, ?> builder) {
                this.origins = Collections.unmodifiableSet(new LinkedHashSet<>(builder.origins()));
                this.name = builder.name();
                this.maxFrameLength = builder.maxFrameLength();
            }

            @Override
            public Set<String> origins() {
                return origins;
            }

            @Override
            public String name() {
                return name;
            }

            @Override
            public int maxFrameLength() {
                return maxFrameLength;
            }

            @Override
            public String toString() {
                return "WsConfig{"
                        + "origins=" + origins + ","
                        + "name=" + name + ","
                        + "maxFrameLength=" + maxFrameLength
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof WsConfig other)) {
                    return false;
                }
                return Objects.equals(origins, other.origins())
                        && Objects.equals(name, other.name())
                        && maxFrameLength == other.maxFrameLength();
            }

            @Override
            public int hashCode() {
                return Objects.hash(origins, name, maxFrameLength);
            }

        }

    }

    /**
     * Fluent API builder for {@link WsConfig}.
     */
    class Builder extends WsConfig.BuilderBase<WsConfig.Builder, WsConfig> implements io.helidon.common.Builder<WsConfig.Builder, WsConfig> {

        private Builder() {
        }

        @Override
        public WsConfig buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new WsConfigImpl(this);
        }

        @Override
        public WsConfig build() {
            return buildPrototype();
        }

    }

}
