/*
 * Copyright (c) 2024 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.webserver.security;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Supplier;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Generated;
import io.helidon.common.config.Config;
import io.helidon.security.ClassToInstanceStore;

/**
 * Configuration of a {@link io.helidon.webserver.security.SecurityHandler}.
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.processor.BlueprintProcessor", trigger = "io.helidon.webserver.security.SecurityHandlerConfigBlueprint")
public interface SecurityHandlerConfig extends SecurityHandlerConfigBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static SecurityHandlerConfig.Builder builder() {
        return new SecurityHandlerConfig.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static SecurityHandlerConfig.Builder builder(SecurityHandlerConfig instance) {
        return SecurityHandlerConfig.builder().from(instance);
    }

    /**
     * Create a new instance from configuration.
     *
     * @param config used to configure the new instance
     * @return a new instance configured from configuration
     */
    static SecurityHandlerConfig create(Config config) {
        return SecurityHandlerConfig.builder().config(config).buildPrototype();
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static SecurityHandlerConfig create() {
        return SecurityHandlerConfig.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link SecurityHandler}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends SecurityHandlerConfig.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends SecurityHandlerConfig> implements Prototype.ConfiguredBuilder<BUILDER, PROTOTYPE> {

        private final List<SecurityHandler.QueryParamHandler> queryParams = new ArrayList<>();
        private final List<String> sockets = new ArrayList<>();
        private final Set<String> rolesAllowed = new LinkedHashSet<>();
        private Boolean audit;
        private Boolean authenticate;
        private Boolean authenticationOptional;
        private Boolean authorize;
        private boolean combined;
        private ClassToInstanceStore<Object> customObjects;
        private Config config;
        private String auditEventType;
        private String auditMessageFormat;
        private String authenticator;
        private String authorizer;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(SecurityHandlerConfig prototype) {
            addRolesAllowed(prototype.rolesAllowed());
            authenticator(prototype.authenticator());
            authorizer(prototype.authorizer());
            authenticate(prototype.authenticate());
            authenticationOptional(prototype.authenticationOptional());
            audit(prototype.audit());
            authorize(prototype.authorize());
            auditEventType(prototype.auditEventType());
            auditMessageFormat(prototype.auditMessageFormat());
            addQueryParams(prototype.queryParams());
            customObjects(prototype.customObjects());
            addSockets(prototype.sockets());
            this.config = prototype.config().orElse(null);
            combined(prototype.combined());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(SecurityHandlerConfig.BuilderBase<?, ?> builder) {
            addRolesAllowed(builder.rolesAllowed());
            builder.authenticator().ifPresent(this::authenticator);
            builder.authorizer().ifPresent(this::authorizer);
            builder.authenticate().ifPresent(this::authenticate);
            builder.authenticationOptional().ifPresent(this::authenticationOptional);
            builder.audit().ifPresent(this::audit);
            builder.authorize().ifPresent(this::authorize);
            builder.auditEventType().ifPresent(this::auditEventType);
            builder.auditMessageFormat().ifPresent(this::auditMessageFormat);
            addQueryParams(builder.queryParams());
            builder.customObjects().ifPresent(this::customObjects);
            addSockets(builder.sockets());
            builder.config().ifPresent(this::config);
            combined(builder.combined());
            return self();
        }

        /**
         * Register a custom object for security request(s).
         * This creates a hard dependency on a specific security provider, so use with care.
         *
         * @param object  An object expected by security provider
         * @return updated builder instance
         */
        public BUILDER addObject(Object object) {
            SecurityConfigSupport.SecurityHandlerCustomMethods.addObject(this, object);
            return self();
        }

        /**
         * Register a custom object for security request(s).
         * This creates a hard dependency on a specific security provider, so use with care.
         *
         * @param objectType
         * @param object  An object expected by security provider
         * @return updated builder instance
         */
        public BUILDER addObject(Class<?> objectType, Object object) {
            SecurityConfigSupport.SecurityHandlerCustomMethods.addObject(this, objectType, object);
            return self();
        }

        /**
         * Update builder from configuration (node of this type).
         * If a value is present in configuration, it would override currently configured values.
         *
         * @param config configuration instance used to obtain values to update this builder
         * @return updated builder instance
         */
        @Override
        public BUILDER config(Config config) {
            Objects.requireNonNull(config);
            this.config = config;
            config.get("roles-allowed").asList(String.class).map(java.util.Set::copyOf).ifPresent(this::rolesAllowed);
            config.get("authenticator").as(String.class).ifPresent(this::authenticator);
            config.get("authorizer").as(String.class).ifPresent(this::authorizer);
            config.get("authenticate").as(Boolean.class).ifPresent(this::authenticate);
            config.get("authentication-optional").as(Boolean.class).ifPresent(this::authenticationOptional);
            config.get("audit").as(Boolean.class).ifPresent(this::audit);
            config.get("authorize").as(Boolean.class).ifPresent(this::authorize);
            config.get("audit-event-type").as(String.class).ifPresent(this::auditEventType);
            config.get("audit-message-format").as(String.class).ifPresent(this::auditMessageFormat);
            config.get("sockets").asList(String.class).ifPresent(this::sockets);
            return self();
        }

        /**
         * An array of allowed roles for this path - must have a security provider supporting roles (either authentication
         * or authorization provider).
         * This method enables authentication and authorization (you can disable them again by calling
         * {@link SecurityHandler#skipAuthorization()}
         * and {@link #authenticationOptional()} if needed).
         *
         * @param rolesAllowed if subject is any of these roles, allow access
         * @return updated builder instance
         * @see #rolesAllowed()
         */
        public BUILDER rolesAllowed(Set<? extends String> rolesAllowed) {
            Objects.requireNonNull(rolesAllowed);
            this.rolesAllowed.clear();
            this.rolesAllowed.addAll(rolesAllowed);
            return self();
        }

        /**
         * An array of allowed roles for this path - must have a security provider supporting roles (either authentication
         * or authorization provider).
         * This method enables authentication and authorization (you can disable them again by calling
         * {@link SecurityHandler#skipAuthorization()}
         * and {@link #authenticationOptional()} if needed).
         *
         * @param rolesAllowed if subject is any of these roles, allow access
         * @return updated builder instance
         * @see #rolesAllowed()
         */
        public BUILDER addRolesAllowed(Set<? extends String> rolesAllowed) {
            Objects.requireNonNull(rolesAllowed);
            this.rolesAllowed.addAll(rolesAllowed);
            return self();
        }

        /**
         * An array of allowed roles for this path - must have a security provider supporting roles (either authentication
         * or authorization provider).
         * This method enables authentication and authorization (you can disable them again by calling
         * {@link SecurityHandler#skipAuthorization()}
         * and {@link #authenticationOptional()} if needed).
         *
         * @param roleAllowed if subject is any of these roles, allow access
         * @return updated builder instance
         * @see #rolesAllowed()
         */
        public BUILDER addRoleAllowed(String roleAllowed) {
            Objects.requireNonNull(roleAllowed);
            this.rolesAllowed.add(roleAllowed);
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #authenticator()
         */
        public BUILDER clearAuthenticator() {
            this.authenticator = null;
            return self();
        }

        /**
         * Use a named authenticator (as supported by security - if not defined, default authenticator is used).
         * Will enable authentication.
         *
         * @param authenticator name of authenticator as configured in {@link io.helidon.security.Security}
         * @return updated builder instance
         * @see #authenticator()
         */
        public BUILDER authenticator(String authenticator) {
            Objects.requireNonNull(authenticator);
            this.authenticator = authenticator;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #authorizer()
         */
        public BUILDER clearAuthorizer() {
            this.authorizer = null;
            return self();
        }

        /**
         * Use a named authorizer (as supported by security - if not defined, default authorizer is used, if none defined, all is
         * permitted).
         * Will enable authorization.
         *
         * @param authorizer name of authorizer as configured in {@link io.helidon.security.Security}
         * @return updated builder instance
         * @see #authorizer()
         */
        public BUILDER authorizer(String authorizer) {
            Objects.requireNonNull(authorizer);
            this.authorizer = authorizer;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #authenticate()
         */
        public BUILDER clearAuthenticate() {
            this.authenticate = null;
            return self();
        }

        /**
         * If called, request will go through authentication process - defaults to false (even if authorize is true).
         *
         * @param authenticate whether to authenticate or not
         * @return updated builder instance
         * @see #authenticate()
         */
        public BUILDER authenticate(boolean authenticate) {
            Objects.requireNonNull(authenticate);
            this.authenticate = authenticate;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #authenticationOptional()
         */
        public BUILDER clearAuthenticationOptional() {
            this.authenticationOptional = null;
            return self();
        }

        /**
         * If called, authentication failure will not abort request and will continue as anonymous (defaults to false).
         *
         * @param authenticationOptional whether authn is optional
         * @return updated builder instance
         * @see #authenticationOptional()
         */
        public BUILDER authenticationOptional(boolean authenticationOptional) {
            Objects.requireNonNull(authenticationOptional);
            this.authenticationOptional = authenticationOptional;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #audit()
         */
        public BUILDER clearAudit() {
            this.audit = null;
            return self();
        }

        /**
         * Whether to audit this request - defaults to false, if enabled, request is audited with event type "request".
         *
         * @param audit whether to audit
         * @return updated builder instance
         * @see #audit()
         */
        public BUILDER audit(boolean audit) {
            Objects.requireNonNull(audit);
            this.audit = audit;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #authorize()
         */
        public BUILDER clearAuthorize() {
            this.authorize = null;
            return self();
        }

        /**
         * Enable authorization for this route.
         *
         * @param authorize whether to authorize
         * @return updated builder instance
         * @see #authorize()
         */
        public BUILDER authorize(boolean authorize) {
            Objects.requireNonNull(authorize);
            this.authorize = authorize;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #auditEventType()
         */
        public BUILDER clearAuditEventType() {
            this.auditEventType = null;
            return self();
        }

        /**
         * Override for event-type, defaults to {@value SecurityHandler#DEFAULT_AUDIT_EVENT_TYPE}.
         *
         * @param auditEventType audit event type to use
         * @return updated builder instance
         * @see #auditEventType()
         */
        public BUILDER auditEventType(String auditEventType) {
            Objects.requireNonNull(auditEventType);
            this.auditEventType = auditEventType;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #auditMessageFormat()
         */
        public BUILDER clearAuditMessageFormat() {
            this.auditMessageFormat = null;
            return self();
        }

        /**
         * Override for audit message format, defaults to {@value SecurityHandler#DEFAULT_AUDIT_MESSAGE_FORMAT}.
         *
         * @param auditMessageFormat audit message format to use
         * @return updated builder instance
         * @see #auditMessageFormat()
         */
        public BUILDER auditMessageFormat(String auditMessageFormat) {
            Objects.requireNonNull(auditMessageFormat);
            this.auditMessageFormat = auditMessageFormat;
            return self();
        }

        /**
         * Query parameter handler(s).
         *
         * @param queryParams query parameters
         * @return updated builder instance
         * @see #queryParams()
         */
        public BUILDER queryParams(List<? extends SecurityHandler.QueryParamHandler> queryParams) {
            Objects.requireNonNull(queryParams);
            this.queryParams.clear();
            this.queryParams.addAll(queryParams);
            return self();
        }

        /**
         * Query parameter handler(s).
         *
         * @param queryParams query parameters
         * @return updated builder instance
         * @see #queryParams()
         */
        public BUILDER addQueryParams(List<? extends SecurityHandler.QueryParamHandler> queryParams) {
            Objects.requireNonNull(queryParams);
            this.queryParams.addAll(queryParams);
            return self();
        }

        /**
         * Query parameter handler(s).
         *
         * @param queryParam query parameters
         * @return updated builder instance
         * @see #queryParams()
         */
        public BUILDER addQueryParam(SecurityHandler.QueryParamHandler queryParam) {
            Objects.requireNonNull(queryParam);
            this.queryParams.add(queryParam);
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #customObjects()
         */
        public BUILDER clearCustomObjects() {
            this.customObjects = null;
            return self();
        }

        /**
         * A store of custom objects, that can be used to customize specific security providers.
         *
         * @param customObjects custom objects
         * @return updated builder instance
         * @see #customObjects()
         */
        public BUILDER customObjects(ClassToInstanceStore<Object> customObjects) {
            Objects.requireNonNull(customObjects);
            this.customObjects = customObjects;
            return self();
        }

        /**
         * List of sockets this configuration should be applied to.
         * If empty, the configuration is applied to all configured sockets.
         *
         * @param sockets list of sockets
         * @return updated builder instance
         * @see #sockets()
         */
        public BUILDER sockets(List<? extends String> sockets) {
            Objects.requireNonNull(sockets);
            this.sockets.clear();
            this.sockets.addAll(sockets);
            return self();
        }

        /**
         * List of sockets this configuration should be applied to.
         * If empty, the configuration is applied to all configured sockets.
         *
         * @param sockets list of sockets
         * @return updated builder instance
         * @see #sockets()
         */
        public BUILDER addSockets(List<? extends String> sockets) {
            Objects.requireNonNull(sockets);
            this.sockets.addAll(sockets);
            return self();
        }

        /**
         * Whether this is a combined handler. Internal use.
         *
         * @param combined if combined handler
         * @return updated builder instance
         * @see #combined()
         */
        public BUILDER combined(boolean combined) {
            this.combined = combined;
            return self();
        }

        /**
         * An array of allowed roles for this path - must have a security provider supporting roles (either authentication
         * or authorization provider).
         * This method enables authentication and authorization (you can disable them again by calling
         * {@link SecurityHandler#skipAuthorization()}
         * and {@link #authenticationOptional()} if needed).
         *
         * @return the roles allowed
         */
        public Set<String> rolesAllowed() {
            return rolesAllowed;
        }

        /**
         * Use a named authenticator (as supported by security - if not defined, default authenticator is used).
         * Will enable authentication.
         *
         * @return the authenticator
         */
        public Optional<String> authenticator() {
            return Optional.ofNullable(authenticator);
        }

        /**
         * Use a named authorizer (as supported by security - if not defined, default authorizer is used, if none defined, all is
         * permitted).
         * Will enable authorization.
         *
         * @return the authorizer
         */
        public Optional<String> authorizer() {
            return Optional.ofNullable(authorizer);
        }

        /**
         * If called, request will go through authentication process - defaults to false (even if authorize is true).
         *
         * @return the authenticate
         */
        public Optional<Boolean> authenticate() {
            return Optional.ofNullable(authenticate);
        }

        /**
         * If called, authentication failure will not abort request and will continue as anonymous (defaults to false).
         *
         * @return the authentication optional
         */
        public Optional<Boolean> authenticationOptional() {
            return Optional.ofNullable(authenticationOptional);
        }

        /**
         * Whether to audit this request - defaults to false, if enabled, request is audited with event type "request".
         *
         * @return the audit
         */
        public Optional<Boolean> audit() {
            return Optional.ofNullable(audit);
        }

        /**
         * Enable authorization for this route.
         *
         * @return the authorize
         */
        public Optional<Boolean> authorize() {
            return Optional.ofNullable(authorize);
        }

        /**
         * Override for event-type, defaults to {@value SecurityHandler#DEFAULT_AUDIT_EVENT_TYPE}.
         *
         * @return the audit event type
         */
        public Optional<String> auditEventType() {
            return Optional.ofNullable(auditEventType);
        }

        /**
         * Override for audit message format, defaults to {@value SecurityHandler#DEFAULT_AUDIT_MESSAGE_FORMAT}.
         *
         * @return the audit message format
         */
        public Optional<String> auditMessageFormat() {
            return Optional.ofNullable(auditMessageFormat);
        }

        /**
         * Query parameter handler(s).
         *
         * @return the query params
         */
        public List<SecurityHandler.QueryParamHandler> queryParams() {
            return queryParams;
        }

        /**
         * A store of custom objects, that can be used to customize specific security providers.
         *
         * @return the custom objects
         */
        public Optional<ClassToInstanceStore<Object>> customObjects() {
            return Optional.ofNullable(customObjects);
        }

        /**
         * List of sockets this configuration should be applied to.
         * If empty, the configuration is applied to all configured sockets.
         *
         * @return the sockets
         */
        public List<String> sockets() {
            return sockets;
        }

        /**
         * Whether this is a combined handler. Internal use.
         *
         * @return the combined
         */
        public boolean combined() {
            return combined;
        }

        /**
         * If this instance was configured, this would be the config instance used.
         *
         * @return config node used to configure this builder, or empty if not configured
         */
        public Optional<Config> config() {
            return Optional.ofNullable(config);
        }

        @Override
        public String toString() {
            return "SecurityHandlerConfigBuilder{"
                    + "rolesAllowed=" + rolesAllowed + ","
                    + "authenticator=" + authenticator + ","
                    + "authorizer=" + authorizer + ","
                    + "authenticate=" + authenticate + ","
                    + "authenticationOptional=" + authenticationOptional + ","
                    + "audit=" + audit + ","
                    + "authorize=" + authorize + ","
                    + "auditEventType=" + auditEventType + ","
                    + "auditMessageFormat=" + auditMessageFormat + ","
                    + "queryParams=" + queryParams + ","
                    + "customObjects=" + customObjects + ","
                    + "sockets=" + sockets + ","
                    + "config=" + config + ","
                    + "combined=" + combined
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
            new SecurityConfigSupport.SecurityHandlerDecorator().decorate(this);
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
        }

        /**
         * Use a named authenticator (as supported by security - if not defined, default authenticator is used).
         * Will enable authentication.
         *
         * @param authenticator name of authenticator as configured in {@link io.helidon.security.Security}
         * @return updated builder instance
         * @see #authenticator()
         */
        BUILDER authenticator(Optional<String> authenticator) {
            Objects.requireNonNull(authenticator);
            this.authenticator = authenticator.map(java.lang.String.class::cast).orElse(this.authenticator);
            return self();
        }

        /**
         * Use a named authorizer (as supported by security - if not defined, default authorizer is used, if none defined, all is
         * permitted).
         * Will enable authorization.
         *
         * @param authorizer name of authorizer as configured in {@link io.helidon.security.Security}
         * @return updated builder instance
         * @see #authorizer()
         */
        BUILDER authorizer(Optional<String> authorizer) {
            Objects.requireNonNull(authorizer);
            this.authorizer = authorizer.map(java.lang.String.class::cast).orElse(this.authorizer);
            return self();
        }

        /**
         * If called, request will go through authentication process - defaults to false (even if authorize is true).
         *
         * @param authenticate whether to authenticate or not
         * @return updated builder instance
         * @see #authenticate()
         */
        BUILDER authenticate(Optional<Boolean> authenticate) {
            Objects.requireNonNull(authenticate);
            this.authenticate = authenticate.map(java.lang.Boolean.class::cast).orElse(this.authenticate);
            return self();
        }

        /**
         * If called, authentication failure will not abort request and will continue as anonymous (defaults to false).
         *
         * @param authenticationOptional whether authn is optional
         * @return updated builder instance
         * @see #authenticationOptional()
         */
        BUILDER authenticationOptional(Optional<Boolean> authenticationOptional) {
            Objects.requireNonNull(authenticationOptional);
            this.authenticationOptional = authenticationOptional.map(java.lang.Boolean.class::cast).orElse(this.authenticationOptional);
            return self();
        }

        /**
         * Whether to audit this request - defaults to false, if enabled, request is audited with event type "request".
         *
         * @param audit whether to audit
         * @return updated builder instance
         * @see #audit()
         */
        BUILDER audit(Optional<Boolean> audit) {
            Objects.requireNonNull(audit);
            this.audit = audit.map(java.lang.Boolean.class::cast).orElse(this.audit);
            return self();
        }

        /**
         * Enable authorization for this route.
         *
         * @param authorize whether to authorize
         * @return updated builder instance
         * @see #authorize()
         */
        BUILDER authorize(Optional<Boolean> authorize) {
            Objects.requireNonNull(authorize);
            this.authorize = authorize.map(java.lang.Boolean.class::cast).orElse(this.authorize);
            return self();
        }

        /**
         * Override for event-type, defaults to {@value SecurityHandler#DEFAULT_AUDIT_EVENT_TYPE}.
         *
         * @param auditEventType audit event type to use
         * @return updated builder instance
         * @see #auditEventType()
         */
        BUILDER auditEventType(Optional<String> auditEventType) {
            Objects.requireNonNull(auditEventType);
            this.auditEventType = auditEventType.map(java.lang.String.class::cast).orElse(this.auditEventType);
            return self();
        }

        /**
         * Override for audit message format, defaults to {@value SecurityHandler#DEFAULT_AUDIT_MESSAGE_FORMAT}.
         *
         * @param auditMessageFormat audit message format to use
         * @return updated builder instance
         * @see #auditMessageFormat()
         */
        BUILDER auditMessageFormat(Optional<String> auditMessageFormat) {
            Objects.requireNonNull(auditMessageFormat);
            this.auditMessageFormat = auditMessageFormat.map(java.lang.String.class::cast).orElse(this.auditMessageFormat);
            return self();
        }

        /**
         * A store of custom objects, that can be used to customize specific security providers.
         *
         * @param customObjects custom objects
         * @return updated builder instance
         * @see #customObjects()
         */
        BUILDER customObjects(Optional<ClassToInstanceStore<Object>> customObjects) {
            Objects.requireNonNull(customObjects);
            this.customObjects = customObjects.map(io.helidon.security.ClassToInstanceStore.class::cast).orElse(this.customObjects);
            return self();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class SecurityHandlerConfigImpl implements SecurityHandlerConfig, Supplier<SecurityHandler> {

            private final boolean combined;
            private final List<SecurityHandler.QueryParamHandler> queryParams;
            private final List<String> sockets;
            private final Optional<Config> config;
            private final Optional<ClassToInstanceStore<Object>> customObjects;
            private final Optional<Boolean> audit;
            private final Optional<Boolean> authenticate;
            private final Optional<Boolean> authenticationOptional;
            private final Optional<Boolean> authorize;
            private final Optional<String> auditEventType;
            private final Optional<String> auditMessageFormat;
            private final Optional<String> authenticator;
            private final Optional<String> authorizer;
            private final Set<String> rolesAllowed;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected SecurityHandlerConfigImpl(SecurityHandlerConfig.BuilderBase<?, ?> builder) {
                this.rolesAllowed = Collections.unmodifiableSet(new LinkedHashSet<>(builder.rolesAllowed()));
                this.authenticator = builder.authenticator();
                this.authorizer = builder.authorizer();
                this.authenticate = builder.authenticate();
                this.authenticationOptional = builder.authenticationOptional();
                this.audit = builder.audit();
                this.authorize = builder.authorize();
                this.auditEventType = builder.auditEventType();
                this.auditMessageFormat = builder.auditMessageFormat();
                this.queryParams = List.copyOf(builder.queryParams());
                this.customObjects = builder.customObjects();
                this.sockets = List.copyOf(builder.sockets());
                this.config = builder.config();
                this.combined = builder.combined();
            }

            @Override
            public SecurityHandler build() {
                return SecurityHandler.create(this);
            }

            @Override
            public SecurityHandler get() {
                return build();
            }

            @Override
            public Set<String> rolesAllowed() {
                return rolesAllowed;
            }

            @Override
            public Optional<String> authenticator() {
                return authenticator;
            }

            @Override
            public Optional<String> authorizer() {
                return authorizer;
            }

            @Override
            public Optional<Boolean> authenticate() {
                return authenticate;
            }

            @Override
            public Optional<Boolean> authenticationOptional() {
                return authenticationOptional;
            }

            @Override
            public Optional<Boolean> audit() {
                return audit;
            }

            @Override
            public Optional<Boolean> authorize() {
                return authorize;
            }

            @Override
            public Optional<String> auditEventType() {
                return auditEventType;
            }

            @Override
            public Optional<String> auditMessageFormat() {
                return auditMessageFormat;
            }

            @Override
            public List<SecurityHandler.QueryParamHandler> queryParams() {
                return queryParams;
            }

            @Override
            public Optional<ClassToInstanceStore<Object>> customObjects() {
                return customObjects;
            }

            @Override
            public List<String> sockets() {
                return sockets;
            }

            @Override
            public Optional<Config> config() {
                return config;
            }

            @Override
            public boolean combined() {
                return combined;
            }

            @Override
            public String toString() {
                return "SecurityHandlerConfig{"
                        + "rolesAllowed=" + rolesAllowed + ","
                        + "authenticator=" + authenticator + ","
                        + "authorizer=" + authorizer + ","
                        + "authenticate=" + authenticate + ","
                        + "authenticationOptional=" + authenticationOptional + ","
                        + "audit=" + audit + ","
                        + "authorize=" + authorize + ","
                        + "auditEventType=" + auditEventType + ","
                        + "auditMessageFormat=" + auditMessageFormat + ","
                        + "queryParams=" + queryParams + ","
                        + "customObjects=" + customObjects + ","
                        + "sockets=" + sockets + ","
                        + "config=" + config + ","
                        + "combined=" + combined
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof SecurityHandlerConfig other)) {
                    return false;
                }
                return Objects.equals(rolesAllowed, other.rolesAllowed())
                        && Objects.equals(authenticator, other.authenticator())
                        && Objects.equals(authorizer, other.authorizer())
                        && Objects.equals(authenticate, other.authenticate())
                        && Objects.equals(authenticationOptional, other.authenticationOptional())
                        && Objects.equals(audit, other.audit())
                        && Objects.equals(authorize, other.authorize())
                        && Objects.equals(auditEventType, other.auditEventType())
                        && Objects.equals(auditMessageFormat, other.auditMessageFormat())
                        && Objects.equals(queryParams, other.queryParams())
                        && Objects.equals(customObjects, other.customObjects())
                        && Objects.equals(sockets, other.sockets())
                        && Objects.equals(config, other.config())
                        && combined == other.combined();
            }

            @Override
            public int hashCode() {
                return Objects.hash(rolesAllowed, authenticator, authorizer, authenticate, authenticationOptional, audit, authorize, auditEventType, auditMessageFormat, queryParams, customObjects, sockets, config, combined);
            }

        }

    }

    /**
     * Fluent API builder for {@link SecurityHandler}.
     */
    class Builder extends SecurityHandlerConfig.BuilderBase<SecurityHandlerConfig.Builder, SecurityHandlerConfig> implements io.helidon.common.Builder<SecurityHandlerConfig.Builder, SecurityHandler> {

        private Builder() {
        }

        @Override
        public SecurityHandlerConfig buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new SecurityHandlerConfigImpl(this);
        }

        @Override
        public SecurityHandler build() {
            return SecurityHandler.create(this.buildPrototype());
        }

    }

}
