/*
 * Copyright (c) 2023 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.webserver.security;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Supplier;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Errors;
import io.helidon.common.Generated;
import io.helidon.common.config.Config;
import io.helidon.security.Security;

/**
 * Configuration of security feature fow webserver.
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.processor.BlueprintProcessor", trigger = "io.helidon.webserver.security.SecurityFeatureConfigBlueprint")
public interface SecurityFeatureConfig extends SecurityFeatureConfigBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static SecurityFeatureConfig.Builder builder() {
        return new SecurityFeatureConfig.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static SecurityFeatureConfig.Builder builder(SecurityFeatureConfig instance) {
        return SecurityFeatureConfig.builder().from(instance);
    }

    /**
     * Create a new instance from configuration.
     *
     * @param config used to configure the new instance
     * @return a new instance configured from configuration
     */
    static SecurityFeatureConfig create(Config config) {
        return SecurityFeatureConfig.builder().config(config).buildPrototype();
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static SecurityFeatureConfig create() {
        return SecurityFeatureConfig.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link SecurityFeature}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends SecurityFeatureConfig.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends SecurityFeatureConfig> implements Prototype.ConfiguredBuilder<BUILDER, PROTOTYPE> {

        private final List<PathsConfig> paths = new ArrayList<>();
        private Config config;
        private double weight = 800.0;
        private Security security;
        private SecurityHandler defaults = SecurityHandler.create();
        private String name = "security";

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(SecurityFeatureConfig prototype) {
            weight(prototype.weight());
            defaults(prototype.defaults());
            addPaths(prototype.paths());
            security(prototype.security());
            name(prototype.name());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(SecurityFeatureConfig.BuilderBase<?, ?> builder) {
            weight(builder.weight());
            defaults(builder.defaults());
            addPaths(builder.paths());
            builder.security().ifPresent(this::security);
            name(builder.name());
            return self();
        }

        /**
         * Update builder from configuration (node of this type).
         * If a value is present in configuration, it would override currently configured values.
         *
         * @param config configuration instance used to obtain values to update this builder
         * @return updated builder instance
         */
        @Override
        public BUILDER config(Config config) {
            Objects.requireNonNull(config);
            this.config = config;
            config.get("weight").as(Double.class).ifPresent(this::weight);
            config.get("defaults").map(SecurityHandlerConfig::create).ifPresent(this::defaults);
            config.get("paths").mapList(PathsConfig::create).ifPresent(this::paths);
            config.get("security").map(Security::create).ifPresent(this::security);
            return self();
        }

        /**
         * Weight of the security feature. Value is:
         * {@value io.helidon.webserver.security.SecurityFeature#WEIGHT}.
         *
         * @param weight weight of the feature
         * @return updated builder instance
         * @see #weight()
         */
        public BUILDER weight(double weight) {
            this.weight = weight;
            return self();
        }

        /**
         * The default security handler.
         *
         * @param defaults security handler defaults
         * @return updated builder instance
         * @see #defaults()
         */
        public BUILDER defaults(SecurityHandler defaults) {
            Objects.requireNonNull(defaults);
            this.defaults = defaults;
            return self();
        }

        /**
         * The default security handler.
         *
         * @param defaultsConfig security handler defaults
         * @return updated builder instance
         * @see #defaults()
         */
        public BUILDER defaults(SecurityHandlerConfig defaultsConfig) {
            Objects.requireNonNull(defaultsConfig);
            this.defaults = SecurityHandler.create(defaultsConfig);
            return self();
        }

        /**
         * The default security handler.
         *
         * @param consumer consumer of builder for
         *                 security handler defaults
         * @return updated builder instance
         * @see #defaults()
         */
        public BUILDER defaults(Consumer<SecurityHandlerConfig.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = SecurityHandler.builder();
            consumer.accept(builder);
            this.defaults(builder.build());
            return self();
        }

        /**
         * The default security handler.
         *
         * @param supplier supplier of
         *                 security handler defaults
         * @return updated builder instance
         * @see #defaults()
         */
        public BUILDER defaults(Supplier<? extends SecurityHandler> supplier) {
            Objects.requireNonNull(supplier);
            this.defaults(supplier.get());
            return self();
        }

        /**
         * Configuration for webserver paths.
         *
         * @param paths path configuration
         * @return updated builder instance
         * @see #paths()
         */
        public BUILDER paths(List<? extends PathsConfig> paths) {
            Objects.requireNonNull(paths);
            this.paths.clear();
            this.paths.addAll(paths);
            return self();
        }

        /**
         * Configuration for webserver paths.
         *
         * @param paths path configuration
         * @return updated builder instance
         * @see #paths()
         */
        public BUILDER addPaths(List<? extends PathsConfig> paths) {
            Objects.requireNonNull(paths);
            this.paths.addAll(paths);
            return self();
        }

        /**
         * Configuration for webserver paths.
         *
         * @param path path configuration
         * @return updated builder instance
         * @see #paths()
         */
        public BUILDER addPath(PathsConfig path) {
            Objects.requireNonNull(path);
            this.paths.add(path);
            return self();
        }

        /**
         * Configuration for webserver paths.
         *
         * @param consumer path configuration
         * @return updated builder instance
         * @see #paths()
         */
        public BUILDER addPath(Consumer<PathsConfig.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = PathsConfig.builder();
            consumer.accept(builder);
            this.paths.add(builder.build());
            return self();
        }

        /**
         * Security associated with this feature.
         * If not specified here, the feature uses security registered with
         * {@link io.helidon.common.context.Contexts#globalContext()}, if not found, it creates a new
         * instance from root of configuration (using {@code security} key).
         * <p>
         * This configuration allows usage of a different security instance for a specific security feature setup.
         *
         * @param security security instance to be used to handle security in this feature configuration
         * @return updated builder instance
         * @see #security()
         */
        public BUILDER security(Security security) {
            Objects.requireNonNull(security);
            this.security = security;
            return self();
        }

        /**
         * Security associated with this feature.
         * If not specified here, the feature uses security registered with
         * {@link io.helidon.common.context.Contexts#globalContext()}, if not found, it creates a new
         * instance from root of configuration (using {@code security} key).
         * <p>
         * This configuration allows usage of a different security instance for a specific security feature setup.
         *
         * @param consumer consumer of builder for
         *                 security instance to be used to handle security in this feature configuration
         * @return updated builder instance
         * @see #security()
         */
        public BUILDER security(Consumer<Security.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = Security.builder();
            consumer.accept(builder);
            this.security(builder.build());
            return self();
        }

        /**
         * Security associated with this feature.
         * If not specified here, the feature uses security registered with
         * {@link io.helidon.common.context.Contexts#globalContext()}, if not found, it creates a new
         * instance from root of configuration (using {@code security} key).
         * <p>
         * This configuration allows usage of a different security instance for a specific security feature setup.
         *
         * @param supplier supplier of
         *                 security instance to be used to handle security in this feature configuration
         * @return updated builder instance
         * @see #security()
         */
        public BUILDER security(Supplier<? extends Security> supplier) {
            Objects.requireNonNull(supplier);
            this.security(supplier.get());
            return self();
        }

        /**
         * Name of this instance.
         *
         * @param name instance name
         * @return updated builder instance
         * @see #name()
         */
        public BUILDER name(String name) {
            Objects.requireNonNull(name);
            this.name = name;
            return self();
        }

        /**
         * Weight of the security feature. Value is:
         * {@value io.helidon.webserver.security.SecurityFeature#WEIGHT}.
         *
         * @return the weight
         */
        public double weight() {
            return weight;
        }

        /**
         * The default security handler.
         *
         * @return the defaults
         */
        public SecurityHandler defaults() {
            return defaults;
        }

        /**
         * Configuration for webserver paths.
         *
         * @return the paths
         */
        public List<PathsConfig> paths() {
            return paths;
        }

        /**
         * Security associated with this feature.
         * If not specified here, the feature uses security registered with
         * {@link io.helidon.common.context.Contexts#globalContext()}, if not found, it creates a new
         * instance from root of configuration (using {@code security} key).
         * <p>
         * This configuration allows usage of a different security instance for a specific security feature setup.
         *
         * @return the security
         */
        public Optional<Security> security() {
            return Optional.ofNullable(security);
        }

        /**
         * Name of this instance.
         *
         * @return the name
         */
        public String name() {
            return name;
        }

        /**
         * If this instance was configured, this would be the config instance used.
         *
         * @return config node used to configure this builder, or empty if not configured
         */
        public Optional<Config> config() {
            return Optional.ofNullable(config);
        }

        @Override
        public String toString() {
            return "SecurityFeatureConfigBuilder{"
                    + "weight=" + weight + ","
                    + "defaults=" + defaults + ","
                    + "paths=" + paths + ","
                    + "security=" + security + ","
                    + "name=" + name
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
            new SecurityConfigSupport.SecurityFeatureConfigDecorator().decorate(this);
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
            Errors.Collector collector = Errors.collector();
            if (security == null) {
                collector.fatal(getClass(), "Property \"security\" must not be null, but not set");
            }
            collector.collect().checkValid();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class SecurityFeatureConfigImpl implements SecurityFeatureConfig, Supplier<SecurityFeature> {

            private final double weight;
            private final List<PathsConfig> paths;
            private final Security security;
            private final SecurityHandler defaults;
            private final String name;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected SecurityFeatureConfigImpl(SecurityFeatureConfig.BuilderBase<?, ?> builder) {
                this.weight = builder.weight();
                this.defaults = builder.defaults();
                this.paths = List.copyOf(builder.paths());
                this.security = builder.security().get();
                this.name = builder.name();
            }

            @Override
            public SecurityFeature build() {
                return SecurityFeature.create(this);
            }

            @Override
            public SecurityFeature get() {
                return build();
            }

            @Override
            public double weight() {
                return weight;
            }

            @Override
            public SecurityHandler defaults() {
                return defaults;
            }

            @Override
            public List<PathsConfig> paths() {
                return paths;
            }

            @Override
            public Security security() {
                return security;
            }

            @Override
            public String name() {
                return name;
            }

            @Override
            public String toString() {
                return "SecurityFeatureConfig{"
                        + "weight=" + weight + ","
                        + "defaults=" + defaults + ","
                        + "paths=" + paths + ","
                        + "security=" + security + ","
                        + "name=" + name
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof SecurityFeatureConfig other)) {
                    return false;
                }
                return weight == other.weight()
                        && Objects.equals(defaults, other.defaults())
                        && Objects.equals(paths, other.paths())
                        && Objects.equals(security, other.security())
                        && Objects.equals(name, other.name());
            }

            @Override
            public int hashCode() {
                return Objects.hash(weight, defaults, paths, security, name);
            }

        }

    }

    /**
     * Fluent API builder for {@link SecurityFeature}.
     */
    class Builder extends SecurityFeatureConfig.BuilderBase<SecurityFeatureConfig.Builder, SecurityFeatureConfig> implements io.helidon.common.Builder<SecurityFeatureConfig.Builder, SecurityFeature> {

        private Builder() {
        }

        @Override
        public SecurityFeatureConfig buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new SecurityFeatureConfigImpl(this);
        }

        @Override
        public SecurityFeature build() {
            return SecurityFeature.create(this.buildPrototype());
        }

    }

}
