/*
 * Copyright (c) 2023 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.webserver.cors;

import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Supplier;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Generated;
import io.helidon.common.config.Config;

/**
 * Configuration of CORS feature.
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.processor.BlueprintProcessor", trigger = "io.helidon.webserver.cors.CorsConfigBlueprint")
public interface CorsConfig extends CorsConfigBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static CorsConfig.Builder builder() {
        return new CorsConfig.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static CorsConfig.Builder builder(CorsConfig instance) {
        return CorsConfig.builder().from(instance);
    }

    /**
     * Create a new instance from configuration.
     *
     * @param config used to configure the new instance
     * @return a new instance configured from configuration
     */
    static CorsConfig create(Config config) {
        return CorsConfig.builder().config(config).buildPrototype();
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static CorsConfig create() {
        return CorsConfig.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link CorsFeature}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends CorsConfig.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends CorsConfig> implements Prototype.ConfiguredBuilder<BUILDER, PROTOTYPE> {

        private final Set<String> sockets = new LinkedHashSet<>();
        private boolean enabled = true;
        private Config config;
        private double weight = 950.0;
        private String name = "cors";

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(CorsConfig prototype) {
            weight(prototype.weight());
            addSockets(prototype.sockets());
            name(prototype.name());
            enabled(prototype.enabled());
            this.config = prototype.config().orElse(null);
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(CorsConfig.BuilderBase<?, ?> builder) {
            weight(builder.weight());
            addSockets(builder.sockets());
            name(builder.name());
            enabled(builder.enabled());
            builder.config().ifPresent(this::config);
            return self();
        }

        /**
         * Update builder from configuration (node of this type).
         * If a value is present in configuration, it would override currently configured values.
         *
         * @param config configuration instance used to obtain values to update this builder
         * @return updated builder instance
         */
        @Override
        public BUILDER config(Config config) {
            Objects.requireNonNull(config);
            this.config = config;
            config.get("weight").as(Double.class).ifPresent(this::weight);
            config.get("sockets").asList(String.class).map(java.util.Set::copyOf).ifPresent(this::sockets);
            config.get("enabled").as(Boolean.class).ifPresent(this::enabled);
            return self();
        }

        /**
         * Weight of the CORS feature. As it is used by other features, the default is quite high:
         * {@value CorsFeature#WEIGHT}.
         *
         * @param weight weight of the feature
         * @return updated builder instance
         * @see #weight()
         */
        public BUILDER weight(double weight) {
            this.weight = weight;
            return self();
        }

        /**
         * List of sockets to register this feature on. If empty, it would get registered on all sockets.
         *
         * @param sockets socket names to register on, defaults to empty (all available sockets)
         * @return updated builder instance
         * @see #sockets()
         */
        public BUILDER sockets(Set<? extends String> sockets) {
            Objects.requireNonNull(sockets);
            this.sockets.clear();
            this.sockets.addAll(sockets);
            return self();
        }

        /**
         * List of sockets to register this feature on. If empty, it would get registered on all sockets.
         *
         * @param sockets socket names to register on, defaults to empty (all available sockets)
         * @return updated builder instance
         * @see #sockets()
         */
        public BUILDER addSockets(Set<? extends String> sockets) {
            Objects.requireNonNull(sockets);
            this.sockets.addAll(sockets);
            return self();
        }

        /**
         * List of sockets to register this feature on. If empty, it would get registered on all sockets.
         *
         * @param socket socket names to register on, defaults to empty (all available sockets)
         * @return updated builder instance
         * @see #sockets()
         */
        public BUILDER addSocket(String socket) {
            Objects.requireNonNull(socket);
            this.sockets.add(socket);
            return self();
        }

        /**
         * Name of this instance.
         *
         * @param name instance name
         * @return updated builder instance
         * @see #name()
         */
        public BUILDER name(String name) {
            Objects.requireNonNull(name);
            this.name = name;
            return self();
        }

        /**
         * This feature can be disabled.
         *
         * @param enabled whether the feature is enabled
         * @return updated builder instance
         * @see #enabled()
         */
        public BUILDER enabled(boolean enabled) {
            this.enabled = enabled;
            return self();
        }

        /**
         * Weight of the CORS feature. As it is used by other features, the default is quite high:
         * {@value CorsFeature#WEIGHT}.
         *
         * @return the weight
         */
        public double weight() {
            return weight;
        }

        /**
         * List of sockets to register this feature on. If empty, it would get registered on all sockets.
         *
         * @return the sockets
         */
        public Set<String> sockets() {
            return sockets;
        }

        /**
         * Name of this instance.
         *
         * @return the name
         */
        public String name() {
            return name;
        }

        /**
         * This feature can be disabled.
         *
         * @return the enabled
         */
        public boolean enabled() {
            return enabled;
        }

        /**
         * If this instance was configured, this would be the config instance used.
         *
         * @return config node used to configure this builder, or empty if not configured
         */
        public Optional<Config> config() {
            return Optional.ofNullable(config);
        }

        @Override
        public String toString() {
            return "CorsConfigBuilder{"
                    + "weight=" + weight + ","
                    + "sockets=" + sockets + ","
                    + "name=" + name + ","
                    + "enabled=" + enabled + ","
                    + "config=" + config
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class CorsConfigImpl implements CorsConfig, Supplier<CorsFeature> {

            private final boolean enabled;
            private final double weight;
            private final Optional<Config> config;
            private final Set<String> sockets;
            private final String name;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected CorsConfigImpl(CorsConfig.BuilderBase<?, ?> builder) {
                this.weight = builder.weight();
                this.sockets = Collections.unmodifiableSet(new LinkedHashSet<>(builder.sockets()));
                this.name = builder.name();
                this.enabled = builder.enabled();
                this.config = builder.config();
            }

            @Override
            public CorsFeature build() {
                return CorsFeature.create(this);
            }

            @Override
            public CorsFeature get() {
                return build();
            }

            @Override
            public double weight() {
                return weight;
            }

            @Override
            public Set<String> sockets() {
                return sockets;
            }

            @Override
            public String name() {
                return name;
            }

            @Override
            public boolean enabled() {
                return enabled;
            }

            @Override
            public Optional<Config> config() {
                return config;
            }

            @Override
            public String toString() {
                return "CorsConfig{"
                        + "weight=" + weight + ","
                        + "sockets=" + sockets + ","
                        + "name=" + name + ","
                        + "enabled=" + enabled + ","
                        + "config=" + config
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof CorsConfig other)) {
                    return false;
                }
                return weight == other.weight()
                        && Objects.equals(sockets, other.sockets())
                        && Objects.equals(name, other.name())
                        && enabled == other.enabled()
                        && Objects.equals(config, other.config());
            }

            @Override
            public int hashCode() {
                return Objects.hash(weight, sockets, name, enabled, config);
            }

        }

    }

    /**
     * Fluent API builder for {@link CorsFeature}.
     */
    class Builder extends CorsConfig.BuilderBase<CorsConfig.Builder, CorsConfig> implements io.helidon.common.Builder<CorsConfig.Builder, CorsFeature> {

        private Builder() {
        }

        @Override
        public CorsConfig buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new CorsConfigImpl(this);
        }

        @Override
        public CorsFeature build() {
            return CorsFeature.create(this.buildPrototype());
        }

    }

}
