/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.webserver.cors;

import io.helidon.common.http.Http;
import io.helidon.config.Config;
import io.helidon.webserver.cors.Aggregator;
import io.helidon.webserver.cors.CorsSupportBase;
import io.helidon.webserver.cors.CrossOriginConfig;
import io.helidon.webserver.cors.LogHelper;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.function.BiFunction;
import java.util.function.Supplier;
import java.util.logging.Logger;

class CorsSupportHelper<Q, R> {
    static final int SUCCESS_RANGE = 300;
    static final String ORIGIN_DENIED = "CORS origin is denied";
    static final String ORIGIN_NOT_IN_ALLOWED_LIST = "CORS origin is not in allowed list";
    static final String METHOD_NOT_IN_ALLOWED_LIST = "CORS method is not in allowed list";
    static final String HEADERS_NOT_IN_ALLOWED_LIST = "CORS headers not in allowed list";
    static final Logger LOGGER = Logger.getLogger(CorsSupportHelper.class.getName());
    private static final Supplier<Optional<CrossOriginConfig>> EMPTY_SECONDARY_SUPPLIER = Optional::empty;
    private final String name;
    private final Aggregator aggregator;
    private final Supplier<Optional<CrossOriginConfig>> secondaryCrossOriginLookup;

    public static String normalize(String path) {
        int length = path.length();
        if (length == 0) {
            return path;
        }
        int beginIndex = path.charAt(0) == '/' ? 1 : 0;
        int endIndex = path.charAt(length - 1) == '/' ? length - 1 : length;
        return endIndex <= beginIndex ? "" : path.substring(beginIndex, endIndex);
    }

    public static Set<String> parseHeader(String header) {
        if (header == null) {
            return Collections.emptySet();
        }
        HashSet<String> result = new HashSet<String>();
        StringTokenizer tokenizer = new StringTokenizer(header, ",");
        while (tokenizer.hasMoreTokens()) {
            String value = tokenizer.nextToken().trim();
            if (value.length() <= 0) continue;
            result.add(value);
        }
        return result;
    }

    public static Set<String> parseHeader(List<String> headers) {
        if (headers == null) {
            return Collections.emptySet();
        }
        return CorsSupportHelper.parseHeader(headers.stream().reduce("", (a, b) -> a + "," + b));
    }

    public static <Q, R> CorsSupportHelper<Q, R> create() {
        return CorsSupportHelper.builder().build();
    }

    private CorsSupportHelper(Builder<Q, R> builder) {
        this.name = builder.name;
        this.aggregator = builder.aggregatorBuilder.build();
        this.secondaryCrossOriginLookup = builder.secondaryCrossOriginLookup;
    }

    public static <Q, R> Builder<Q, R> builder() {
        return new Builder();
    }

    public boolean isActive() {
        return this.aggregator.isEnabled();
    }

    public Optional<R> processRequest(CorsSupportBase.RequestAdapter<Q> requestAdapter, CorsSupportBase.ResponseAdapter<R> responseAdapter) {
        if (!this.isActive()) {
            this.decisionLog(() -> String.format("CORS ignoring request %s; processing is inactive", requestAdapter));
            requestAdapter.next();
            return Optional.empty();
        }
        RequestType requestType = this.requestType(requestAdapter);
        if (requestType == RequestType.NORMAL) {
            this.decisionLog("passing normal request through unchanged");
            return Optional.empty();
        }
        switch (requestType) {
            case PREFLIGHT: {
                return Optional.of(this.processCorsPreFlightRequest(requestAdapter, responseAdapter));
            }
            case CORS: {
                return this.processCorsRequest(requestAdapter, responseAdapter);
            }
        }
        throw new IllegalArgumentException("Unexpected value for enum RequestType");
    }

    public String toString() {
        return String.format("CorsSupportHelper{name='%s', isActive=%s, crossOriginConfigs=%s, secondaryCrossOriginLookup=%s}", this.name, this.isActive(), this.aggregator, this.secondaryCrossOriginLookup == EMPTY_SECONDARY_SUPPLIER ? "(not set)" : "(set)");
    }

    public void prepareResponse(CorsSupportBase.RequestAdapter<Q> requestAdapter, CorsSupportBase.ResponseAdapter<R> responseAdapter) {
        if (!this.isActive()) {
            this.decisionLog(() -> String.format("CORS ignoring request %s; CORS processing is inactive", requestAdapter));
            return;
        }
        RequestType requestType = this.requestType(requestAdapter, true);
        if (requestType == RequestType.CORS) {
            CrossOriginConfig crossOrigin = responseAdapter.status() == Http.Status.NOT_FOUND_404.code() ? CrossOriginConfig.builder().allowOrigins(requestAdapter.firstHeader("Origin").orElse("*")).allowMethods(requestAdapter.method()).build() : this.aggregator.lookupCrossOrigin(requestAdapter.path(), requestAdapter.method(), this.secondaryCrossOriginLookup).orElseThrow(() -> new IllegalArgumentException("Could not locate expected CORS information while preparing response to request " + requestAdapter));
            this.addCorsHeadersToResponse(crossOrigin, requestAdapter, responseAdapter);
        }
    }

    RequestType requestType(CorsSupportBase.RequestAdapter<Q> requestAdapter, boolean silent) {
        if (this.isRequestTypeNormal(requestAdapter, silent)) {
            return RequestType.NORMAL;
        }
        return this.inferCORSRequestType(requestAdapter, silent);
    }

    RequestType requestType(CorsSupportBase.RequestAdapter<Q> requestAdapter) {
        return this.requestType(requestAdapter, false);
    }

    Aggregator aggregator() {
        return this.aggregator;
    }

    private boolean isRequestTypeNormal(CorsSupportBase.RequestAdapter<Q> requestAdapter, boolean silent) {
        Optional<String> originOpt = requestAdapter.firstHeader("Origin");
        Optional<String> hostOpt = requestAdapter.firstHeader("Host");
        boolean result = originOpt.isEmpty() || hostOpt.isPresent() && originOpt.get().contains("://" + hostOpt.get());
        LogHelper.logIsRequestTypeNormal(result, silent, requestAdapter, originOpt, hostOpt);
        return result;
    }

    private RequestType inferCORSRequestType(CorsSupportBase.RequestAdapter<Q> requestAdapter, boolean silent) {
        String methodName = requestAdapter.method();
        boolean isMethodOPTION = methodName.equalsIgnoreCase(Http.Method.OPTIONS.name());
        boolean requestContainsAccessControlRequestMethodHeader = requestAdapter.headerContainsKey("Access-Control-Request-Method");
        RequestType result = isMethodOPTION && requestContainsAccessControlRequestMethodHeader ? RequestType.PREFLIGHT : RequestType.CORS;
        LogHelper.logInferRequestType(result, silent, requestAdapter, methodName, requestContainsAccessControlRequestMethodHeader);
        return result;
    }

    Optional<R> processCorsRequest(CorsSupportBase.RequestAdapter<Q> requestAdapter, CorsSupportBase.ResponseAdapter<R> responseAdapter) {
        Optional<CrossOriginConfig> crossOriginOpt = this.aggregator.lookupCrossOrigin(requestAdapter.path(), requestAdapter.method(), this.secondaryCrossOriginLookup);
        if (crossOriginOpt.isEmpty()) {
            return Optional.of(this.forbid(requestAdapter, responseAdapter, ORIGIN_DENIED, () -> "no matching CORS configuration for path " + requestAdapter.path()));
        }
        CrossOriginConfig crossOriginConfig = crossOriginOpt.get();
        List<String> allowedOrigins = Arrays.asList(crossOriginConfig.allowOrigins());
        Optional<String> originOpt = requestAdapter.firstHeader("Origin");
        if (!allowedOrigins.contains("*") && !CorsSupportHelper.contains(originOpt, allowedOrigins, String::equals)) {
            return Optional.of(this.forbid(requestAdapter, responseAdapter, ORIGIN_NOT_IN_ALLOWED_LIST, () -> String.format("actual: %s, allowed: %s", originOpt.orElse("(MISSING)"), allowedOrigins)));
        }
        return Optional.empty();
    }

    void addCorsHeadersToResponse(CrossOriginConfig crossOrigin, CorsSupportBase.RequestAdapter<Q> requestAdapter, CorsSupportBase.ResponseAdapter<R> responseAdapter) {
        String origin = requestAdapter.firstHeader("Origin").orElseThrow(CorsSupportHelper.noRequiredHeaderExcFactory("Origin"));
        if (crossOrigin.allowCredentials()) {
            new LogHelper.Headers().add("Access-Control-Allow-Credentials", "true").add("Access-Control-Allow-Origin", origin).add("Vary", "Origin").setAndLog(responseAdapter::header, "allow-credentials was set in CORS config");
        } else {
            List<String> allowedOrigins = Arrays.asList(crossOrigin.allowOrigins());
            new LogHelper.Headers().add("Access-Control-Allow-Origin", allowedOrigins.contains("*") ? "*" : origin).add("Vary", "Origin").setAndLog(responseAdapter::header, "allow-credentials was not set in CORS config");
        }
        LogHelper.Headers headers = new LogHelper.Headers();
        CorsSupportHelper.formatHeader(crossOrigin.exposeHeaders()).ifPresent(h -> headers.add("Access-Control-Expose-Headers", h));
        headers.setAndLog(responseAdapter::header, "expose-headers was set in CORS config");
    }

    R processCorsPreFlightRequest(CorsSupportBase.RequestAdapter<Q> requestAdapter, CorsSupportBase.ResponseAdapter<R> responseAdapter) {
        Optional<String> originOpt = requestAdapter.firstHeader("Origin");
        if (originOpt.isEmpty()) {
            return this.forbid(requestAdapter, responseAdapter, CorsSupportHelper.noRequiredHeader("Origin"));
        }
        String requestedMethod = requestAdapter.firstHeader("Access-Control-Request-Method").get();
        Optional<CrossOriginConfig> crossOriginOpt = this.aggregator.lookupCrossOrigin(requestAdapter.path(), requestedMethod, this.secondaryCrossOriginLookup);
        if (crossOriginOpt.isEmpty()) {
            return this.forbid(requestAdapter, responseAdapter, ORIGIN_DENIED, () -> String.format("no matching CORS configuration for path %s and requested method %s", requestAdapter.path(), requestedMethod));
        }
        CrossOriginConfig crossOrigin = crossOriginOpt.get();
        List<String> allowedOrigins = Arrays.asList(crossOrigin.allowOrigins());
        if (!allowedOrigins.contains("*") && !CorsSupportHelper.contains(originOpt, allowedOrigins, String::equals)) {
            return this.forbid(requestAdapter, responseAdapter, ORIGIN_NOT_IN_ALLOWED_LIST, () -> "actual origin: " + (String)originOpt.get() + ", allowedOrigins: " + allowedOrigins);
        }
        List<String> allowedMethods = Arrays.asList(crossOrigin.allowMethods());
        if (!allowedMethods.contains("*") && !CorsSupportHelper.contains(requestedMethod, allowedMethods, String::equalsIgnoreCase)) {
            return this.forbid(requestAdapter, responseAdapter, METHOD_NOT_IN_ALLOWED_LIST, () -> String.format("header %s requested method %s but allowedMethods is %s", "Access-Control-Request-Method", requestedMethod, allowedMethods));
        }
        Set<String> requestHeaders = CorsSupportHelper.parseHeader(requestAdapter.allHeaders("Access-Control-Request-Headers"));
        List<String> allowedHeaders = Arrays.asList(crossOrigin.allowHeaders());
        if (!allowedHeaders.contains("*") && !CorsSupportHelper.contains(requestHeaders, allowedHeaders)) {
            return this.forbid(requestAdapter, responseAdapter, HEADERS_NOT_IN_ALLOWED_LIST, () -> String.format("requested headers %s incompatible with allowed headers %s", requestHeaders, allowedHeaders));
        }
        LogHelper.Headers headers = new LogHelper.Headers().add("Access-Control-Allow-Origin", originOpt.get());
        if (crossOrigin.allowCredentials()) {
            headers.add("Access-Control-Allow-Credentials", "true", "allowCredentials config was set");
        }
        headers.add("Access-Control-Allow-Methods", requestedMethod);
        CorsSupportHelper.formatHeader(requestHeaders.toArray()).ifPresent(h -> headers.add("Access-Control-Allow-Headers", h));
        long maxAgeSeconds = crossOrigin.maxAgeSeconds();
        if (maxAgeSeconds > 0L) {
            headers.add("Access-Control-Max-Age", maxAgeSeconds, "maxAgeSeconds > 0");
        }
        headers.setAndLog(responseAdapter::header, "headers set on preflight request");
        return responseAdapter.ok();
    }

    static <T> Optional<String> formatHeader(T[] array) {
        if (array == null || array.length == 0) {
            return Optional.empty();
        }
        int i = 0;
        StringBuilder builder = new StringBuilder();
        while (true) {
            builder.append(array[i++].toString());
            if (i == array.length) break;
            builder.append(", ");
        }
        return Optional.of(builder.toString());
    }

    static boolean contains(Optional<String> item, Collection<String> collection, BiFunction<String, String, Boolean> eq) {
        return item.isPresent() && CorsSupportHelper.contains(item.get(), collection, eq);
    }

    static boolean contains(String item, Collection<String> collection, BiFunction<String, String, Boolean> eq) {
        for (String s : collection) {
            if (!eq.apply(item, s).booleanValue()) continue;
            return true;
        }
        return false;
    }

    static boolean contains(Collection<String> left, Collection<String> right) {
        for (String s : left) {
            if (CorsSupportHelper.contains(s, right, String::equalsIgnoreCase)) continue;
            return false;
        }
        return true;
    }

    private static Supplier<IllegalArgumentException> noRequiredHeaderExcFactory(String header) {
        return () -> new IllegalArgumentException(CorsSupportHelper.noRequiredHeader(header));
    }

    private static String noRequiredHeader(String header) {
        return "CORS request does not have required header " + header;
    }

    private R forbid(CorsSupportBase.RequestAdapter<Q> requestAdapter, CorsSupportBase.ResponseAdapter<R> responseAdapter, String reason) {
        return this.forbid(requestAdapter, responseAdapter, reason, null);
    }

    private R forbid(CorsSupportBase.RequestAdapter<Q> requestAdapter, CorsSupportBase.ResponseAdapter<R> responseAdapter, String publicReason, Supplier<String> privateExplanation) {
        this.decisionLog(() -> String.format("CORS denying request %s: %s", requestAdapter, publicReason + (String)(privateExplanation == null ? "" : "; " + (String)privateExplanation.get())));
        return responseAdapter.forbidden(publicReason);
    }

    private void decisionLog(Supplier<String> messageSupplier) {
        if (LOGGER.isLoggable(LogHelper.DECISION_LEVEL)) {
            this.decisionLog(messageSupplier.get());
        }
    }

    private void decisionLog(String message) {
        LOGGER.log(LogHelper.DECISION_LEVEL, () -> String.format("CORS:%s %s", this.name, message));
    }

    public static class Builder<Q, R>
    implements io.helidon.common.Builder<CorsSupportHelper<Q, R>> {
        private Supplier<Optional<CrossOriginConfig>> secondaryCrossOriginLookup = EMPTY_SECONDARY_SUPPLIER;
        private final Aggregator.Builder aggregatorBuilder = Aggregator.builder();
        private String name;
        private boolean requestDefaultBehaviorIfNone;

        public Builder<Q, R> secondaryLookupSupplier(Supplier<Optional<CrossOriginConfig>> secondaryLookup) {
            this.secondaryCrossOriginLookup = secondaryLookup;
            return this;
        }

        public Builder<Q, R> config(Config config) {
            this.aggregatorBuilder.config(config);
            return this;
        }

        public Builder<Q, R> mappedConfig(Config config) {
            this.aggregatorBuilder.mappedConfig(config);
            return this;
        }

        public Builder<Q, R> name(String name) {
            Objects.requireNonNull(name, "CORS support name is optional but cannot be null");
            this.name = name;
            return this;
        }

        public Builder<Q, R> requestDefaultBehaviorIfNone() {
            this.requestDefaultBehaviorIfNone = true;
            return this;
        }

        private boolean shouldRequestDefaultBehavior() {
            return this.requestDefaultBehaviorIfNone && (this.secondaryCrossOriginLookup == null || this.secondaryCrossOriginLookup == EMPTY_SECONDARY_SUPPLIER);
        }

        public CorsSupportHelper<Q, R> build() {
            if (this.shouldRequestDefaultBehavior()) {
                this.aggregatorBuilder.requestDefaultBehaviorIfNone();
            }
            CorsSupportHelper result = new CorsSupportHelper(this);
            LOGGER.config(() -> String.format("CorsSupportHelper configured as: %s", result.toString()));
            return result;
        }

        Aggregator.Builder aggregatorBuilder() {
            return this.aggregatorBuilder;
        }
    }

    public static enum RequestType {
        NORMAL,
        CORS,
        PREFLIGHT;

    }
}

