/*
 * Copyright (c) 2024 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.webserver.context;

import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Supplier;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Generated;
import io.helidon.common.config.Config;

/**
 * Configuration of context feature.
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.codegen.BuilderCodegen", trigger = "io.helidon.webserver.context.ContextFeatureConfigBlueprint")
public interface ContextFeatureConfig extends ContextFeatureConfigBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static ContextFeatureConfig.Builder builder() {
        return new ContextFeatureConfig.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static ContextFeatureConfig.Builder builder(ContextFeatureConfig instance) {
        return ContextFeatureConfig.builder().from(instance);
    }

    /**
     * Create a new instance from configuration.
     *
     * @param config used to configure the new instance
     * @return a new instance configured from configuration
     */
    static ContextFeatureConfig create(Config config) {
        return ContextFeatureConfig.builder().config(config).buildPrototype();
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static ContextFeatureConfig create() {
        return ContextFeatureConfig.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link ContextFeature}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends ContextFeatureConfig.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends ContextFeatureConfig> implements Prototype.ConfiguredBuilder<BUILDER, PROTOTYPE> {

        private final Set<String> sockets = new LinkedHashSet<>();
        private Config config;
        private double weight = 1100.0;
        private String name = "context";

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance. This method disables automatic service discovery.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(ContextFeatureConfig prototype) {
            weight(prototype.weight());
            addSockets(prototype.sockets());
            name(prototype.name());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(ContextFeatureConfig.BuilderBase<?, ?> builder) {
            weight(builder.weight());
            addSockets(builder.sockets);
            name(builder.name());
            return self();
        }

        /**
         * Update builder from configuration (node of this type).
         * If a value is present in configuration, it would override currently configured values.
         *
         * @param config configuration instance used to obtain values to update this builder
         * @return updated builder instance
         */
        @Override
        public BUILDER config(Config config) {
            Objects.requireNonNull(config);
            this.config = config;
            config.get("weight").as(Double.class).ifPresent(this::weight);
            config.get("sockets").asList(String.class).map(java.util.Set::copyOf).ifPresent(this::sockets);
            return self();
        }

        /**
         * Weight of the context feature. As it is used by other features, the default is quite high:
         * {@value io.helidon.webserver.context.ContextFeature#WEIGHT}.
         *
         * @param weight weight of the feature
         * @return updated builder instance
         * @see #weight()
         */
        public BUILDER weight(double weight) {
            this.weight = weight;
            return self();
        }

        /**
         * List of sockets to register this feature on. If empty, it would get registered on all sockets.
         *
         * @param sockets socket names to register on, defaults to empty (all available sockets)
         * @return updated builder instance
         * @see #sockets()
         */
        public BUILDER sockets(Set<? extends String> sockets) {
            Objects.requireNonNull(sockets);
            this.sockets.clear();
            this.sockets.addAll(sockets);
            return self();
        }

        /**
         * List of sockets to register this feature on. If empty, it would get registered on all sockets.
         *
         * @param sockets socket names to register on, defaults to empty (all available sockets)
         * @return updated builder instance
         * @see #sockets()
         */
        public BUILDER addSockets(Set<? extends String> sockets) {
            Objects.requireNonNull(sockets);
            this.sockets.addAll(sockets);
            return self();
        }

        /**
         * Name of this instance.
         *
         * @param name instance name
         * @return updated builder instance
         * @see #name()
         */
        public BUILDER name(String name) {
            Objects.requireNonNull(name);
            this.name = name;
            return self();
        }

        /**
         * Weight of the context feature. As it is used by other features, the default is quite high:
         * {@value io.helidon.webserver.context.ContextFeature#WEIGHT}.
         *
         * @return the weight
         */
        public double weight() {
            return weight;
        }

        /**
         * List of sockets to register this feature on. If empty, it would get registered on all sockets.
         *
         * @return the sockets
         */
        public Set<String> sockets() {
            return sockets;
        }

        /**
         * Name of this instance.
         *
         * @return the name
         */
        public String name() {
            return name;
        }

        /**
         * If this instance was configured, this would be the config instance used.
         *
         * @return config node used to configure this builder, or empty if not configured
         */
        public Optional<Config> config() {
            return Optional.ofNullable(config);
        }

        @Override
        public String toString() {
            return "ContextFeatureConfigBuilder{"
                    + "weight=" + weight + ","
                    + "sockets=" + sockets + ","
                    + "name=" + name
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class ContextFeatureConfigImpl implements ContextFeatureConfig, Supplier<ContextFeature> {

            private final double weight;
            private final Set<String> sockets;
            private final String name;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected ContextFeatureConfigImpl(ContextFeatureConfig.BuilderBase<?, ?> builder) {
                this.weight = builder.weight();
                this.sockets = Collections.unmodifiableSet(new LinkedHashSet<>(builder.sockets()));
                this.name = builder.name();
            }

            @Override
            public ContextFeature build() {
                return ContextFeature.create(this);
            }

            @Override
            public ContextFeature get() {
                return build();
            }

            @Override
            public double weight() {
                return weight;
            }

            @Override
            public Set<String> sockets() {
                return sockets;
            }

            @Override
            public String name() {
                return name;
            }

            @Override
            public String toString() {
                return "ContextFeatureConfig{"
                        + "weight=" + weight + ","
                        + "sockets=" + sockets + ","
                        + "name=" + name
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof ContextFeatureConfig other)) {
                    return false;
                }
                return weight == other.weight()
                    && Objects.equals(sockets, other.sockets())
                    && Objects.equals(name, other.name());
            }

            @Override
            public int hashCode() {
                return Objects.hash(weight, sockets, name);
            }

        }

    }

    /**
     * Fluent API builder for {@link ContextFeature}.
     */
    class Builder extends ContextFeatureConfig.BuilderBase<ContextFeatureConfig.Builder, ContextFeatureConfig> implements io.helidon.common.Builder<ContextFeatureConfig.Builder, ContextFeature> {

        private Builder() {
        }

        @Override
        public ContextFeatureConfig buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new ContextFeatureConfigImpl(this);
        }

        @Override
        public ContextFeature build() {
            return ContextFeature.create(this.buildPrototype());
        }

    }

}
