/*
 * Copyright (c) 2024 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.webserver.accesslog;

import java.time.Clock;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Supplier;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Generated;
import io.helidon.common.config.Config;

/**
 * Configuration of access log feature.
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.codegen.BuilderCodegen", trigger = "io.helidon.webserver.accesslog.AccessLogConfigBlueprint")
public interface AccessLogConfig extends AccessLogConfigBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static AccessLogConfig.Builder builder() {
        return new AccessLogConfig.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static AccessLogConfig.Builder builder(AccessLogConfig instance) {
        return AccessLogConfig.builder().from(instance);
    }

    /**
     * Create a new instance from configuration.
     *
     * @param config used to configure the new instance
     * @return a new instance configured from configuration
     */
    static AccessLogConfig create(Config config) {
        return AccessLogConfig.builder().config(config).buildPrototype();
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static AccessLogConfig create() {
        return AccessLogConfig.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link AccessLogFeature}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends AccessLogConfig.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends AccessLogConfig> implements Prototype.ConfiguredBuilder<BUILDER, PROTOTYPE> {

        private final List<AccessLogEntry> entries = new ArrayList<>();
        private final Set<String> sockets = new LinkedHashSet<>();
        private boolean enabled = true;
        private Clock clock = Clock.systemDefaultZone();
        private Config config;
        private double weight = 1000.0;
        private String format;
        private String loggerName = "io.helidon.webserver.AccessLog";
        private String name = "access-log";

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(AccessLogConfig prototype) {
            addEntries(prototype.entries());
            clock(prototype.clock());
            loggerName(prototype.loggerName());
            weight(prototype.weight());
            addSockets(prototype.sockets());
            name(prototype.name());
            format(prototype.format());
            enabled(prototype.enabled());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(AccessLogConfig.BuilderBase<?, ?> builder) {
            addEntries(builder.entries());
            clock(builder.clock());
            loggerName(builder.loggerName());
            weight(builder.weight());
            addSockets(builder.sockets());
            name(builder.name());
            builder.format().ifPresent(this::format);
            enabled(builder.enabled());
            return self();
        }

        /**
         * Use default log format.
         * <p>
         * Clears the current log entries and replaces them with default log format.
         * <p>
         * Helidon log format uses the following log entries (in this order):
         * <ul>
         *     <li>{@link HostLogEntry}</li>
         *     <li>{@link UserLogEntry}</li>
         *     <li>{@link TimestampLogEntry}</li>
         *     <li>{@link RequestLineLogEntry}</li>
         *     <li>{@link StatusLogEntry}</li>
         *     <li>{@link SizeLogEntry}</li>
         *     <li>{@link TimeTakenLogEntry} configured for
         *          {@link java.util.concurrent.TimeUnit#MICROSECONDS}</li>
         * </ul>
         *
         * @return updated builder instance
         */
        public BUILDER defaultLogFormat() {
            AccessLogConfigSupport.CustomMethods.defaultLogFormat(this);
            return self();
        }

        /**
         * Use {@code common} log format.
         * <p>
         * Clears the current log entries and replaces them with {@code common} log format.
         * <p>
         * {@code common} log format uses the following log entries (in this order):
         * <ul>
         *     <li>{@link HostLogEntry}</li>
         *     <li>{@link UserIdLogEntry}</li>
         *     <li>{@link UserLogEntry}</li>
         *     <li>{@link TimestampLogEntry}</li>
         *     <li>{@link RequestLineLogEntry}</li>
         *     <li>{@link StatusLogEntry}</li>
         *     <li>{@link SizeLogEntry}</li>
         * </ul>
         *
         * @return updated builder instance
         */
        public BUILDER commonLogFormat() {
            AccessLogConfigSupport.CustomMethods.commonLogFormat(this);
            return self();
        }

        /**
         * Update builder from configuration (node of this type).
         * If a value is present in configuration, it would override currently configured values.
         *
         * @param config configuration instance used to obtain values to update this builder
         * @return updated builder instance
         */
        @Override
        public BUILDER config(Config config) {
            Objects.requireNonNull(config);
            this.config = config;
            config.get("logger-name").as(String.class).ifPresent(this::loggerName);
            config.get("weight").as(Double.class).ifPresent(this::weight);
            config.get("sockets").asList(String.class).map(java.util.Set::copyOf).ifPresent(this::sockets);
            config.get("format").as(String.class).ifPresent(this::format);
            config.get("enabled").as(Boolean.class).ifPresent(this::enabled);
            return self();
        }

        /**
         * Configured log entries.
         * If both entries and {@link #format()} are defined, {@link #format()} is used.
         *
         * @param entries list of access log entries
         * @return updated builder instance
         * @see #entries()
         */
        public BUILDER entries(List<? extends AccessLogEntry> entries) {
            Objects.requireNonNull(entries);
            this.entries.clear();
            this.entries.addAll(entries);
            return self();
        }

        /**
         * Configured log entries.
         * If both entries and {@link #format()} are defined, {@link #format()} is used.
         *
         * @param entries list of access log entries
         * @return updated builder instance
         * @see #entries()
         */
        public BUILDER addEntries(List<? extends AccessLogEntry> entries) {
            Objects.requireNonNull(entries);
            this.entries.addAll(entries);
            return self();
        }

        /**
         * Configured log entries.
         * If both entries and {@link #format()} are defined, {@link #format()} is used.
         *
         * @param entry list of access log entries
         * @return updated builder instance
         * @see #entries()
         */
        public BUILDER addEntry(AccessLogEntry entry) {
            Objects.requireNonNull(entry);
            this.entries.add(entry);
            return self();
        }

        /**
         * Configure an alternative clock to use, such as {@link java.time.Clock#fixed(java.time.Instant, java.time.ZoneId)}.
         * Defaults to {@link java.time.Clock#systemDefaultZone()}.
         *
         * @param clock clock to use to get current times
         * @return updated builder instance
         * @see #clock()
         */
        public BUILDER clock(Clock clock) {
            Objects.requireNonNull(clock);
            this.clock = clock;
            return self();
        }

        /**
         * Name of the logger used to obtain access log logger from {@link System#getLogger(String)}.
         * Defaults to {@value AccessLogFeature#DEFAULT_LOGGER_NAME}.
         *
         * @param loggerName name of the logger to use
         * @return updated builder instance
         * @see #loggerName()
         */
        public BUILDER loggerName(String loggerName) {
            Objects.requireNonNull(loggerName);
            this.loggerName = loggerName;
            return self();
        }

        /**
         * Weight of the access log feature. We need to log access for anything happening on the server, so weight is high:
         * {@value io.helidon.webserver.accesslog.AccessLogFeature#WEIGHT}.
         *
         * @param weight weight of the feature
         * @return updated builder instance
         * @see #weight()
         */
        public BUILDER weight(double weight) {
            this.weight = weight;
            return self();
        }

        /**
         * List of sockets to register this feature on. If empty, it would get registered on all sockets.
         * The logger used will have the expected logger with a suffix of the socket name.
         *
         * @param sockets socket names to register on, defaults to empty (all available sockets)
         * @return updated builder instance
         * @see #sockets()
         */
        public BUILDER sockets(Set<? extends String> sockets) {
            Objects.requireNonNull(sockets);
            this.sockets.clear();
            this.sockets.addAll(sockets);
            return self();
        }

        /**
         * List of sockets to register this feature on. If empty, it would get registered on all sockets.
         * The logger used will have the expected logger with a suffix of the socket name.
         *
         * @param sockets socket names to register on, defaults to empty (all available sockets)
         * @return updated builder instance
         * @see #sockets()
         */
        public BUILDER addSockets(Set<? extends String> sockets) {
            Objects.requireNonNull(sockets);
            this.sockets.addAll(sockets);
            return self();
        }

        /**
         * Name of this instance.
         *
         * @param name instance name
         * @return updated builder instance
         * @see #name()
         */
        public BUILDER name(String name) {
            Objects.requireNonNull(name);
            this.name = name;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #format()
         */
        public BUILDER clearFormat() {
            this.format = null;
            return self();
        }

        /**
         * The format for log entries (similar to the Apache {@code LogFormat}).
         * <table class="config">
         *     <caption>Log format elements</caption>
         *     <tr>
         *         <td>%h</td>
         *         <td>IP address of the remote host</td>
         *         <td>{@link HostLogEntry}</td>
         *     </tr>
         *     <tr>
         *         <td>%l</td>
         *         <td>The client identity. This is always undefined in Helidon.</td>
         *         <td>{@link UserIdLogEntry}</td>
         *     </tr>
         *     <tr>
         *         <td>%u</td>
         *         <td>User ID as asserted by Helidon Security.</td>
         *         <td>{@link UserLogEntry}</td>
         *     </tr>
         *     <tr>
         *         <td>%t</td>
         *         <td>The timestamp</td>
         *         <td>{@link TimestampLogEntry}</td>
         *     </tr>
         *     <tr>
         *         <td>%r</td>
         *         <td>The request line ({@code "GET /favicon.ico HTTP/1.0"})</td>
         *         <td>{@link RequestLineLogEntry}</td>
         *     </tr>
         *     <tr>
         *         <td>%s</td>
         *         <td>The status code returned to the client</td>
         *         <td>{@link StatusLogEntry}</td>
         *     </tr>
         *     <tr>
         *         <td>%b</td>
         *         <td>The entity size in bytes</td>
         *         <td>{@link SizeLogEntry}</td>
         *     </tr>
         *     <tr>
         *         <td>%D</td>
         *         <td>The time taken in microseconds (start of request until last byte written)</td>
         *         <td>{@link TimeTakenLogEntry}</td>
         *     </tr>
         *     <tr>
         *         <td>%T</td>
         *         <td>The time taken in seconds (start of request until last byte written), integer</td>
         *         <td>{@link TimeTakenLogEntry}</td>
         *     </tr>
         *     <tr>
         *         <td>%{header-name}i</td>
         *         <td>Value of header {@code header-name}</td>
         *         <td>{@link HeaderLogEntry}</td>
         *     </tr>
         * </table>
         *
         * @param format format string, such as {@code %h %l %u %t %r %b %{Referer}i}
         * @return updated builder instance
         * @see #format()
         */
        public BUILDER format(String format) {
            Objects.requireNonNull(format);
            this.format = format;
            return self();
        }

        /**
         * Whether this feature will be enabled.
         *
         * @param enabled whether enabled
         * @return updated builder instance
         * @see #enabled()
         */
        public BUILDER enabled(boolean enabled) {
            this.enabled = enabled;
            return self();
        }

        /**
         * Configured log entries.
         * If both entries and {@link #format()} are defined, {@link #format()} is used.
         *
         * @return the entries
         */
        public List<AccessLogEntry> entries() {
            return entries;
        }

        /**
         * Configure an alternative clock to use, such as {@link java.time.Clock#fixed(java.time.Instant, java.time.ZoneId)}.
         * Defaults to {@link java.time.Clock#systemDefaultZone()}.
         *
         * @return the clock
         */
        public Clock clock() {
            return clock;
        }

        /**
         * Name of the logger used to obtain access log logger from {@link System#getLogger(String)}.
         * Defaults to {@value AccessLogFeature#DEFAULT_LOGGER_NAME}.
         *
         * @return the logger name
         */
        public String loggerName() {
            return loggerName;
        }

        /**
         * Weight of the access log feature. We need to log access for anything happening on the server, so weight is high:
         * {@value io.helidon.webserver.accesslog.AccessLogFeature#WEIGHT}.
         *
         * @return the weight
         */
        public double weight() {
            return weight;
        }

        /**
         * List of sockets to register this feature on. If empty, it would get registered on all sockets.
         * The logger used will have the expected logger with a suffix of the socket name.
         *
         * @return the sockets
         */
        public Set<String> sockets() {
            return sockets;
        }

        /**
         * Name of this instance.
         *
         * @return the name
         */
        public String name() {
            return name;
        }

        /**
         * The format for log entries (similar to the Apache {@code LogFormat}).
         * <table class="config">
         *     <caption>Log format elements</caption>
         *     <tr>
         *         <td>%h</td>
         *         <td>IP address of the remote host</td>
         *         <td>{@link HostLogEntry}</td>
         *     </tr>
         *     <tr>
         *         <td>%l</td>
         *         <td>The client identity. This is always undefined in Helidon.</td>
         *         <td>{@link UserIdLogEntry}</td>
         *     </tr>
         *     <tr>
         *         <td>%u</td>
         *         <td>User ID as asserted by Helidon Security.</td>
         *         <td>{@link UserLogEntry}</td>
         *     </tr>
         *     <tr>
         *         <td>%t</td>
         *         <td>The timestamp</td>
         *         <td>{@link TimestampLogEntry}</td>
         *     </tr>
         *     <tr>
         *         <td>%r</td>
         *         <td>The request line ({@code "GET /favicon.ico HTTP/1.0"})</td>
         *         <td>{@link RequestLineLogEntry}</td>
         *     </tr>
         *     <tr>
         *         <td>%s</td>
         *         <td>The status code returned to the client</td>
         *         <td>{@link StatusLogEntry}</td>
         *     </tr>
         *     <tr>
         *         <td>%b</td>
         *         <td>The entity size in bytes</td>
         *         <td>{@link SizeLogEntry}</td>
         *     </tr>
         *     <tr>
         *         <td>%D</td>
         *         <td>The time taken in microseconds (start of request until last byte written)</td>
         *         <td>{@link TimeTakenLogEntry}</td>
         *     </tr>
         *     <tr>
         *         <td>%T</td>
         *         <td>The time taken in seconds (start of request until last byte written), integer</td>
         *         <td>{@link TimeTakenLogEntry}</td>
         *     </tr>
         *     <tr>
         *         <td>%{header-name}i</td>
         *         <td>Value of header {@code header-name}</td>
         *         <td>{@link HeaderLogEntry}</td>
         *     </tr>
         * </table>
         *
         * @return the format
         */
        public Optional<String> format() {
            return Optional.ofNullable(format);
        }

        /**
         * Whether this feature will be enabled.
         *
         * @return the enabled
         */
        public boolean enabled() {
            return enabled;
        }

        /**
         * If this instance was configured, this would be the config instance used.
         *
         * @return config node used to configure this builder, or empty if not configured
         */
        public Optional<Config> config() {
            return Optional.ofNullable(config);
        }

        @Override
        public String toString() {
            return "AccessLogConfigBuilder{"
                    + "entries=" + entries + ","
                    + "clock=" + clock + ","
                    + "loggerName=" + loggerName + ","
                    + "weight=" + weight + ","
                    + "sockets=" + sockets + ","
                    + "name=" + name + ","
                    + "format=" + format + ","
                    + "enabled=" + enabled
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
            new AccessLogConfigSupport.BuilderDecorator().decorate(this);
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
        }

        /**
         * The format for log entries (similar to the Apache {@code LogFormat}).
         * <table class="config">
         *     <caption>Log format elements</caption>
         *     <tr>
         *         <td>%h</td>
         *         <td>IP address of the remote host</td>
         *         <td>{@link HostLogEntry}</td>
         *     </tr>
         *     <tr>
         *         <td>%l</td>
         *         <td>The client identity. This is always undefined in Helidon.</td>
         *         <td>{@link UserIdLogEntry}</td>
         *     </tr>
         *     <tr>
         *         <td>%u</td>
         *         <td>User ID as asserted by Helidon Security.</td>
         *         <td>{@link UserLogEntry}</td>
         *     </tr>
         *     <tr>
         *         <td>%t</td>
         *         <td>The timestamp</td>
         *         <td>{@link TimestampLogEntry}</td>
         *     </tr>
         *     <tr>
         *         <td>%r</td>
         *         <td>The request line ({@code "GET /favicon.ico HTTP/1.0"})</td>
         *         <td>{@link RequestLineLogEntry}</td>
         *     </tr>
         *     <tr>
         *         <td>%s</td>
         *         <td>The status code returned to the client</td>
         *         <td>{@link StatusLogEntry}</td>
         *     </tr>
         *     <tr>
         *         <td>%b</td>
         *         <td>The entity size in bytes</td>
         *         <td>{@link SizeLogEntry}</td>
         *     </tr>
         *     <tr>
         *         <td>%D</td>
         *         <td>The time taken in microseconds (start of request until last byte written)</td>
         *         <td>{@link TimeTakenLogEntry}</td>
         *     </tr>
         *     <tr>
         *         <td>%T</td>
         *         <td>The time taken in seconds (start of request until last byte written), integer</td>
         *         <td>{@link TimeTakenLogEntry}</td>
         *     </tr>
         *     <tr>
         *         <td>%{header-name}i</td>
         *         <td>Value of header {@code header-name}</td>
         *         <td>{@link HeaderLogEntry}</td>
         *     </tr>
         * </table>
         *
         * @param format format string, such as {@code %h %l %u %t %r %b %{Referer}i}
         * @return updated builder instance
         * @see #format()
         */
        BUILDER format(Optional<String> format) {
            Objects.requireNonNull(format);
            this.format = format.map(java.lang.String.class::cast).orElse(this.format);
            return self();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class AccessLogConfigImpl implements AccessLogConfig, Supplier<AccessLogFeature> {

            private final boolean enabled;
            private final Clock clock;
            private final double weight;
            private final List<AccessLogEntry> entries;
            private final Optional<String> format;
            private final Set<String> sockets;
            private final String loggerName;
            private final String name;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected AccessLogConfigImpl(AccessLogConfig.BuilderBase<?, ?> builder) {
                this.entries = List.copyOf(builder.entries());
                this.clock = builder.clock();
                this.loggerName = builder.loggerName();
                this.weight = builder.weight();
                this.sockets = Collections.unmodifiableSet(new LinkedHashSet<>(builder.sockets()));
                this.name = builder.name();
                this.format = builder.format();
                this.enabled = builder.enabled();
            }

            @Override
            public AccessLogFeature build() {
                return AccessLogFeature.create(this);
            }

            @Override
            public AccessLogFeature get() {
                return build();
            }

            @Override
            public List<AccessLogEntry> entries() {
                return entries;
            }

            @Override
            public Clock clock() {
                return clock;
            }

            @Override
            public String loggerName() {
                return loggerName;
            }

            @Override
            public double weight() {
                return weight;
            }

            @Override
            public Set<String> sockets() {
                return sockets;
            }

            @Override
            public String name() {
                return name;
            }

            @Override
            public Optional<String> format() {
                return format;
            }

            @Override
            public boolean enabled() {
                return enabled;
            }

            @Override
            public String toString() {
                return "AccessLogConfig{"
                        + "entries=" + entries + ","
                        + "clock=" + clock + ","
                        + "loggerName=" + loggerName + ","
                        + "weight=" + weight + ","
                        + "sockets=" + sockets + ","
                        + "name=" + name + ","
                        + "format=" + format + ","
                        + "enabled=" + enabled
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof AccessLogConfig other)) {
                    return false;
                }
                return Objects.equals(entries, other.entries())
                    && Objects.equals(clock, other.clock())
                    && Objects.equals(loggerName, other.loggerName())
                    && weight == other.weight()
                    && Objects.equals(sockets, other.sockets())
                    && Objects.equals(name, other.name())
                    && Objects.equals(format, other.format())
                    && enabled == other.enabled();
            }

            @Override
            public int hashCode() {
                return Objects.hash(entries, clock, loggerName, weight, sockets, name, format, enabled);
            }

        }

    }

    /**
     * Fluent API builder for {@link AccessLogFeature}.
     */
    class Builder extends AccessLogConfig.BuilderBase<AccessLogConfig.Builder, AccessLogConfig> implements io.helidon.common.Builder<AccessLogConfig.Builder, AccessLogFeature> {

        private Builder() {
        }

        @Override
        public AccessLogConfig buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new AccessLogConfigImpl(this);
        }

        @Override
        public AccessLogFeature build() {
            return AccessLogFeature.create(this.buildPrototype());
        }

    }

}
