/*
 * Copyright (c) 2025 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.webclient.websocket;

import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Supplier;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Generated;
import io.helidon.common.config.Config;
import io.helidon.common.config.ConfigBuilderSupport;
import io.helidon.webclient.api.HttpClientConfig;

/**
 * WebSocket full webclient configuration.
 * The client configuration also contains all necessary configuration for HTTP, as WebSocket upgrades from HTTP.
 *
 * @see io.helidon.webclient.api.WebClient#client(io.helidon.webclient.spi.Protocol,
 *      io.helidon.webclient.spi.ProtocolConfig)
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.codegen.BuilderCodegen", trigger = "io.helidon.webclient.websocket.WsClientConfigBlueprint")
public interface WsClientConfig extends WsClientConfigBlueprint, Prototype.Api, HttpClientConfig {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static WsClientConfig.Builder builder() {
        return new WsClientConfig.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static WsClientConfig.Builder builder(WsClientConfig instance) {
        return WsClientConfig.builder().from(instance);
    }

    /**
     * Create a new instance from configuration.
     *
     * @param config used to configure the new instance
     * @return a new instance configured from configuration
     */
    static WsClientConfig create(Config config) {
        return WsClientConfig.builder().config(config).buildPrototype();
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static WsClientConfig create() {
        return WsClientConfig.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link WsClient}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends WsClientConfig.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends WsClientConfig> extends HttpClientConfig.BuilderBase<BUILDER, PROTOTYPE> implements ConfigBuilderSupport.ConfiguredBuilder<BUILDER, PROTOTYPE> {

        private Config config;
        private WsClientProtocolConfig protocolConfig = WsClientProtocolConfig.create();

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance. This method disables automatic service discovery.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(WsClientConfig prototype) {
            super.from(prototype);
            protocolConfig(prototype.protocolConfig());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(WsClientConfig.BuilderBase<?, ?> builder) {
            super.from(builder);
            protocolConfig(builder.protocolConfig());
            return self();
        }

        /**
         * Update builder from configuration (node of this type).
         * If a value is present in configuration, it would override currently configured values.
         *
         * @param config configuration instance used to obtain values to update this builder
         * @return updated builder instance
         */
        @Override
        public BUILDER config(Config config) {
            Objects.requireNonNull(config);
            this.config = config;
            super.config(config);
            config.get("protocol-config").map(WsClientProtocolConfig::create).ifPresent(this::protocolConfig);
            return self();
        }

        /**
         * WebSocket specific configuration.
         *
         * @param protocolConfig protocol specific configuration
         * @return updated builder instance
         * @see #protocolConfig()
         */
        public BUILDER protocolConfig(WsClientProtocolConfig protocolConfig) {
            Objects.requireNonNull(protocolConfig);
            this.protocolConfig = protocolConfig;
            return self();
        }

        /**
         * WebSocket specific configuration.
         *
         * @param consumer consumer of builder for
         *                 protocol specific configuration
         * @return updated builder instance
         * @see #protocolConfig()
         */
        public BUILDER protocolConfig(Consumer<WsClientProtocolConfig.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = WsClientProtocolConfig.builder();
            consumer.accept(builder);
            this.protocolConfig(builder.build());
            return self();
        }

        /**
         * WebSocket specific configuration.
         *
         * @param supplier supplier of
         *                 protocol specific configuration
         * @return updated builder instance
         * @see #protocolConfig()
         */
        public BUILDER protocolConfig(Supplier<? extends WsClientProtocolConfig> supplier) {
            Objects.requireNonNull(supplier);
            this.protocolConfig(supplier.get());
            return self();
        }

        /**
         * WebSocket specific configuration.
         *
         * @return the protocol config
         */
        public WsClientProtocolConfig protocolConfig() {
            return protocolConfig;
        }

        /**
         * If this instance was configured, this would be the config instance used.
         *
         * @return config node used to configure this builder, or empty if not configured
         */
        public Optional<Config> config() {
            return Optional.ofNullable(config);
        }

        @Override
        public String toString() {
            return "WsClientConfigBuilder{"
                    + "protocolConfig=" + protocolConfig
                    + "};"
                    + super.toString();
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
            super.preBuildPrototype();
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
            super.validatePrototype();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class WsClientConfigImpl extends HttpClientConfigImpl implements WsClientConfig, Supplier<WsClient> {

            private final WsClientProtocolConfig protocolConfig;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected WsClientConfigImpl(WsClientConfig.BuilderBase<?, ?> builder) {
                super(builder);
                this.protocolConfig = builder.protocolConfig();
            }

            @Override
            public WsClient build() {
                return WsClient.create(this);
            }

            @Override
            public WsClient get() {
                return build();
            }

            @Override
            public WsClientProtocolConfig protocolConfig() {
                return protocolConfig;
            }

            @Override
            public String toString() {
                return "WsClientConfig{"
                        + "protocolConfig=" + protocolConfig
                        + "};"
                        + super.toString();
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof WsClientConfig other)) {
                    return false;
                }
                return super.equals(other)
                            && Objects.equals(protocolConfig, other.protocolConfig());
            }

            @Override
            public int hashCode() {
                return 31 * super.hashCode() + Objects.hash(protocolConfig);
            }

        }

    }

    /**
     * Fluent API builder for {@link WsClient}.
     */
    class Builder extends WsClientConfig.BuilderBase<WsClientConfig.Builder, WsClientConfig> implements io.helidon.common.Builder<WsClientConfig.Builder, WsClient> {

        private Builder() {
        }

        @Override
        public WsClientConfig buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new WsClientConfigImpl(this);
        }

        @Override
        public WsClient build() {
            return WsClient.create(this.buildPrototype());
        }

    }

}
