/*
 * Copyright (c) 2025 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.webclient.jsonrpc;

import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Supplier;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Generated;
import io.helidon.common.config.Config;
import io.helidon.common.config.ConfigBuilderSupport;
import io.helidon.webclient.api.HttpClientConfig;

/**
 * Configuration of a JSON-RPC client.
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.codegen.BuilderCodegen", trigger = "io.helidon.webclient.jsonrpc.JsonRpcClientConfigBlueprint")
public interface JsonRpcClientConfig extends JsonRpcClientConfigBlueprint, Prototype.Api, HttpClientConfig {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static JsonRpcClientConfig.Builder builder() {
        return new JsonRpcClientConfig.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static JsonRpcClientConfig.Builder builder(JsonRpcClientConfig instance) {
        return JsonRpcClientConfig.builder().from(instance);
    }

    /**
     * Create a new instance from configuration.
     *
     * @param config used to configure the new instance
     * @return a new instance configured from configuration
     */
    static JsonRpcClientConfig create(Config config) {
        return JsonRpcClientConfig.builder().config(config).buildPrototype();
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static JsonRpcClientConfig create() {
        return JsonRpcClientConfig.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link JsonRpcClient}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends JsonRpcClientConfig.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends JsonRpcClientConfig> extends HttpClientConfig.BuilderBase<BUILDER, PROTOTYPE> implements ConfigBuilderSupport.ConfiguredBuilder<BUILDER, PROTOTYPE> {

        private Config config;
        private JsonRpcClientProtocolConfig protocolConfig = JsonRpcClientProtocolConfig.create();

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance. This method disables automatic service discovery.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(JsonRpcClientConfig prototype) {
            super.from(prototype);
            protocolConfig(prototype.protocolConfig());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(JsonRpcClientConfig.BuilderBase<?, ?> builder) {
            super.from(builder);
            protocolConfig(builder.protocolConfig());
            return self();
        }

        /**
         * Update builder from configuration (node of this type).
         * If a value is present in configuration, it would override currently configured values.
         *
         * @param config configuration instance used to obtain values to update this builder
         * @return updated builder instance
         */
        @Override
        public BUILDER config(Config config) {
            Objects.requireNonNull(config);
            this.config = config;
            super.config(config);
            config.get("protocol-config").map(JsonRpcClientProtocolConfig::create).ifPresent(this::protocolConfig);
            return self();
        }

        /**
         * JSON-RPC specific configuration.
         *
         * @param protocolConfig protocol specific configuration
         * @return updated builder instance
         * @see #protocolConfig()
         */
        public BUILDER protocolConfig(JsonRpcClientProtocolConfig protocolConfig) {
            Objects.requireNonNull(protocolConfig);
            this.protocolConfig = protocolConfig;
            return self();
        }

        /**
         * JSON-RPC specific configuration.
         *
         * @param consumer consumer of builder for
         *                 protocol specific configuration
         * @return updated builder instance
         * @see #protocolConfig()
         */
        public BUILDER protocolConfig(Consumer<JsonRpcClientProtocolConfig.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = JsonRpcClientProtocolConfig.builder();
            consumer.accept(builder);
            this.protocolConfig(builder.build());
            return self();
        }

        /**
         * JSON-RPC specific configuration.
         *
         * @param supplier supplier of
         *                 protocol specific configuration
         * @return updated builder instance
         * @see #protocolConfig()
         */
        public BUILDER protocolConfig(Supplier<? extends JsonRpcClientProtocolConfig> supplier) {
            Objects.requireNonNull(supplier);
            this.protocolConfig(supplier.get());
            return self();
        }

        /**
         * JSON-RPC specific configuration.
         *
         * @return the protocol config
         */
        public JsonRpcClientProtocolConfig protocolConfig() {
            return protocolConfig;
        }

        /**
         * If this instance was configured, this would be the config instance used.
         *
         * @return config node used to configure this builder, or empty if not configured
         */
        public Optional<Config> config() {
            return Optional.ofNullable(config);
        }

        @Override
        public String toString() {
            return "JsonRpcClientConfigBuilder{"
                    + "protocolConfig=" + protocolConfig
                    + "};"
                    + super.toString();
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
            super.preBuildPrototype();
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
            super.validatePrototype();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class JsonRpcClientConfigImpl extends HttpClientConfigImpl implements JsonRpcClientConfig, Supplier<JsonRpcClient> {

            private final JsonRpcClientProtocolConfig protocolConfig;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected JsonRpcClientConfigImpl(JsonRpcClientConfig.BuilderBase<?, ?> builder) {
                super(builder);
                this.protocolConfig = builder.protocolConfig();
            }

            @Override
            public JsonRpcClient build() {
                return JsonRpcClient.create(this);
            }

            @Override
            public JsonRpcClient get() {
                return build();
            }

            @Override
            public JsonRpcClientProtocolConfig protocolConfig() {
                return protocolConfig;
            }

            @Override
            public String toString() {
                return "JsonRpcClientConfig{"
                        + "protocolConfig=" + protocolConfig
                        + "};"
                        + super.toString();
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof JsonRpcClientConfig other)) {
                    return false;
                }
                return super.equals(other)
                            && Objects.equals(protocolConfig, other.protocolConfig());
            }

            @Override
            public int hashCode() {
                return 31 * super.hashCode() + Objects.hash(protocolConfig);
            }

        }

    }

    /**
     * Fluent API builder for {@link JsonRpcClient}.
     */
    class Builder extends JsonRpcClientConfig.BuilderBase<JsonRpcClientConfig.Builder, JsonRpcClientConfig> implements io.helidon.common.Builder<JsonRpcClientConfig.Builder, JsonRpcClient> {

        private Builder() {
        }

        @Override
        public JsonRpcClientConfig buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new JsonRpcClientConfigImpl(this);
        }

        @Override
        public JsonRpcClient build() {
            return JsonRpcClient.create(this.buildPrototype());
        }

    }

}
