/*
 * Copyright (c) 2024 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.webclient.http2;

import java.util.Objects;
import java.util.function.Consumer;
import java.util.function.Supplier;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Generated;
import io.helidon.webclient.api.HttpClientConfig;

/**
 * HTTP/2 full webclient configuration.
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.processor.BlueprintProcessor", trigger = "io.helidon.webclient.http2.Http2ClientConfigBlueprint")
public interface Http2ClientConfig extends Http2ClientConfigBlueprint, Prototype.Api, HttpClientConfig {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static Http2ClientConfig.Builder builder() {
        return new Http2ClientConfig.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static Http2ClientConfig.Builder builder(Http2ClientConfig instance) {
        return Http2ClientConfig.builder().from(instance);
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static Http2ClientConfig create() {
        return Http2ClientConfig.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link Http2Client}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends Http2ClientConfig.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends Http2ClientConfig> extends HttpClientConfig.BuilderBase<BUILDER, PROTOTYPE> implements Prototype.Builder<BUILDER, PROTOTYPE> {

        private Http2ClientProtocolConfig protocolConfig = Http2ClientProtocolConfig.create();

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(Http2ClientConfig prototype) {
            super.from(prototype);
            protocolConfig(prototype.protocolConfig());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(Http2ClientConfig.BuilderBase<?, ?> builder) {
            super.from(builder);
            protocolConfig(builder.protocolConfig());
            return self();
        }

        /**
         * HTTP/2 specific configuration.
         *
         * @param protocolConfig protocol specific configuration
         * @return updated builder instance
         * @see #protocolConfig()
         */
        public BUILDER protocolConfig(Http2ClientProtocolConfig protocolConfig) {
            Objects.requireNonNull(protocolConfig);
            this.protocolConfig = protocolConfig;
            return self();
        }

        /**
         * HTTP/2 specific configuration.
         *
         * @param consumer consumer of builder for
         *                 protocol specific configuration
         * @return updated builder instance
         * @see #protocolConfig()
         */
        public BUILDER protocolConfig(Consumer<Http2ClientProtocolConfig.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = Http2ClientProtocolConfig.builder();
            consumer.accept(builder);
            this.protocolConfig(builder.build());
            return self();
        }

        /**
         * HTTP/2 specific configuration.
         *
         * @param supplier supplier of
         *                 protocol specific configuration
         * @return updated builder instance
         * @see #protocolConfig()
         */
        public BUILDER protocolConfig(Supplier<? extends Http2ClientProtocolConfig> supplier) {
            Objects.requireNonNull(supplier);
            this.protocolConfig(supplier.get());
            return self();
        }

        /**
         * HTTP/2 specific configuration.
         *
         * @return the protocol config
         */
        public Http2ClientProtocolConfig protocolConfig() {
            return protocolConfig;
        }

        @Override
        public String toString() {
            return "Http2ClientConfigBuilder{"
                    + "protocolConfig=" + protocolConfig
                    + "};"
                    + super.toString();
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
            super.preBuildPrototype();
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
            super.validatePrototype();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class Http2ClientConfigImpl extends HttpClientConfigImpl implements Http2ClientConfig, Supplier<Http2Client> {

            private final Http2ClientProtocolConfig protocolConfig;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected Http2ClientConfigImpl(Http2ClientConfig.BuilderBase<?, ?> builder) {
                super(builder);
                this.protocolConfig = builder.protocolConfig();
            }

            @Override
            public Http2Client build() {
                return Http2Client.create(this);
            }

            @Override
            public Http2Client get() {
                return build();
            }

            @Override
            public Http2ClientProtocolConfig protocolConfig() {
                return protocolConfig;
            }

            @Override
            public String toString() {
                return "Http2ClientConfig{"
                        + "protocolConfig=" + protocolConfig
                        + "};"
                        + super.toString();
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof Http2ClientConfig other)) {
                    return false;
                }
                return super.equals(other)
                        && Objects.equals(protocolConfig, other.protocolConfig());
            }

            @Override
            public int hashCode() {
                return 31 * super.hashCode() + Objects.hash(protocolConfig);
            }

        }

    }

    /**
     * Fluent API builder for {@link Http2Client}.
     */
    class Builder extends Http2ClientConfig.BuilderBase<Http2ClientConfig.Builder, Http2ClientConfig> implements io.helidon.common.Builder<Http2ClientConfig.Builder, Http2Client> {

        private Builder() {
        }

        @Override
        public Http2ClientConfig buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new Http2ClientConfigImpl(this);
        }

        @Override
        public Http2Client build() {
            return Http2Client.create(this.buildPrototype());
        }

    }

}
