/*
 * Copyright (c) 2025 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.webclient.grpc;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Errors;
import io.helidon.common.Generated;

import io.grpc.CallCredentials;
import io.grpc.ClientInterceptor;

/**
 * Interface generated from definition. Please add javadoc to the definition interface.
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.codegen.BuilderCodegen", trigger = "io.helidon.webclient.grpc.GrpcServiceDescriptorBlueprint")
public interface GrpcServiceDescriptor extends GrpcServiceDescriptorBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static GrpcServiceDescriptor.Builder builder() {
        return new GrpcServiceDescriptor.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static GrpcServiceDescriptor.Builder builder(GrpcServiceDescriptor instance) {
        return GrpcServiceDescriptor.builder().from(instance);
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static GrpcServiceDescriptor create() {
        return GrpcServiceDescriptor.builder().buildPrototype();
    }

    /**
     * Service name.
     *
     * @return the server name
     */
    @Override
    String serviceName();

    /**
     * Map of names to gRPC method descriptors.
     *
     * @return method map
     */
    @Override
    Map<String, GrpcClientMethodDescriptor> methods();

    /**
     * Ordered list of method interceptors.
     *
     * @return list of interceptors
     */
    @Override
    List<ClientInterceptor> interceptors();

    /**
     * Credentials for this call, if any.
     *
     * @return optional credentials
     */
    @Override
    Optional<CallCredentials> callCredentials();

    /**
     * Fluent API builder base for {@link GrpcServiceDescriptor}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends GrpcServiceDescriptor.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends GrpcServiceDescriptor> implements Prototype.Builder<BUILDER, PROTOTYPE> {

        private final List<ClientInterceptor> interceptors = new ArrayList<>();
        private final Map<String, GrpcClientMethodDescriptor> methods = new LinkedHashMap<>();
        private boolean isInterceptorsMutated;
        private CallCredentials callCredentials;
        private String serviceName;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance. This method disables automatic service discovery.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(GrpcServiceDescriptor prototype) {
            serviceName(prototype.serviceName());
            addMethods(prototype.methods());
            if (!isInterceptorsMutated) {
                interceptors.clear();
            }
            addInterceptors(prototype.interceptors());
            callCredentials(prototype.callCredentials());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(GrpcServiceDescriptor.BuilderBase<?, ?> builder) {
            builder.serviceName().ifPresent(this::serviceName);
            addMethods(builder.methods);
            if (isInterceptorsMutated) {
                if (builder.isInterceptorsMutated) {
                    addInterceptors(builder.interceptors);
                }
            } else {
                interceptors.clear();
                addInterceptors(builder.interceptors);
            }
            builder.callCredentials().ifPresent(this::callCredentials);
            return self();
        }

        /**
         * Service name.
         *
         * @param serviceName the server name
         * @return updated builder instance
         * @see #serviceName()
         */
        public BUILDER serviceName(String serviceName) {
            Objects.requireNonNull(serviceName);
            this.serviceName = serviceName;
            return self();
        }

        /**
         * Map of names to gRPC method descriptors.
         * This method replaces all values with the new ones.
         *
         * @param methods method map
         * @return updated builder instance
         * @see #methods()
         */
        public BUILDER methods(Map<String, ? extends GrpcClientMethodDescriptor> methods) {
            Objects.requireNonNull(methods);
            this.methods.clear();
            this.methods.putAll(methods);
            return self();
        }

        /**
         * Map of names to gRPC method descriptors.
         * This method keeps existing values, then puts all new values into the map.
         *
         * @param methods method map
         * @return updated builder instance
         * @see #methods()
         */
        public BUILDER addMethods(Map<String, ? extends GrpcClientMethodDescriptor> methods) {
            Objects.requireNonNull(methods);
            this.methods.putAll(methods);
            return self();
        }

        /**
         * Map of names to gRPC method descriptors.
         * This method adds a new value to the map, or replaces it if the key already exists.
         *
         * @param key key to add or replace
         * @param method new value for the key
         * @return updated builder instance
         * @see #methods()
         */
        public BUILDER putMethod(String key, GrpcClientMethodDescriptor method) {
            Objects.requireNonNull(key);
            Objects.requireNonNull(method);
            this.methods.put(key, method);
            return self();
        }

        /**
         * Ordered list of method interceptors.
         *
         * @param interceptors list of interceptors
         * @return updated builder instance
         * @see #interceptors()
         */
        public BUILDER interceptors(List<? extends ClientInterceptor> interceptors) {
            Objects.requireNonNull(interceptors);
            isInterceptorsMutated = true;
            this.interceptors.clear();
            this.interceptors.addAll(interceptors);
            return self();
        }

        /**
         * Ordered list of method interceptors.
         *
         * @param interceptors list of interceptors
         * @return updated builder instance
         * @see #interceptors()
         */
        public BUILDER addInterceptors(List<? extends ClientInterceptor> interceptors) {
            Objects.requireNonNull(interceptors);
            isInterceptorsMutated = true;
            this.interceptors.addAll(interceptors);
            return self();
        }

        /**
         * Ordered list of method interceptors.
         *
         * @param interceptor list of interceptors
         * @return updated builder instance
         * @see #interceptors()
         */
        public BUILDER addInterceptor(ClientInterceptor interceptor) {
            Objects.requireNonNull(interceptor);
            this.interceptors.add(interceptor);
            isInterceptorsMutated = true;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #callCredentials()
         */
        public BUILDER clearCallCredentials() {
            this.callCredentials = null;
            return self();
        }

        /**
         * Credentials for this call, if any.
         *
         * @param callCredentials optional credentials
         * @return updated builder instance
         * @see #callCredentials()
         */
        public BUILDER callCredentials(CallCredentials callCredentials) {
            Objects.requireNonNull(callCredentials);
            this.callCredentials = callCredentials;
            return self();
        }

        /**
         * Service name.
         *
         * @return the service name
         */
        public Optional<String> serviceName() {
            return Optional.ofNullable(serviceName);
        }

        /**
         * Map of names to gRPC method descriptors.
         *
         * @return the methods
         */
        public Map<String, GrpcClientMethodDescriptor> methods() {
            return methods;
        }

        /**
         * Ordered list of method interceptors.
         *
         * @return the interceptors
         */
        public List<ClientInterceptor> interceptors() {
            return interceptors;
        }

        /**
         * Credentials for this call, if any.
         *
         * @return the call credentials
         */
        public Optional<CallCredentials> callCredentials() {
            return Optional.ofNullable(callCredentials);
        }

        @Override
        public String toString() {
            return "GrpcServiceDescriptorBuilder{"
                    + "serviceName=" + serviceName + ","
                    + "methods=" + methods + ","
                    + "interceptors=" + interceptors + ","
                    + "callCredentials=" + callCredentials
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
            Errors.Collector collector = Errors.collector();
            if (serviceName == null) {
                collector.fatal(getClass(), "Property \"serviceName\" must not be null, but not set");
            }
            collector.collect().checkValid();
        }

        /**
         * Credentials for this call, if any.
         *
         * @param callCredentials optional credentials
         * @return updated builder instance
         * @see #callCredentials()
         */
        BUILDER callCredentials(Optional<? extends CallCredentials> callCredentials) {
            Objects.requireNonNull(callCredentials);
            this.callCredentials = callCredentials.map(io.grpc.CallCredentials.class::cast).orElse(this.callCredentials);
            return self();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class GrpcServiceDescriptorImpl implements GrpcServiceDescriptor {

            private final List<ClientInterceptor> interceptors;
            private final Map<String, GrpcClientMethodDescriptor> methods;
            private final Optional<CallCredentials> callCredentials;
            private final String serviceName;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected GrpcServiceDescriptorImpl(GrpcServiceDescriptor.BuilderBase<?, ?> builder) {
                this.serviceName = builder.serviceName().get();
                this.methods = Collections.unmodifiableMap(new LinkedHashMap<>(builder.methods()));
                this.interceptors = List.copyOf(builder.interceptors());
                this.callCredentials = builder.callCredentials().map(Function.identity());
            }

            @Override
            public String serviceName() {
                return serviceName;
            }

            @Override
            public Map<String, GrpcClientMethodDescriptor> methods() {
                return methods;
            }

            @Override
            public List<ClientInterceptor> interceptors() {
                return interceptors;
            }

            @Override
            public Optional<CallCredentials> callCredentials() {
                return callCredentials;
            }

            @Override
            public String toString() {
                return "GrpcServiceDescriptor{"
                        + "serviceName=" + serviceName + ","
                        + "methods=" + methods + ","
                        + "interceptors=" + interceptors + ","
                        + "callCredentials=" + callCredentials
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof GrpcServiceDescriptor other)) {
                    return false;
                }
                return Objects.equals(serviceName, other.serviceName())
                    && Objects.equals(methods, other.methods())
                    && Objects.equals(interceptors, other.interceptors())
                    && Objects.equals(callCredentials, other.callCredentials());
            }

            @Override
            public int hashCode() {
                return Objects.hash(serviceName, methods, interceptors, callCredentials);
            }

        }

    }

    /**
     * Fluent API builder for {@link GrpcServiceDescriptor}.
     */
    class Builder extends GrpcServiceDescriptor.BuilderBase<GrpcServiceDescriptor.Builder, GrpcServiceDescriptor> implements io.helidon.common.Builder<GrpcServiceDescriptor.Builder, GrpcServiceDescriptor> {

        private Builder() {
        }

        @Override
        public GrpcServiceDescriptor buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new GrpcServiceDescriptorImpl(this);
        }

        @Override
        public GrpcServiceDescriptor build() {
            return buildPrototype();
        }

    }

}
