/*
 * Copyright (c) 2025 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.webclient.grpc;

import java.time.Duration;
import java.util.Objects;
import java.util.Optional;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Generated;
import io.helidon.config.Config;
import io.helidon.config.ConfigBuilderSupport;

/**
 * Configuration of a gRPC client.
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.codegen.BuilderCodegen", trigger = "io.helidon.webclient.grpc.GrpcClientProtocolConfigBlueprint")
public interface GrpcClientProtocolConfig extends GrpcClientProtocolConfigBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static GrpcClientProtocolConfig.Builder builder() {
        return new GrpcClientProtocolConfig.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static GrpcClientProtocolConfig.Builder builder(GrpcClientProtocolConfig instance) {
        return GrpcClientProtocolConfig.builder().from(instance);
    }

    /**
     * Create a new instance from configuration.
     *
     * @param config used to configure the new instance
     * @return a new instance configured from configuration
     */
    static GrpcClientProtocolConfig create(Config config) {
        return GrpcClientProtocolConfig.builder().config(config).buildPrototype();
    }

    /**
     * Create a new instance from configuration.
     *
     * @param config used to configure the new instance
     * @return a new instance configured from configuration
     * @deprecated use {@link #create(io.helidon.config.Config)}
     */
    @Deprecated
    static GrpcClientProtocolConfig create(io.helidon.common.config.Config config) {
        return create(Config.config(config));
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static GrpcClientProtocolConfig create() {
        return GrpcClientProtocolConfig.builder().buildPrototype();
    }

    /**
     * Name identifying this client protocol. Defaults to type.
     *
     * @return name of client protocol
     */
    @Override
    String name();

    /**
     * How long to wait for the next HTTP/2 data frame to arrive in underlying stream.
     * Whether this is a fatal error or not is controlled by {@link #abortPollTimeExpired()}.
     *
     * @return poll time as a duration
     * @see io.helidon.common.socket.SocketOptions#readTimeout()
     */
    @Override
    Duration pollWaitTime();

    /**
     * Whether to continue retrying after a poll wait timeout expired or not. If a read
     * operation timeouts out and this flag is set to {@code false}, the event is logged
     * and the client will retry. Otherwise, an exception is thrown.
     *
     * @return abort timeout flag
     */
    @Override
    boolean abortPollTimeExpired();

    /**
     * How often to send a heartbeat (HTTP/2 ping) to check if the connection is still
     * alive. This is useful for long-running, streaming gRPC calls. It is turned off by
     * default but can be enabled by setting the period to a value greater than 0.
     *
     * @return heartbeat period
     */
    @Override
    Duration heartbeatPeriod();

    /**
     * Initial buffer size used to serialize gRPC request payloads. Buffers shall grow
     * according to the payload size, but setting this initial buffer size to a larger value
     * may improve performance for certain applications.
     *
     * @return initial buffer size
     */
    @Override
    int initBufferSize();

    /**
     * When data has been received from the server but not yet requested by the client
     * (i.e., listener), the implementation will wait for this duration before signaling
     * an error. If data is requested and more data is still in the queue, this time
     * wait restarts until the next request is received. If duration expires, a
     * status of {@link io.grpc.Status#CANCELLED} is reported in the call to
     * {@link io.grpc.ClientCall.Listener#onClose(io.grpc.Status, io.grpc.Metadata)}.
     *
     * @return duration to wait for the next data request from listener
     * @see io.grpc.ClientCall.Listener#request(int)
     */
    @Override
    Duration nextRequestWaitTime();

    /**
     * Fluent API builder base for {@link GrpcClientProtocolConfig}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends GrpcClientProtocolConfig.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends GrpcClientProtocolConfig> implements ConfigBuilderSupport.ConfiguredBuilder<BUILDER, PROTOTYPE> {

        private boolean abortPollTimeExpired = false;
        private Config config;
        private Duration heartbeatPeriod = Duration.parse("PT0S");
        private Duration nextRequestWaitTime = Duration.parse("PT1S");
        private Duration pollWaitTime = Duration.parse("PT10S");
        private int initBufferSize = 2048;
        private String name = "grpc";

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance. This method disables automatic service discovery.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(GrpcClientProtocolConfig prototype) {
            name(prototype.name());
            pollWaitTime(prototype.pollWaitTime());
            abortPollTimeExpired(prototype.abortPollTimeExpired());
            heartbeatPeriod(prototype.heartbeatPeriod());
            initBufferSize(prototype.initBufferSize());
            nextRequestWaitTime(prototype.nextRequestWaitTime());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(GrpcClientProtocolConfig.BuilderBase<?, ?> builder) {
            name(builder.name());
            pollWaitTime(builder.pollWaitTime());
            abortPollTimeExpired(builder.abortPollTimeExpired());
            heartbeatPeriod(builder.heartbeatPeriod());
            initBufferSize(builder.initBufferSize());
            nextRequestWaitTime(builder.nextRequestWaitTime());
            return self();
        }

        /**
         * Update builder from configuration (node of this type).
         * If a value is present in configuration, it would override currently configured values.
         *
         * @param config configuration instance used to obtain values to update this builder
         * @return updated builder instance
         * @deprecated use {@link #config(io.helidon.config.Config)}
         */
        @Deprecated
        public BUILDER config(io.helidon.common.config.Config config) {
            return config(Config.config(config));
        }

        /**
         * Update builder from configuration (node of this type).
         * If a value is present in configuration, it would override currently configured values.
         *
         * @param config configuration instance used to obtain values to update this builder
         * @return updated builder instance
         */
        @Override
        public BUILDER config(Config config) {
            Objects.requireNonNull(config);
            this.config = config;
            config.get("name").as(String.class).ifPresent(this::name);
            config.get("poll-wait-time").as(Duration.class).ifPresent(this::pollWaitTime);
            config.get("abort-poll-time-expired").as(Boolean.class).ifPresent(this::abortPollTimeExpired);
            config.get("heartbeat-period").as(Duration.class).ifPresent(this::heartbeatPeriod);
            config.get("init-buffer-size").as(Integer.class).ifPresent(this::initBufferSize);
            config.get("next-request-wait-time").as(Duration.class).ifPresent(this::nextRequestWaitTime);
            return self();
        }

        /**
         * Name identifying this client protocol. Defaults to type.
         *
         * @param name name of client protocol
         * @return updated builder instance
         * @see #name()
         */
        public BUILDER name(String name) {
            Objects.requireNonNull(name);
            this.name = name;
            return self();
        }

        /**
         * How long to wait for the next HTTP/2 data frame to arrive in underlying stream.
         * Whether this is a fatal error or not is controlled by {@link #abortPollTimeExpired()}.
         *
         * @param pollWaitTime poll time as a duration
         * @return updated builder instance
         * @see io.helidon.common.socket.SocketOptions#readTimeout()
         * @see #pollWaitTime()
         */
        public BUILDER pollWaitTime(Duration pollWaitTime) {
            Objects.requireNonNull(pollWaitTime);
            this.pollWaitTime = pollWaitTime;
            return self();
        }

        /**
         * Whether to continue retrying after a poll wait timeout expired or not. If a read
         * operation timeouts out and this flag is set to {@code false}, the event is logged
         * and the client will retry. Otherwise, an exception is thrown.
         *
         * @param abortPollTimeExpired abort timeout flag
         * @return updated builder instance
         * @see #abortPollTimeExpired()
         */
        public BUILDER abortPollTimeExpired(boolean abortPollTimeExpired) {
            this.abortPollTimeExpired = abortPollTimeExpired;
            return self();
        }

        /**
         * How often to send a heartbeat (HTTP/2 ping) to check if the connection is still
         * alive. This is useful for long-running, streaming gRPC calls. It is turned off by
         * default but can be enabled by setting the period to a value greater than 0.
         *
         * @param heartbeatPeriod heartbeat period
         * @return updated builder instance
         * @see #heartbeatPeriod()
         */
        public BUILDER heartbeatPeriod(Duration heartbeatPeriod) {
            Objects.requireNonNull(heartbeatPeriod);
            this.heartbeatPeriod = heartbeatPeriod;
            return self();
        }

        /**
         * Initial buffer size used to serialize gRPC request payloads. Buffers shall grow
         * according to the payload size, but setting this initial buffer size to a larger value
         * may improve performance for certain applications.
         *
         * @param initBufferSize initial buffer size
         * @return updated builder instance
         * @see #initBufferSize()
         */
        public BUILDER initBufferSize(int initBufferSize) {
            this.initBufferSize = initBufferSize;
            return self();
        }

        /**
         * When data has been received from the server but not yet requested by the client
         * (i.e., listener), the implementation will wait for this duration before signaling
         * an error. If data is requested and more data is still in the queue, this time
         * wait restarts until the next request is received. If duration expires, a
         * status of {@link io.grpc.Status#CANCELLED} is reported in the call to
         * {@link io.grpc.ClientCall.Listener#onClose(io.grpc.Status, io.grpc.Metadata)}.
         *
         * @param nextRequestWaitTime duration to wait for the next data request from listener
         * @return updated builder instance
         * @see io.grpc.ClientCall.Listener#request(int)
         * @see #nextRequestWaitTime()
         */
        public BUILDER nextRequestWaitTime(Duration nextRequestWaitTime) {
            Objects.requireNonNull(nextRequestWaitTime);
            this.nextRequestWaitTime = nextRequestWaitTime;
            return self();
        }

        /**
         * Name identifying this client protocol. Defaults to type.
         *
         * @return the name
         */
        public String name() {
            return name;
        }

        /**
         * How long to wait for the next HTTP/2 data frame to arrive in underlying stream.
         * Whether this is a fatal error or not is controlled by {@link #abortPollTimeExpired()}.
         *
         * @return the poll wait time
         * @see io.helidon.common.socket.SocketOptions#readTimeout()
         * @see #pollWaitTime()
         */
        public Duration pollWaitTime() {
            return pollWaitTime;
        }

        /**
         * Whether to continue retrying after a poll wait timeout expired or not. If a read
         * operation timeouts out and this flag is set to {@code false}, the event is logged
         * and the client will retry. Otherwise, an exception is thrown.
         *
         * @return the abort poll time expired
         */
        public boolean abortPollTimeExpired() {
            return abortPollTimeExpired;
        }

        /**
         * How often to send a heartbeat (HTTP/2 ping) to check if the connection is still
         * alive. This is useful for long-running, streaming gRPC calls. It is turned off by
         * default but can be enabled by setting the period to a value greater than 0.
         *
         * @return the heartbeat period
         */
        public Duration heartbeatPeriod() {
            return heartbeatPeriod;
        }

        /**
         * Initial buffer size used to serialize gRPC request payloads. Buffers shall grow
         * according to the payload size, but setting this initial buffer size to a larger value
         * may improve performance for certain applications.
         *
         * @return the init buffer size
         */
        public int initBufferSize() {
            return initBufferSize;
        }

        /**
         * When data has been received from the server but not yet requested by the client
         * (i.e., listener), the implementation will wait for this duration before signaling
         * an error. If data is requested and more data is still in the queue, this time
         * wait restarts until the next request is received. If duration expires, a
         * status of {@link io.grpc.Status#CANCELLED} is reported in the call to
         * {@link io.grpc.ClientCall.Listener#onClose(io.grpc.Status, io.grpc.Metadata)}.
         *
         * @return the next request wait time
         * @see io.grpc.ClientCall.Listener#request(int)
         * @see #nextRequestWaitTime()
         */
        public Duration nextRequestWaitTime() {
            return nextRequestWaitTime;
        }

        /**
         * If this instance was configured, this would be the config instance used.
         *
         * @return config node used to configure this builder, or empty if not configured
         */
        public Optional<io.helidon.common.config.Config> config() {
            return Optional.ofNullable(config);
        }

        @Override
        public String toString() {
            return "GrpcClientProtocolConfigBuilder{"
                    + "name=" + name + ","
                    + "pollWaitTime=" + pollWaitTime + ","
                    + "abortPollTimeExpired=" + abortPollTimeExpired + ","
                    + "heartbeatPeriod=" + heartbeatPeriod + ","
                    + "initBufferSize=" + initBufferSize + ","
                    + "nextRequestWaitTime=" + nextRequestWaitTime
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class GrpcClientProtocolConfigImpl implements GrpcClientProtocolConfig {

            private final boolean abortPollTimeExpired;
            private final Duration heartbeatPeriod;
            private final Duration nextRequestWaitTime;
            private final Duration pollWaitTime;
            private final int initBufferSize;
            private final String name;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected GrpcClientProtocolConfigImpl(GrpcClientProtocolConfig.BuilderBase<?, ?> builder) {
                this.name = builder.name();
                this.pollWaitTime = builder.pollWaitTime();
                this.abortPollTimeExpired = builder.abortPollTimeExpired();
                this.heartbeatPeriod = builder.heartbeatPeriod();
                this.initBufferSize = builder.initBufferSize();
                this.nextRequestWaitTime = builder.nextRequestWaitTime();
            }

            @Override
            public String name() {
                return name;
            }

            @Override
            public Duration pollWaitTime() {
                return pollWaitTime;
            }

            @Override
            public boolean abortPollTimeExpired() {
                return abortPollTimeExpired;
            }

            @Override
            public Duration heartbeatPeriod() {
                return heartbeatPeriod;
            }

            @Override
            public int initBufferSize() {
                return initBufferSize;
            }

            @Override
            public Duration nextRequestWaitTime() {
                return nextRequestWaitTime;
            }

            @Override
            public String toString() {
                return "GrpcClientProtocolConfig{"
                        + "name=" + name + ","
                        + "pollWaitTime=" + pollWaitTime + ","
                        + "abortPollTimeExpired=" + abortPollTimeExpired + ","
                        + "heartbeatPeriod=" + heartbeatPeriod + ","
                        + "initBufferSize=" + initBufferSize + ","
                        + "nextRequestWaitTime=" + nextRequestWaitTime
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof GrpcClientProtocolConfig other)) {
                    return false;
                }
                return Objects.equals(name, other.name())
                    && Objects.equals(pollWaitTime, other.pollWaitTime())
                    && abortPollTimeExpired == other.abortPollTimeExpired()
                    && Objects.equals(heartbeatPeriod, other.heartbeatPeriod())
                    && initBufferSize == other.initBufferSize()
                    && Objects.equals(nextRequestWaitTime, other.nextRequestWaitTime());
            }

            @Override
            public int hashCode() {
                return Objects.hash(name, pollWaitTime, abortPollTimeExpired, heartbeatPeriod, initBufferSize, nextRequestWaitTime);
            }

        }

    }

    /**
     * Fluent API builder for {@link GrpcClientProtocolConfig}.
     */
    class Builder extends GrpcClientProtocolConfig.BuilderBase<GrpcClientProtocolConfig.Builder, GrpcClientProtocolConfig> implements io.helidon.common.Builder<GrpcClientProtocolConfig.Builder, GrpcClientProtocolConfig> {

        private Builder() {
        }

        @Override
        public GrpcClientProtocolConfig buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new GrpcClientProtocolConfigImpl(this);
        }

        @Override
        public GrpcClientProtocolConfig build() {
            return buildPrototype();
        }

    }

}
