/*
 * Copyright (c) 2025 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.webclient.grpc;

import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Supplier;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Generated;
import io.helidon.common.config.Config;
import io.helidon.common.config.ConfigBuilderSupport;
import io.helidon.webclient.api.HttpClientConfig;

/**
 * Configuration of a grpc client.
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.codegen.BuilderCodegen", trigger = "io.helidon.webclient.grpc.GrpcClientConfigBlueprint")
public interface GrpcClientConfig extends GrpcClientConfigBlueprint, Prototype.Api, HttpClientConfig {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static GrpcClientConfig.Builder builder() {
        return new GrpcClientConfig.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static GrpcClientConfig.Builder builder(GrpcClientConfig instance) {
        return GrpcClientConfig.builder().from(instance);
    }

    /**
     * Create a new instance from configuration.
     *
     * @param config used to configure the new instance
     * @return a new instance configured from configuration
     */
    static GrpcClientConfig create(Config config) {
        return GrpcClientConfig.builder().config(config).buildPrototype();
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static GrpcClientConfig create() {
        return GrpcClientConfig.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link GrpcClient}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends GrpcClientConfig.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends GrpcClientConfig> extends HttpClientConfig.BuilderBase<BUILDER, PROTOTYPE> implements ConfigBuilderSupport.ConfiguredBuilder<BUILDER, PROTOTYPE> {

        private ClientUriSupplier clientUriSupplier;
        private Config config;
        private GrpcClientProtocolConfig protocolConfig = GrpcClientProtocolConfig.create();

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance. This method disables automatic service discovery.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(GrpcClientConfig prototype) {
            super.from(prototype);
            protocolConfig(prototype.protocolConfig());
            clientUriSupplier(prototype.clientUriSupplier());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(GrpcClientConfig.BuilderBase<?, ?> builder) {
            super.from(builder);
            protocolConfig(builder.protocolConfig());
            builder.clientUriSupplier().ifPresent(this::clientUriSupplier);
            return self();
        }

        /**
         * Update builder from configuration (node of this type).
         * If a value is present in configuration, it would override currently configured values.
         *
         * @param config configuration instance used to obtain values to update this builder
         * @return updated builder instance
         */
        @Override
        public BUILDER config(Config config) {
            Objects.requireNonNull(config);
            this.config = config;
            super.config(config);
            config.get("protocol-config").map(GrpcClientProtocolConfig::create).ifPresent(this::protocolConfig);
            return self();
        }

        /**
         * gRPC specific configuration.
         *
         * @param protocolConfig protocol specific configuration
         * @return updated builder instance
         * @see #protocolConfig()
         */
        public BUILDER protocolConfig(GrpcClientProtocolConfig protocolConfig) {
            Objects.requireNonNull(protocolConfig);
            this.protocolConfig = protocolConfig;
            return self();
        }

        /**
         * gRPC specific configuration.
         *
         * @param consumer consumer of builder for
         *                 protocol specific configuration
         * @return updated builder instance
         * @see #protocolConfig()
         */
        public BUILDER protocolConfig(Consumer<GrpcClientProtocolConfig.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = GrpcClientProtocolConfig.builder();
            consumer.accept(builder);
            this.protocolConfig(builder.build());
            return self();
        }

        /**
         * gRPC specific configuration.
         *
         * @param supplier supplier of
         *                 protocol specific configuration
         * @return updated builder instance
         * @see #protocolConfig()
         */
        public BUILDER protocolConfig(Supplier<? extends GrpcClientProtocolConfig> supplier) {
            Objects.requireNonNull(supplier);
            this.protocolConfig(supplier.get());
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #clientUriSupplier()
         */
        public BUILDER clearClientUriSupplier() {
            this.clientUriSupplier = null;
            return self();
        }

        /**
         * A {@link io.helidon.webclient.grpc.ClientUriSupplier} that can dynamically
         * provide zero or more {@link io.helidon.webclient.api.ClientUri}s to connect.
         *
         * @param clientUriSupplier a supplier for zero or more client URIs
         * @return updated builder instance
         * @see #clientUriSupplier()
         */
        public BUILDER clientUriSupplier(ClientUriSupplier clientUriSupplier) {
            Objects.requireNonNull(clientUriSupplier);
            this.clientUriSupplier = clientUriSupplier;
            return self();
        }

        /**
         * gRPC specific configuration.
         *
         * @return the protocol config
         */
        public GrpcClientProtocolConfig protocolConfig() {
            return protocolConfig;
        }

        /**
         * A {@link io.helidon.webclient.grpc.ClientUriSupplier} that can dynamically
         * provide zero or more {@link io.helidon.webclient.api.ClientUri}s to connect.
         *
         * @return the client uri supplier
         */
        public Optional<ClientUriSupplier> clientUriSupplier() {
            return Optional.ofNullable(clientUriSupplier);
        }

        /**
         * If this instance was configured, this would be the config instance used.
         *
         * @return config node used to configure this builder, or empty if not configured
         */
        public Optional<Config> config() {
            return Optional.ofNullable(config);
        }

        @Override
        public String toString() {
            return "GrpcClientConfigBuilder{"
                    + "protocolConfig=" + protocolConfig + ","
                    + "clientUriSupplier=" + clientUriSupplier
                    + "};"
                    + super.toString();
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
            super.preBuildPrototype();
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
            super.validatePrototype();
        }

        /**
         * A {@link io.helidon.webclient.grpc.ClientUriSupplier} that can dynamically
         * provide zero or more {@link io.helidon.webclient.api.ClientUri}s to connect.
         *
         * @param clientUriSupplier a supplier for zero or more client URIs
         * @return updated builder instance
         * @see #clientUriSupplier()
         */
        BUILDER clientUriSupplier(Optional<? extends ClientUriSupplier> clientUriSupplier) {
            Objects.requireNonNull(clientUriSupplier);
            this.clientUriSupplier = clientUriSupplier.map(io.helidon.webclient.grpc.ClientUriSupplier.class::cast).orElse(this.clientUriSupplier);
            return self();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class GrpcClientConfigImpl extends HttpClientConfigImpl implements GrpcClientConfig, Supplier<GrpcClient> {

            private final GrpcClientProtocolConfig protocolConfig;
            private final Optional<ClientUriSupplier> clientUriSupplier;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected GrpcClientConfigImpl(GrpcClientConfig.BuilderBase<?, ?> builder) {
                super(builder);
                this.protocolConfig = builder.protocolConfig();
                this.clientUriSupplier = builder.clientUriSupplier();
            }

            @Override
            public GrpcClient build() {
                return GrpcClient.create(this);
            }

            @Override
            public GrpcClient get() {
                return build();
            }

            @Override
            public GrpcClientProtocolConfig protocolConfig() {
                return protocolConfig;
            }

            @Override
            public Optional<ClientUriSupplier> clientUriSupplier() {
                return clientUriSupplier;
            }

            @Override
            public String toString() {
                return "GrpcClientConfig{"
                        + "protocolConfig=" + protocolConfig + ","
                        + "clientUriSupplier=" + clientUriSupplier
                        + "};"
                        + super.toString();
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof GrpcClientConfig other)) {
                    return false;
                }
                return super.equals(other)
                            && Objects.equals(protocolConfig, other.protocolConfig())
                    && Objects.equals(clientUriSupplier, other.clientUriSupplier());
            }

            @Override
            public int hashCode() {
                return 31 * super.hashCode() + Objects.hash(protocolConfig, clientUriSupplier);
            }

        }

    }

    /**
     * Fluent API builder for {@link GrpcClient}.
     */
    class Builder extends GrpcClientConfig.BuilderBase<GrpcClientConfig.Builder, GrpcClientConfig> implements io.helidon.common.Builder<GrpcClientConfig.Builder, GrpcClient> {

        private Builder() {
        }

        @Override
        public GrpcClientConfig buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new GrpcClientConfigImpl(this);
        }

        @Override
        public GrpcClient build() {
            return GrpcClient.create(this.buildPrototype());
        }

    }

}
