/*
 * Copyright (c) 2025 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.webclient.api;

import java.time.Duration;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Supplier;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Errors;
import io.helidon.common.Generated;
import io.helidon.common.config.Config;
import io.helidon.common.tls.Tls;
import io.helidon.common.tls.TlsConfig;

/**
 * Common configuration for HTTP protocols.
 */
@Generated(value = "io.helidon.builder.codegen.BuilderCodegen", trigger = "io.helidon.webclient.api.HttpConfigBaseBlueprint")
public interface HttpConfigBase extends HttpConfigBaseBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static HttpConfigBase.Builder builder() {
        return new HttpConfigBase.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static HttpConfigBase.Builder builder(HttpConfigBase instance) {
        return HttpConfigBase.builder().from(instance);
    }

    /**
     * Create a new instance from configuration.
     *
     * @param config used to configure the new instance
     * @return a new instance configured from configuration
     */
    static HttpConfigBase create(Config config) {
        return new Builder().config(config).build();
    }

    /**
     * Fluent API builder base for {@link HttpConfigBase}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends HttpConfigBase.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends HttpConfigBase> implements Prototype.ConfiguredBuilder<BUILDER, PROTOTYPE> {

        private final Map<String, String> properties = new LinkedHashMap<>();
        private boolean followRedirects = true;
        private boolean keepAlive = true;
        private Config config;
        private Duration connectTimeout;
        private Duration readTimeout;
        private int maxRedirects = 10;
        private Proxy proxy;
        private Tls tls;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance. This method disables automatic service discovery.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(HttpConfigBase prototype) {
            followRedirects(prototype.followRedirects());
            maxRedirects(prototype.maxRedirects());
            tls(prototype.tls());
            readTimeout(prototype.readTimeout());
            connectTimeout(prototype.connectTimeout());
            keepAlive(prototype.keepAlive());
            proxy(prototype.proxy());
            addProperties(prototype.properties());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(HttpConfigBase.BuilderBase<?, ?> builder) {
            followRedirects(builder.followRedirects());
            maxRedirects(builder.maxRedirects());
            builder.tls().ifPresent(this::tls);
            builder.readTimeout().ifPresent(this::readTimeout);
            builder.connectTimeout().ifPresent(this::connectTimeout);
            keepAlive(builder.keepAlive());
            builder.proxy().ifPresent(this::proxy);
            addProperties(builder.properties);
            return self();
        }

        /**
         * Update builder from configuration (node of this type).
         * If a value is present in configuration, it would override currently configured values.
         *
         * @param config configuration instance used to obtain values to update this builder
         * @return updated builder instance
         */
        @Override
        public BUILDER config(Config config) {
            Objects.requireNonNull(config);
            this.config = config;
            config.get("follow-redirects").as(Boolean.class).ifPresent(this::followRedirects);
            config.get("max-redirects").as(Integer.class).ifPresent(this::maxRedirects);
            config.get("tls").map(TlsConfig::create).ifPresent(this::tls);
            config.get("read-timeout").as(Duration.class).ifPresent(this::readTimeout);
            config.get("connect-timeout").as(Duration.class).ifPresent(this::connectTimeout);
            config.get("keep-alive").as(Boolean.class).ifPresent(this::keepAlive);
            config.get("proxy").map(Proxy::create).ifPresent(this::proxy);
            config.get("properties").detach().asMap().ifPresent(this::properties);
            return self();
        }

        /**
         * Whether to follow redirects.
         *
         * @param followRedirects whether to follow redirects
         * @return updated builder instance
         * @see #followRedirects()
         */
        public BUILDER followRedirects(boolean followRedirects) {
            this.followRedirects = followRedirects;
            return self();
        }

        /**
         * Max number of followed redirects.
         * This is ignored if {@link #followRedirects()} option is {@code false}.
         *
         * @param maxRedirects max number of followed redirects
         * @return updated builder instance
         * @see #maxRedirects()
         */
        public BUILDER maxRedirects(int maxRedirects) {
            this.maxRedirects = maxRedirects;
            return self();
        }

        /**
         * TLS configuration for any TLS request from this client.
         * TLS can also be configured per request.
         * TLS is used when the protocol is set to {@code https}.
         *
         * @param tls TLS configuration to use
         * @return updated builder instance
         * @see #tls()
         */
        public BUILDER tls(Tls tls) {
            Objects.requireNonNull(tls);
            this.tls = tls;
            return self();
        }

        /**
         * TLS configuration for any TLS request from this client.
         * TLS can also be configured per request.
         * TLS is used when the protocol is set to {@code https}.
         *
         * @param tlsConfig TLS configuration to use
         * @return updated builder instance
         * @see #tls()
         */
        public BUILDER tls(TlsConfig tlsConfig) {
            Objects.requireNonNull(tlsConfig);
            this.tls = Tls.create(tlsConfig);
            return self();
        }

        /**
         * TLS configuration for any TLS request from this client.
         * TLS can also be configured per request.
         * TLS is used when the protocol is set to {@code https}.
         *
         * @param consumer consumer of builder for
         *                 TLS configuration to use
         * @return updated builder instance
         * @see #tls()
         */
        public BUILDER tls(Consumer<TlsConfig.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = TlsConfig.builder();
            consumer.accept(builder);
            this.tls(builder.build());
            return self();
        }

        /**
         * TLS configuration for any TLS request from this client.
         * TLS can also be configured per request.
         * TLS is used when the protocol is set to {@code https}.
         *
         * @param supplier supplier of
         *                 TLS configuration to use
         * @return updated builder instance
         * @see #tls()
         */
        public BUILDER tls(Supplier<? extends Tls> supplier) {
            Objects.requireNonNull(supplier);
            this.tls(supplier.get());
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #readTimeout()
         */
        public BUILDER clearReadTimeout() {
            this.readTimeout = null;
            return self();
        }

        /**
         * Read timeout.
         *
         * @param readTimeout read timeout
         * @return updated builder instance
         * @see #readTimeout()
         */
        public BUILDER readTimeout(Duration readTimeout) {
            Objects.requireNonNull(readTimeout);
            this.readTimeout = readTimeout;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #connectTimeout()
         */
        public BUILDER clearConnectTimeout() {
            this.connectTimeout = null;
            return self();
        }

        /**
         * Connect timeout.
         *
         * @param connectTimeout connect timeout
         * @return updated builder instance
         * @see #connectTimeout()
         */
        public BUILDER connectTimeout(Duration connectTimeout) {
            Objects.requireNonNull(connectTimeout);
            this.connectTimeout = connectTimeout;
            return self();
        }

        /**
         * Determines if connection keep alive is enabled (NOT socket keep alive, but HTTP connection keep alive, to re-use
         * the same connection for multiple requests).
         *
         * @param keepAlive keep alive for this connection
         * @return updated builder instance
         * @see io.helidon.common.socket.SocketOptions#socketKeepAlive()
         * @see #keepAlive()
         */
        public BUILDER keepAlive(boolean keepAlive) {
            this.keepAlive = keepAlive;
            return self();
        }

        /**
         * Proxy configuration to be used for requests.
         *
         * @param proxy proxy to use, defaults to {@link Proxy#noProxy()}
         * @return updated builder instance
         * @see #proxy()
         */
        public BUILDER proxy(Proxy proxy) {
            Objects.requireNonNull(proxy);
            this.proxy = proxy;
            return self();
        }

        /**
         * Proxy configuration to be used for requests.
         *
         * @param consumer consumer of builder for
         *                 proxy to use, defaults to {@link Proxy#noProxy()}
         * @return updated builder instance
         * @see #proxy()
         */
        public BUILDER proxy(Consumer<Proxy.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = Proxy.builder();
            consumer.accept(builder);
            this.proxy(builder.build());
            return self();
        }

        /**
         * Proxy configuration to be used for requests.
         *
         * @param supplier supplier of
         *                 proxy to use, defaults to {@link Proxy#noProxy()}
         * @return updated builder instance
         * @see #proxy()
         */
        public BUILDER proxy(Supplier<? extends Proxy> supplier) {
            Objects.requireNonNull(supplier);
            this.proxy(supplier.get());
            return self();
        }

        /**
         * This method replaces all values with the new ones.
         *
         * @param properties map of client properties
         * @return updated builder instance
         * @see #properties()
         */
        public BUILDER properties(Map<? extends String, ? extends String> properties) {
            Objects.requireNonNull(properties);
            this.properties.clear();
            this.properties.putAll(properties);
            return self();
        }

        /**
         * This method keeps existing values, then puts all new values into the map.
         *
         * @param properties map of client properties
         * @return updated builder instance
         * @see #properties()
         */
        public BUILDER addProperties(Map<? extends String, ? extends String> properties) {
            Objects.requireNonNull(properties);
            this.properties.putAll(properties);
            return self();
        }

        /**
         * This method adds a new value to the map, or replaces it if the key already exists.
         *
         * @param key key to add or replace
         * @param property new value for the key
         * @return updated builder instance
         * @see #properties()
         */
        public BUILDER putProperty(String key, String property) {
            Objects.requireNonNull(key);
            Objects.requireNonNull(property);
            this.properties.put(key, property);
            return self();
        }

        /**
         * Whether to follow redirects.
         *
         * @return the follow redirects
         */
        public boolean followRedirects() {
            return followRedirects;
        }

        /**
         * Max number of followed redirects.
         * This is ignored if {@link #followRedirects()} option is {@code false}.
         *
         * @return the max redirects
         */
        public int maxRedirects() {
            return maxRedirects;
        }

        /**
         * TLS configuration for any TLS request from this client.
         * TLS can also be configured per request.
         * TLS is used when the protocol is set to {@code https}.
         *
         * @return the tls
         */
        public Optional<Tls> tls() {
            return Optional.ofNullable(tls);
        }

        /**
         * Read timeout.
         *
         * @return the read timeout
         * @see io.helidon.common.socket.SocketOptions#readTimeout()
         */
        public Optional<Duration> readTimeout() {
            return Optional.ofNullable(readTimeout);
        }

        /**
         * Connect timeout.
         *
         * @return the connect timeout
         * @see io.helidon.common.socket.SocketOptions#connectTimeout()
         */
        public Optional<Duration> connectTimeout() {
            return Optional.ofNullable(connectTimeout);
        }

        /**
         * Determines if connection keep alive is enabled (NOT socket keep alive, but HTTP connection keep alive, to re-use
         * the same connection for multiple requests).
         *
         * @return the keep alive
         * @see io.helidon.common.socket.SocketOptions#socketKeepAlive()
         * @see #keepAlive()
         */
        public boolean keepAlive() {
            return keepAlive;
        }

        /**
         * Proxy configuration to be used for requests.
         *
         * @return the proxy
         */
        public Optional<Proxy> proxy() {
            return Optional.ofNullable(proxy);
        }

        /**
         * Properties configured for this client. These properties are propagated through client request, to be used by
         * services (and possibly for other purposes).
         *
         * @return the properties
         */
        public Map<String, String> properties() {
            return properties;
        }

        /**
         * If this instance was configured, this would be the config instance used.
         *
         * @return config node used to configure this builder, or empty if not configured
         */
        public Optional<Config> config() {
            return Optional.ofNullable(config);
        }

        @Override
        public String toString() {
            return "HttpConfigBaseBuilder{"
                    + "followRedirects=" + followRedirects + ","
                    + "maxRedirects=" + maxRedirects + ","
                    + "tls=" + tls + ","
                    + "readTimeout=" + readTimeout + ","
                    + "connectTimeout=" + connectTimeout + ","
                    + "keepAlive=" + keepAlive + ","
                    + "proxy=" + proxy + ","
                    + "properties=" + properties
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
            Errors.Collector collector = Errors.collector();
            if (tls == null) {
                collector.fatal(getClass(), "Property \"tls\" must not be null, but not set");
            }
            if (proxy == null) {
                collector.fatal(getClass(), "Property \"proxy\" must not be null, but not set");
            }
            collector.collect().checkValid();
        }

        /**
         * Read timeout.
         *
         * @param readTimeout read timeout
         * @return updated builder instance
         * @see #readTimeout()
         */
        BUILDER readTimeout(Optional<? extends Duration> readTimeout) {
            Objects.requireNonNull(readTimeout);
            this.readTimeout = readTimeout.map(java.time.Duration.class::cast).orElse(this.readTimeout);
            return self();
        }

        /**
         * Connect timeout.
         *
         * @param connectTimeout connect timeout
         * @return updated builder instance
         * @see #connectTimeout()
         */
        BUILDER connectTimeout(Optional<? extends Duration> connectTimeout) {
            Objects.requireNonNull(connectTimeout);
            this.connectTimeout = connectTimeout.map(java.time.Duration.class::cast).orElse(this.connectTimeout);
            return self();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class HttpConfigBaseImpl implements HttpConfigBase {

            private final boolean followRedirects;
            private final boolean keepAlive;
            private final int maxRedirects;
            private final Map<String, String> properties;
            private final Optional<Duration> connectTimeout;
            private final Optional<Duration> readTimeout;
            private final Proxy proxy;
            private final Tls tls;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected HttpConfigBaseImpl(HttpConfigBase.BuilderBase<?, ?> builder) {
                this.followRedirects = builder.followRedirects();
                this.maxRedirects = builder.maxRedirects();
                this.tls = builder.tls().get();
                this.readTimeout = builder.readTimeout();
                this.connectTimeout = builder.connectTimeout();
                this.keepAlive = builder.keepAlive();
                this.proxy = builder.proxy().get();
                this.properties = Collections.unmodifiableMap(new LinkedHashMap<>(builder.properties()));
            }

            @Override
            public boolean followRedirects() {
                return followRedirects;
            }

            @Override
            public int maxRedirects() {
                return maxRedirects;
            }

            @Override
            public Tls tls() {
                return tls;
            }

            @Override
            public Optional<Duration> readTimeout() {
                return readTimeout;
            }

            @Override
            public Optional<Duration> connectTimeout() {
                return connectTimeout;
            }

            @Override
            public boolean keepAlive() {
                return keepAlive;
            }

            @Override
            public Proxy proxy() {
                return proxy;
            }

            @Override
            public Map<String, String> properties() {
                return properties;
            }

            @Override
            public String toString() {
                return "HttpConfigBase{"
                        + "followRedirects=" + followRedirects + ","
                        + "maxRedirects=" + maxRedirects + ","
                        + "tls=" + tls + ","
                        + "readTimeout=" + readTimeout + ","
                        + "connectTimeout=" + connectTimeout + ","
                        + "keepAlive=" + keepAlive + ","
                        + "proxy=" + proxy + ","
                        + "properties=" + properties
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof HttpConfigBase other)) {
                    return false;
                }
                return followRedirects == other.followRedirects()
                    && maxRedirects == other.maxRedirects()
                    && Objects.equals(tls, other.tls())
                    && Objects.equals(readTimeout, other.readTimeout())
                    && Objects.equals(connectTimeout, other.connectTimeout())
                    && keepAlive == other.keepAlive()
                    && Objects.equals(proxy, other.proxy())
                    && Objects.equals(properties, other.properties());
            }

            @Override
            public int hashCode() {
                return Objects.hash(followRedirects, maxRedirects, tls, readTimeout, connectTimeout, keepAlive, proxy, properties);
            }

        }

    }

    /**
     * Fluent API builder for {@link HttpConfigBase}.
     */
    class Builder extends HttpConfigBase.BuilderBase<HttpConfigBase.Builder, HttpConfigBase> implements io.helidon.common.Builder<HttpConfigBase.Builder, HttpConfigBase> {

        Builder() {
        }

        @Override
        public HttpConfigBase buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new HttpConfigBaseImpl(this);
        }

        @Override
        public HttpConfigBase build() {
            return buildPrototype();
        }

    }

}
