/*
 * Copyright (c) 2024 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.webclient.api;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.ServiceLoader;
import java.util.function.Supplier;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Generated;
import io.helidon.common.HelidonServiceLoader;
import io.helidon.common.config.Config;
import io.helidon.webclient.spi.ProtocolConfig;
import io.helidon.webclient.spi.ProtocolConfigProvider;

/**
 * WebClient configuration.
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.codegen.BuilderCodegen", trigger = "io.helidon.webclient.api.WebClientConfigBlueprint")
public interface WebClientConfig extends WebClientConfigBlueprint, Prototype.Api, HttpClientConfig, HttpConfigBase {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static WebClientConfig.Builder builder() {
        return new WebClientConfig.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static WebClientConfig.Builder builder(WebClientConfig instance) {
        return WebClientConfig.builder().from(instance);
    }

    /**
     * Create a new instance from configuration.
     *
     * @param config used to configure the new instance
     * @return a new instance configured from configuration
     */
    static WebClientConfig create(Config config) {
        return WebClientConfig.builder().config(config).buildPrototype();
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static WebClientConfig create() {
        return WebClientConfig.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link WebClient}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends WebClientConfig.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends WebClientConfig> extends HttpClientConfig.BuilderBase<BUILDER, PROTOTYPE> implements Prototype.ConfiguredBuilder<BUILDER, PROTOTYPE> {

        private final List<ProtocolConfig> protocolConfigs = new ArrayList<>();
        private final List<String> protocolPreference = new ArrayList<>();
        private boolean protocolConfigsDiscoverServices = true;
        private Config config;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(WebClientConfig prototype) {
            super.from(prototype);
            addProtocolConfigs(prototype.protocolConfigs());
            addProtocolPreference(prototype.protocolPreference());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(WebClientConfig.BuilderBase<?, ?> builder) {
            super.from(builder);
            addProtocolConfigs(builder.protocolConfigs());
            addProtocolPreference(builder.protocolPreference());
            return self();
        }

        /**
         * Update builder from configuration (node of this type).
         * If a value is present in configuration, it would override currently configured values.
         *
         * @param config configuration instance used to obtain values to update this builder
         * @return updated builder instance
         */
        @Override
        public BUILDER config(Config config) {
            Objects.requireNonNull(config);
            this.config = config;
            super.config(config);
            return self();
        }

        /**
         * Configuration of client protocols.
         *
         * @param discoverServices whether to discover implementations through service loader
         * @return updated builder instance
         * @see #protocolConfigs()
         */
        public BUILDER protocolConfigsDiscoverServices(boolean discoverServices) {
            this.protocolConfigsDiscoverServices = discoverServices;
            return self();
        }

        /**
         * Configuration of client protocols.
         *
         * @param protocolConfigs client protocol configurations
         * @return updated builder instance
         * @see #protocolConfigs()
         */
        public BUILDER protocolConfigs(List<? extends ProtocolConfig> protocolConfigs) {
            Objects.requireNonNull(protocolConfigs);
            this.protocolConfigs.clear();
            this.protocolConfigs.addAll(protocolConfigs);
            return self();
        }

        /**
         * Configuration of client protocols.
         *
         * @param protocolConfigs client protocol configurations
         * @return updated builder instance
         * @see #protocolConfigs()
         */
        public BUILDER addProtocolConfigs(List<? extends ProtocolConfig> protocolConfigs) {
            Objects.requireNonNull(protocolConfigs);
            this.protocolConfigs.addAll(protocolConfigs);
            return self();
        }

        /**
         * Configuration of client protocols.
         *
         * @param protocolConfig client protocol configurations
         * @return updated builder instance
         * @see #protocolConfigs()
         */
        public BUILDER addProtocolConfig(ProtocolConfig protocolConfig) {
            Objects.requireNonNull(protocolConfig);
            this.protocolConfigs.add(protocolConfig);
            return self();
        }

        /**
         * List of HTTP protocol IDs by order of preference. If left empty, all discovered providers will be used, ordered by
         * weight.
         * <p>
         * For example if both HTTP/2 and HTTP/1.1 providers are available (considering HTTP/2 has higher weights), for ALPN
         * we will send h2 and http/1.1 and decide based on response.
         * If TLS is not used, we would attempt an upgrade (or use prior knowledge if configured in {@link #protocolConfigs()}).
         *
         * @param protocolPreference list of HTTP protocol IDs in order of preference
         * @return updated builder instance
         * @see #protocolPreference()
         */
        public BUILDER protocolPreference(List<? extends String> protocolPreference) {
            Objects.requireNonNull(protocolPreference);
            this.protocolPreference.clear();
            this.protocolPreference.addAll(protocolPreference);
            return self();
        }

        /**
         * List of HTTP protocol IDs by order of preference. If left empty, all discovered providers will be used, ordered by
         * weight.
         * <p>
         * For example if both HTTP/2 and HTTP/1.1 providers are available (considering HTTP/2 has higher weights), for ALPN
         * we will send h2 and http/1.1 and decide based on response.
         * If TLS is not used, we would attempt an upgrade (or use prior knowledge if configured in {@link #protocolConfigs()}).
         *
         * @param protocolPreference list of HTTP protocol IDs in order of preference
         * @return updated builder instance
         * @see #protocolPreference()
         */
        public BUILDER addProtocolPreference(List<? extends String> protocolPreference) {
            Objects.requireNonNull(protocolPreference);
            this.protocolPreference.addAll(protocolPreference);
            return self();
        }

        /**
         * List of HTTP protocol IDs by order of preference. If left empty, all discovered providers will be used, ordered by
         * weight.
         * <p>
         * For example if both HTTP/2 and HTTP/1.1 providers are available (considering HTTP/2 has higher weights), for ALPN
         * we will send h2 and http/1.1 and decide based on response.
         * If TLS is not used, we would attempt an upgrade (or use prior knowledge if configured in {@link #protocolConfigs()}).
         *
         * @param protocolPreference list of HTTP protocol IDs in order of preference
         * @return updated builder instance
         * @see #protocolPreference()
         */
        public BUILDER addProtocolPreference(String protocolPreference) {
            Objects.requireNonNull(protocolPreference);
            this.protocolPreference.add(protocolPreference);
            return self();
        }

        /**
         * Configuration of client protocols.
         *
         * @return the protocol configs
         */
        public List<ProtocolConfig> protocolConfigs() {
            return protocolConfigs;
        }

        /**
         * List of HTTP protocol IDs by order of preference. If left empty, all discovered providers will be used, ordered by
         * weight.
         * <p>
         * For example if both HTTP/2 and HTTP/1.1 providers are available (considering HTTP/2 has higher weights), for ALPN
         * we will send h2 and http/1.1 and decide based on response.
         * If TLS is not used, we would attempt an upgrade (or use prior knowledge if configured in {@link #protocolConfigs()}).
         *
         * @return the protocol preference
         */
        public List<String> protocolPreference() {
            return protocolPreference;
        }

        /**
         * If this instance was configured, this would be the config instance used.
         *
         * @return config node used to configure this builder, or empty if not configured
         */
        public Optional<Config> config() {
            return Optional.ofNullable(config);
        }

        @Override
        public String toString() {
            return "WebClientConfigBuilder{"
                    + "protocolConfigs=" + protocolConfigs + ","
                    + "protocolPreference=" + protocolPreference
                    + "};"
                    + super.toString();
        }

        /**
         * Handles providers and decorators.
         */
        @SuppressWarnings("unchecked")
        protected void preBuildPrototype() {
            super.preBuildPrototype();
            this.config = config == null ? Config.empty() : config;
            {
                var serviceLoader = HelidonServiceLoader.create(ServiceLoader.load(ProtocolConfigProvider.class));
                this.addProtocolConfigs(discoverServices(config, "protocol-configs", serviceLoader, ProtocolConfigProvider.class, ProtocolConfig.class, protocolConfigsDiscoverServices, protocolConfigs));
            }
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
            super.validatePrototype();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class WebClientConfigImpl extends HttpClientConfigImpl implements WebClientConfig, Supplier<WebClient> {

            private final List<ProtocolConfig> protocolConfigs;
            private final List<String> protocolPreference;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected WebClientConfigImpl(WebClientConfig.BuilderBase<?, ?> builder) {
                super(builder);
                this.protocolConfigs = List.copyOf(builder.protocolConfigs());
                this.protocolPreference = List.copyOf(builder.protocolPreference());
            }

            @Override
            public WebClient build() {
                return WebClient.create(this);
            }

            @Override
            public WebClient get() {
                return build();
            }

            @Override
            public List<ProtocolConfig> protocolConfigs() {
                return protocolConfigs;
            }

            @Override
            public List<String> protocolPreference() {
                return protocolPreference;
            }

            @Override
            public String toString() {
                return "WebClientConfig{"
                        + "protocolConfigs=" + protocolConfigs + ","
                        + "protocolPreference=" + protocolPreference
                        + "};"
                        + super.toString();
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof WebClientConfig other)) {
                    return false;
                }
                return super.equals(other)
                        && Objects.equals(protocolConfigs, other.protocolConfigs())
                        && Objects.equals(protocolPreference, other.protocolPreference());
            }

            @Override
            public int hashCode() {
                return 31 * super.hashCode() + Objects.hash(protocolConfigs, protocolPreference);
            }

        }

    }

    /**
     * Fluent API builder for {@link WebClient}.
     */
    class Builder extends WebClientConfig.BuilderBase<WebClientConfig.Builder, WebClientConfig> implements io.helidon.common.Builder<WebClientConfig.Builder, WebClient> {

        private Builder() {
        }

        @Override
        public WebClientConfig buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new WebClientConfigImpl(this);
        }

        @Override
        public WebClient build() {
            return WebClient.create(this.buildPrototype());
        }

    }

}
