/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.security.tools.config;

import io.helidon.common.OptionalHelper;
import io.helidon.common.configurable.Resource;
import io.helidon.common.pki.KeyConfig;
import io.helidon.config.Config;
import io.helidon.security.tools.config.SecureConfigException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.Key;
import java.security.PrivateKey;
import java.security.SecureRandom;
import java.util.Base64;
import java.util.Objects;
import java.util.Optional;
import java.util.logging.Logger;
import javax.crypto.Cipher;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.PBEKeySpec;
import javax.crypto.spec.SecretKeySpec;

public final class CryptUtil {
    private static final Logger LOGGER = Logger.getLogger(CryptUtil.class.getName());
    private static final SecureRandom SECURE_RANDOM = new SecureRandom();
    private static final int SALT_LENGTH = 16;
    private static final int SEED_LENGTH = 16;
    private static final int HASH_ITERATIONS = 10000;
    private static final int KEY_LENGTH = 128;

    private CryptUtil() {
        throw new IllegalStateException("Utility class");
    }

    public static String decryptRsa(Key key, String encryptedBase64) throws SecureConfigException {
        Objects.requireNonNull(key, "Key must be provided for decryption");
        Objects.requireNonNull(encryptedBase64, "Encrypted bytes must be provided for decryption (base64 encoded)");
        try {
            Cipher cipher = Cipher.getInstance("RSA");
            cipher.init(2, key);
            byte[] decrypted = cipher.doFinal(Base64.getDecoder().decode(encryptedBase64));
            return new String(decrypted, StandardCharsets.UTF_8);
        }
        catch (SecureConfigException e) {
            throw e;
        }
        catch (Exception e) {
            throw new SecureConfigException("Failed to decrypt value using RSA. Returning clear text value as is: " + encryptedBase64);
        }
    }

    public static String encryptRsa(Key key, String secret) throws SecureConfigException {
        Objects.requireNonNull(key, "Key must be provided for encryption");
        Objects.requireNonNull(secret, "Secret message must be provided to be encrypted");
        try {
            Cipher cipher = Cipher.getInstance("RSA");
            cipher.init(1, key);
            byte[] encrypted = cipher.doFinal(secret.getBytes(StandardCharsets.UTF_8));
            return Base64.getEncoder().encodeToString(encrypted);
        }
        catch (Exception e) {
            throw new SecureConfigException("Failed to encrypt using RSA key", e);
        }
    }

    public static String encryptAes(char[] masterPassword, String secret) throws SecureConfigException {
        byte[] encryptedMessageBytes;
        Objects.requireNonNull(masterPassword, "Password must be provided for encryption");
        Objects.requireNonNull(secret, "Secret message must be provided to be encrypted");
        byte[] salt = SECURE_RANDOM.generateSeed(16);
        Cipher cipher = CryptUtil.cipher(masterPassword, salt, 1);
        byte[] seed = SECURE_RANDOM.generateSeed(16);
        byte[] secretBytes = secret.getBytes(StandardCharsets.UTF_8);
        byte[] bytesToEncrypt = new byte[secretBytes.length + seed.length];
        System.arraycopy(seed, 0, bytesToEncrypt, 0, seed.length);
        System.arraycopy(secretBytes, 0, bytesToEncrypt, seed.length, secretBytes.length);
        try {
            encryptedMessageBytes = cipher.doFinal(bytesToEncrypt);
        }
        catch (Exception e) {
            throw new SecureConfigException("Failed to encrypt", e);
        }
        byte[] bytesToEncode = new byte[encryptedMessageBytes.length + salt.length];
        System.arraycopy(salt, 0, bytesToEncode, 0, salt.length);
        System.arraycopy(encryptedMessageBytes, 0, bytesToEncode, seed.length, encryptedMessageBytes.length);
        return Base64.getEncoder().encodeToString(bytesToEncode);
    }

    private static Cipher cipher(char[] masterPassword, byte[] salt, int cipherMode) throws SecureConfigException {
        try {
            SecretKeyFactory secretKeyFactory = SecretKeyFactory.getInstance("PBKDF2WithHmacSHA1");
            PBEKeySpec keySpec = new PBEKeySpec(masterPassword, salt, 10000, 128);
            SecretKeySpec spec = new SecretKeySpec(secretKeyFactory.generateSecret(keySpec).getEncoded(), "AES");
            Cipher cipher = Cipher.getInstance("AES/CBC/PKCS5Padding");
            cipher.init(cipherMode, (Key)spec, new IvParameterSpec(salt));
            return cipher;
        }
        catch (Exception e) {
            throw new SecureConfigException("Failed to prepare a cipher instance", e);
        }
    }

    static String decryptAes(char[] masterPassword, String encryptedBase64) throws SecureConfigException {
        Objects.requireNonNull(masterPassword, "Password must be provided for encryption");
        Objects.requireNonNull(encryptedBase64, "Encrypted bytes must be provided for decryption (base64 encoded)");
        try {
            byte[] decodedBytes = Base64.getDecoder().decode(encryptedBase64);
            byte[] salt = new byte[16];
            byte[] encryptedBytes = new byte[decodedBytes.length - 16];
            System.arraycopy(decodedBytes, 0, salt, 0, 16);
            System.arraycopy(decodedBytes, 16, encryptedBytes, 0, encryptedBytes.length);
            Cipher cipher = CryptUtil.cipher(masterPassword, salt, 2);
            byte[] decryptedBytes = cipher.doFinal(encryptedBytes);
            byte[] originalBytes = new byte[decryptedBytes.length - 16];
            System.arraycopy(decryptedBytes, 16, originalBytes, 0, originalBytes.length);
            return new String(originalBytes, StandardCharsets.UTF_8);
        }
        catch (Throwable e) {
            throw new SecureConfigException("Failed to decrypt value using AES. Returning clear text value as is: " + encryptedBase64, e);
        }
    }

    static Optional<char[]> resolveMasterPassword(boolean requireEncryption, Config config) {
        Optional<char[]> result = OptionalHelper.from(CryptUtil.getEnv("SECURE_CONFIG_AES_MASTER_PWD")).or(() -> {
            Optional value = config.get("security.config.aes.insecure-passphrase").value();
            if (value.isPresent() && requireEncryption) {
                LOGGER.warning("Master password is configured as clear text in configuration when encryption is required. This value will be ignored. System property or environment variable expected!!!");
                return Optional.empty();
            }
            return value;
        }).asOptional().map(String::toCharArray);
        if (!result.isPresent()) {
            LOGGER.fine("Securing properties using master password is not available, as master password is not configured");
        }
        return result;
    }

    static Optional<PrivateKey> resolvePrivateKey(Config config) {
        KeyConfig.PemBuilder pemBuilder = KeyConfig.pemBuilder().from(config);
        KeyConfig.KeystoreBuilder keystoreBuilder = KeyConfig.keystoreBuilder().from(config);
        CryptUtil.getEnv("SECURE_CONFIG_RSA_PEM_KEY").map(x$0 -> Paths.get(x$0, new String[0])).ifPresent(path -> pemBuilder.key(Resource.from((Path)path)));
        CryptUtil.getEnv("SECURE_CONFIG_PRIVATE_KEY_PASSPHRASE").map(String::toCharArray).ifPresent(arg_0 -> ((KeyConfig.PemBuilder)pemBuilder).keyPassphrase(arg_0));
        CryptUtil.getEnv("SECURE_CONFIG_RSA_PRIVATE_KEY").map(x$0 -> Paths.get(x$0, new String[0])).ifPresent(path -> keystoreBuilder.keystore(Resource.from((Path)path)));
        CryptUtil.getEnv("SECURE_CONFIG_PRIVATE_KEY_TYPE").ifPresent(arg_0 -> ((KeyConfig.KeystoreBuilder)keystoreBuilder).keystoreType(arg_0));
        CryptUtil.getEnv("SECURE_CONFIG_PRIVATE_KEYSTORE_PASSPHRASE").map(String::toCharArray).ifPresent(arg_0 -> ((KeyConfig.KeystoreBuilder)keystoreBuilder).keystorePassphrase(arg_0));
        CryptUtil.getEnv("SECURE_CONFIG_PRIVATE_KEY_PASSPHRASE").map(String::toCharArray).ifPresent(arg_0 -> ((KeyConfig.KeystoreBuilder)keystoreBuilder).keyPassphrase(arg_0));
        CryptUtil.getEnv("SECURE_CONFIG_PRIVATE_KEY_ALIAS").ifPresent(arg_0 -> ((KeyConfig.KeystoreBuilder)keystoreBuilder).keyAlias(arg_0));
        Optional result = KeyConfig.fullBuilder().updateWith(pemBuilder).updateWith(keystoreBuilder).build().getPrivateKey();
        if (!result.isPresent()) {
            LOGGER.fine("Securing properties using asymmetric cipher is not available, as private key is not configured");
        }
        return result;
    }

    static Optional<String> getEnv(String envVariable) {
        return Optional.ofNullable(System.getenv(envVariable));
    }
}

