/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.security.jwt;

import io.helidon.security.jwt.JwtException;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.net.URI;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.Signature;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.Base64;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.crypto.Mac;
import javax.json.Json;
import javax.json.JsonNumber;
import javax.json.JsonObject;
import javax.json.JsonObjectBuilder;
import javax.json.JsonString;
import javax.json.JsonValue;

public final class JwtUtil {
    private static final DateTimeFormatter YEAR_FORMAT = DateTimeFormatter.ofPattern("yyyy");
    private static final DateTimeFormatter DATE_FORMAT = DateTimeFormatter.ofPattern("yyyy-MM-dd");
    private static final Base64.Decoder URL_DECODER = Base64.getUrlDecoder();
    private static final Base64.Encoder URL_ENCODER = Base64.getUrlEncoder();
    private static final Pattern LOCALE_PATTERN = Pattern.compile("(\\w+)_(\\w+)");

    private JwtUtil() {
    }

    static String base64Url(byte[] bytesToEncode) {
        return URL_ENCODER.encodeToString(bytesToEncode);
    }

    public static BigInteger asBigInteger(JsonObject json, String key, String description) throws JwtException {
        return JwtUtil.getBigInteger(json, key, description).orElseThrow(() -> new JwtException("Key \"" + key + "\" is mandatory for " + description));
    }

    public static String asString(JsonObject json, String key, String description) throws JwtException {
        return JwtUtil.getString(json, key).orElseThrow(() -> new JwtException("Key \"" + key + "\" is mandatory for " + description));
    }

    public static Optional<BigInteger> getBigInteger(JsonObject json, String key, String description) throws JwtException {
        return JwtUtil.getByteArray(json, key, description).map(byteValue -> {
            try {
                return new BigInteger(1, (byte[])byteValue);
            }
            catch (Exception e) {
                throw new JwtException("Failed to get a big decimal for: " + description + ", from value of key " + key, e);
            }
        });
    }

    public static Optional<List<String>> getStrings(JsonObject json, String key) throws JwtException {
        return Optional.ofNullable(json.getJsonArray(key)).map(it -> {
            try {
                return it.stream().map(it2 -> ((JsonString)it2).getString()).collect(Collectors.toList());
            }
            catch (Exception e) {
                throw new JwtException("Invalid value. Expecting a string array for key " + key);
            }
        });
    }

    public static Optional<String> getString(JsonObject json, String key) throws JwtException {
        return Optional.ofNullable(json.getString(key, null));
    }

    public static Optional<byte[]> getByteArray(JsonObject json, String key, String description) throws JwtException {
        return JwtUtil.getString(json, key).map(base64 -> {
            try {
                return URL_DECODER.decode((String)base64);
            }
            catch (Exception e) {
                throw new JwtException("Failed to decode base64 from json for: " + description + ", value: \"" + base64 + '\"', e);
            }
        });
    }

    public static byte[] asByteArray(JsonObject json, String key, String description) throws JwtException {
        return JwtUtil.getByteArray(json, key, description).orElseThrow(() -> new JwtException("Key \"" + key + "\" is mandatory for " + description));
    }

    public static KeyFactory getKeyFactory(String algorithm) throws JwtException {
        try {
            return KeyFactory.getInstance(algorithm);
        }
        catch (NoSuchAlgorithmException e) {
            throw new JwtException("Failed to create key factory for algorithm: \"" + algorithm + "\"", e);
        }
    }

    public static Signature getSignature(String signatureAlgorithm) throws JwtException {
        try {
            return Signature.getInstance(signatureAlgorithm);
        }
        catch (NoSuchAlgorithmException e) {
            throw new JwtException("Failed to get Signature instance for algorithm \"" + signatureAlgorithm + "\"");
        }
    }

    public static Mac getMac(String algorithm) throws JwtException {
        try {
            return Mac.getInstance(algorithm);
        }
        catch (NoSuchAlgorithmException e) {
            throw new JwtException("Failed to get Mac instance for algorithm \"" + algorithm + "\"");
        }
    }

    public static Map<String, JsonValue> transformToJson(Map<String, Object> claims) {
        HashMap<String, JsonValue> result = new HashMap<String, JsonValue>();
        claims.forEach((s, o) -> result.put((String)s, JwtUtil.toJson(o)));
        return result;
    }

    private static JsonValue toJson(Object object) {
        if (object instanceof String) {
            return Json.createValue((String)((String)object));
        }
        if (object instanceof Integer) {
            return Json.createValue((int)((Integer)object));
        }
        if (object instanceof Double) {
            return Json.createValue((double)((Double)object));
        }
        if (object instanceof Long) {
            return Json.createValue((long)((Long)object));
        }
        if (object instanceof BigDecimal) {
            return Json.createValue((BigDecimal)((BigDecimal)object));
        }
        if (object instanceof BigInteger) {
            return Json.createValue((BigInteger)((BigInteger)object));
        }
        if (object instanceof Boolean) {
            return (Boolean)object != false ? JsonValue.TRUE : JsonValue.FALSE;
        }
        if (object instanceof Address) {
            return ((Address)object).getJson();
        }
        return Json.createValue((String)String.valueOf(object));
    }

    private static Locale toLocale(String locale) {
        Matcher matcher = LOCALE_PATTERN.matcher(locale);
        Locale result = matcher.matches() ? new Locale(matcher.group(1), matcher.group(2)) : Locale.forLanguageTag(locale);
        return result;
    }

    static Optional<Address> toAddress(JsonObject json, String name) {
        return Optional.ofNullable(json.getJsonObject(name)).map(Address::new);
    }

    static Optional<List<String>> toScopes(JsonObject json) {
        return JwtUtil.getString(json, "scope").map(it -> Arrays.asList(it.split(" ")));
    }

    static Optional<ZoneId> toTimeZone(JsonObject json, String name) {
        return JwtUtil.getString(json, name).map(ZoneId::of);
    }

    static Optional<LocalDate> toDate(JsonObject json, String name) {
        return JwtUtil.getString(json, name).map(it -> {
            if (it.length() == 4) {
                return LocalDate.parse(it, YEAR_FORMAT);
            }
            return LocalDate.parse(it, DATE_FORMAT);
        });
    }

    static Optional<URI> toUri(JsonObject json, String name) {
        return JwtUtil.getString(json, name).map(URI::create);
    }

    static Optional<Locale> toLocale(JsonObject json, String name) {
        return JwtUtil.getString(json, name).map(JwtUtil::toLocale);
    }

    static Optional<Boolean> toBoolean(JsonObject json, String name) {
        if (json.containsKey((Object)name)) {
            return Optional.of(json.getBoolean(name));
        }
        return Optional.empty();
    }

    static Optional<Instant> toInstant(JsonObject json, String name) {
        return Optional.ofNullable(json.getJsonNumber(name)).map(JsonNumber::longValue).map(Instant::ofEpochSecond);
    }

    static String toDate(LocalDate it) {
        return it.format(DATE_FORMAT);
    }

    public static Object toObject(JsonValue jsonValue) {
        switch (jsonValue.getValueType()) {
            case ARRAY: {
                return jsonValue.toString();
            }
            case OBJECT: {
                return jsonValue.toString();
            }
            case STRING: {
                return ((JsonString)jsonValue).getString();
            }
            case NUMBER: {
                return ((JsonNumber)jsonValue).numberValue();
            }
            case TRUE: {
                return true;
            }
            case FALSE: {
                return false;
            }
            case NULL: {
                return null;
            }
        }
        return jsonValue.toString();
    }

    public static class Address {
        private final Optional<String> formatted;
        private final Optional<String> streetAddress;
        private final Optional<String> locality;
        private final Optional<String> region;
        private final Optional<String> postalCode;
        private final Optional<String> country;

        public Address(JsonObject jsonObject) {
            this.formatted = JwtUtil.getString(jsonObject, "formatted");
            this.streetAddress = JwtUtil.getString(jsonObject, "street_address");
            this.locality = JwtUtil.getString(jsonObject, "locality");
            this.region = JwtUtil.getString(jsonObject, "region");
            this.postalCode = JwtUtil.getString(jsonObject, "postal_code");
            this.country = JwtUtil.getString(jsonObject, "country");
        }

        public Optional<String> getFormatted() {
            return this.formatted;
        }

        public Optional<String> getStreetAddress() {
            return this.streetAddress;
        }

        public Optional<String> getLocality() {
            return this.locality;
        }

        public Optional<String> getRegion() {
            return this.region;
        }

        public Optional<String> getPostalCode() {
            return this.postalCode;
        }

        public Optional<String> getCountry() {
            return this.country;
        }

        JsonObject getJson() {
            JsonObjectBuilder objectBuilder = Json.createObjectBuilder();
            this.formatted.ifPresent(it -> objectBuilder.add("formatted", it));
            this.streetAddress.ifPresent(it -> objectBuilder.add("street_address", it));
            this.locality.ifPresent(it -> objectBuilder.add("locality", it));
            this.region.ifPresent(it -> objectBuilder.add("region", it));
            this.postalCode.ifPresent(it -> objectBuilder.add("postal_code", it));
            this.country.ifPresent(it -> objectBuilder.add("country", it));
            return objectBuilder.build();
        }
    }
}

