/*
 * Copyright (c) 2023 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.scheduling;

import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ScheduledExecutorService;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Errors;
import io.helidon.common.Generated;
import io.helidon.common.config.Config;

/**
 * Interface generated from definition. Please add javadoc to the definition interface.
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.processor.BlueprintProcessor", trigger = "io.helidon.scheduling.TaskConfigBlueprint")
public interface TaskConfig extends TaskConfigBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static TaskConfig.Builder builder() {
        return new TaskConfig.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static TaskConfig.Builder builder(TaskConfig instance) {
        return TaskConfig.builder().from(instance);
    }

    /**
     * Create a new instance from configuration.
     *
     * @param config used to configure the new instance
     * @return a new instance configured from configuration
     */
    static TaskConfig create(Config config) {
        return TaskConfig.builder().config(config).buildPrototype();
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static TaskConfig create() {
        return TaskConfig.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link TaskConfig}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends TaskConfig.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends TaskConfig> implements Prototype.ConfiguredBuilder<BUILDER, PROTOTYPE> {

        private Config config;
        private ScheduledExecutorService executor;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(TaskConfig prototype) {
            executor(prototype.executor());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(TaskConfig.BuilderBase<?, ?> builder) {
            builder.executor().ifPresent(this::executor);
            return self();
        }

        /**
         * Update builder from configuration (node of this type).
         * If a value is present in configuration, it would override currently configured values.
         *
         * @param config configuration instance used to obtain values to update this builder
         * @return updated builder instance
         */
        @Override
        public BUILDER config(Config config) {
            Objects.requireNonNull(config);
            this.config = config;
            return self();
        }

        /**
         * Custom {@link java.util.concurrent.ScheduledExecutorService ScheduledExecutorService} used for executing scheduled task.
         *
         * @param executor custom ScheduledExecutorService
         * @return updated builder instance
         * @see #executor()
         */
        public BUILDER executor(ScheduledExecutorService executor) {
            Objects.requireNonNull(executor);
            this.executor = executor;
            return self();
        }

        /**
         * Custom {@link java.util.concurrent.ScheduledExecutorService ScheduledExecutorService} used for executing scheduled task.
         *
         * @return the executor
         */
        public Optional<ScheduledExecutorService> executor() {
            return Optional.ofNullable(executor);
        }

        /**
         * If this instance was configured, this would be the config instance used.
         *
         * @return config node used to configure this builder, or empty if not configured
         */
        public Optional<Config> config() {
            return Optional.ofNullable(config);
        }

        @Override
        public String toString() {
            return "TaskConfigBuilder{"
                    + "executor=" + executor
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
            new TaskConfigDecorator().decorate(this);
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
            Errors.Collector collector = Errors.collector();
            if (executor == null) {
                collector.fatal(getClass(), "Property \"executor\" must not be null, but not set");
            }
            collector.collect().checkValid();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class TaskConfigImpl implements TaskConfig {

            private final ScheduledExecutorService executor;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected TaskConfigImpl(TaskConfig.BuilderBase<?, ?> builder) {
                this.executor = builder.executor().get();
            }

            @Override
            public ScheduledExecutorService executor() {
                return executor;
            }

            @Override
            public String toString() {
                return "TaskConfig{"
                        + "executor=" + executor
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof TaskConfig other)) {
                    return false;
                }
                return Objects.equals(executor, other.executor());
            }

            @Override
            public int hashCode() {
                return Objects.hash(executor);
            }

        }

    }

    /**
     * Fluent API builder for {@link TaskConfig}.
     */
    class Builder extends TaskConfig.BuilderBase<TaskConfig.Builder, TaskConfig> implements io.helidon.common.Builder<TaskConfig.Builder, TaskConfig> {

        private Builder() {
        }

        @Override
        public TaskConfig buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new TaskConfigImpl(this);
        }

        @Override
        public TaskConfig build() {
            return buildPrototype();
        }

    }

}
