/*
 * Copyright (c) 2023 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.scheduling;

import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Errors;
import io.helidon.common.Generated;
import io.helidon.common.config.Config;

/**
 * Interface generated from definition. Please add javadoc to the definition interface.
 *
 * @see #builder()
 */
@Generated(value = "io.helidon.builder.processor.BlueprintProcessor", trigger = "io.helidon.scheduling.FixedRateConfigBlueprint")
public interface FixedRateConfig extends FixedRateConfigBlueprint, Prototype.Api, TaskConfig {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static FixedRateConfig.Builder builder() {
        return new FixedRateConfig.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static FixedRateConfig.Builder builder(FixedRateConfig instance) {
        return FixedRateConfig.builder().from(instance);
    }

    /**
     * Create a new instance from configuration.
     *
     * @param config used to configure the new instance
     * @return a new instance configured from configuration
     */
    static FixedRateConfig create(Config config) {
        return FixedRateConfig.builder().config(config).buildPrototype();
    }

    /**
     * Fluent API builder base for {@link FixedRate}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends FixedRateConfig.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends FixedRateConfig> extends TaskConfig.BuilderBase<BUILDER, PROTOTYPE> implements Prototype.ConfiguredBuilder<BUILDER, PROTOTYPE> {

        private Config config;
        private FixedRate.DelayType delayType = FixedRate.DelayType.SINCE_PREVIOUS_START;
        private Long delay;
        private long initialDelay = 0L;
        private ScheduledConsumer<FixedRateInvocation> task;
        private TimeUnit timeUnit = TimeUnit.SECONDS;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(FixedRateConfig prototype) {
            super.from(prototype);
            initialDelay(prototype.initialDelay());
            delay(prototype.delay());
            delayType(prototype.delayType());
            task(prototype.task());
            timeUnit(prototype.timeUnit());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(FixedRateConfig.BuilderBase<?, ?> builder) {
            super.from(builder);
            initialDelay(builder.initialDelay());
            builder.delay().ifPresent(this::delay);
            delayType(builder.delayType());
            builder.task().ifPresent(this::task);
            timeUnit(builder.timeUnit());
            return self();
        }

        /**
         * Update builder from configuration (node of this type).
         * If a value is present in configuration, it would override currently configured values.
         *
         * @param config configuration instance used to obtain values to update this builder
         * @return updated builder instance
         */
        @Override
        public BUILDER config(Config config) {
            Objects.requireNonNull(config);
            this.config = config;
            super.config(config);
            config.get("initial-delay").as(Long.class).ifPresent(this::initialDelay);
            config.get("delay").as(Long.class).ifPresent(this::delay);
            config.get("delay-type").as(FixedRate.DelayType.class).ifPresent(this::delayType);
            config.get("time-unit").as(TimeUnit.class).ifPresent(this::timeUnit);
            return self();
        }

        /**
         * Initial delay of the first invocation. Time unit is by default {@link java.util.concurrent.TimeUnit#SECONDS},
         * can be specified with
         * {@link io.helidon.scheduling.FixedRateConfig.Builder#timeUnit(java.util.concurrent.TimeUnit) timeUnit()}.
         *
         * @param initialDelay initial delay value
         * @return updated builder instance
         * @see #initialDelay()
         */
        public BUILDER initialDelay(long initialDelay) {
            this.initialDelay = initialDelay;
            return self();
        }

        /**
         * Fixed rate delay between each invocation. Time unit is by default {@link java.util.concurrent.TimeUnit#SECONDS},
         * can be specified with {@link io.helidon.scheduling.FixedRateConfig.Builder#timeUnit(java.util.concurrent.TimeUnit)}.
         *
         * @param delay delay between each invocation
         * @return updated builder instance
         * @see #delay()
         */
        public BUILDER delay(long delay) {
            this.delay = delay;
            return self();
        }

        /**
         * Configure whether the delay between the invocations should be calculated from the time when previous task started or ended.
         * Delay type is by default {@link FixedRate.DelayType#SINCE_PREVIOUS_START}.
         *
         * @param delayType delay type
         * @return updated builder instance
         * @see #delayType()
         */
        public BUILDER delayType(FixedRate.DelayType delayType) {
            Objects.requireNonNull(delayType);
            this.delayType = delayType;
            return self();
        }

        /**
         * Task to be scheduled for execution.
         *
         * @param task scheduled for execution
         * @return updated builder instance
         * @see #task()
         */
        public BUILDER task(ScheduledConsumer<FixedRateInvocation> task) {
            Objects.requireNonNull(task);
            this.task = task;
            return self();
        }

        /**
         * {@link java.util.concurrent.TimeUnit TimeUnit} used for interpretation of values provided with
         * {@link io.helidon.scheduling.FixedRateConfig.Builder#delay(long)}
         * and {@link io.helidon.scheduling.FixedRateConfig.Builder#initialDelay(long)}.
         *
         * @param timeUnit time unit for interpreting values
         *                 in {@link io.helidon.scheduling.FixedRateConfig.Builder#delay(long)}
         *                 and {@link io.helidon.scheduling.FixedRateConfig.Builder#initialDelay(long)}
         * @return updated builder instance
         * @see #timeUnit()
         */
        public BUILDER timeUnit(TimeUnit timeUnit) {
            Objects.requireNonNull(timeUnit);
            this.timeUnit = timeUnit;
            return self();
        }

        /**
         * Initial delay of the first invocation. Time unit is by default {@link java.util.concurrent.TimeUnit#SECONDS},
         * can be specified with
         * {@link io.helidon.scheduling.FixedRateConfig.Builder#timeUnit(java.util.concurrent.TimeUnit) timeUnit()}.
         *
         * @return the initial delay
         */
        public long initialDelay() {
            return initialDelay;
        }

        /**
         * Fixed rate delay between each invocation. Time unit is by default {@link java.util.concurrent.TimeUnit#SECONDS},
         * can be specified with {@link io.helidon.scheduling.FixedRateConfig.Builder#timeUnit(java.util.concurrent.TimeUnit)}.
         *
         * @return the delay
         */
        public Optional<Long> delay() {
            return Optional.ofNullable(delay);
        }

        /**
         * Configure whether the delay between the invocations should be calculated from the time when previous task started or ended.
         * Delay type is by default {@link FixedRate.DelayType#SINCE_PREVIOUS_START}.
         *
         * @return the delay type
         */
        public FixedRate.DelayType delayType() {
            return delayType;
        }

        /**
         * Task to be scheduled for execution.
         *
         * @return the task
         */
        public Optional<ScheduledConsumer<FixedRateInvocation>> task() {
            return Optional.ofNullable(task);
        }

        /**
         * {@link java.util.concurrent.TimeUnit TimeUnit} used for interpretation of values provided with
         * {@link io.helidon.scheduling.FixedRateConfig.Builder#delay(long)}
         * and {@link io.helidon.scheduling.FixedRateConfig.Builder#initialDelay(long)}.
         *
         * @return the time unit
         */
        public TimeUnit timeUnit() {
            return timeUnit;
        }

        /**
         * If this instance was configured, this would be the config instance used.
         *
         * @return config node used to configure this builder, or empty if not configured
         */
        public Optional<Config> config() {
            return Optional.ofNullable(config);
        }

        @Override
        public String toString() {
            return "FixedRateConfigBuilder{"
                    + "initialDelay=" + initialDelay + ","
                    + "delay=" + delay + ","
                    + "delayType=" + delayType + ","
                    + "task=" + task + ","
                    + "timeUnit=" + timeUnit
                    + "};"
                    + super.toString();
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
            super.preBuildPrototype();
            new TaskConfigDecorator().decorate(this);
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
            super.validatePrototype();
            Errors.Collector collector = Errors.collector();
            if (delay == null) {
                collector.fatal(getClass(), "Property \"delay\" is required, but not set");
            }
            if (task == null) {
                collector.fatal(getClass(), "Property \"task\" must not be null, but not set");
            }
            collector.collect().checkValid();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class FixedRateConfigImpl extends TaskConfigImpl implements FixedRateConfig, Supplier<FixedRate> {

            private final FixedRate.DelayType delayType;
            private final long delay;
            private final long initialDelay;
            private final ScheduledConsumer<FixedRateInvocation> task;
            private final TimeUnit timeUnit;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected FixedRateConfigImpl(FixedRateConfig.BuilderBase<?, ?> builder) {
                super(builder);
                this.initialDelay = builder.initialDelay();
                this.delay = builder.delay().get();
                this.delayType = builder.delayType();
                this.task = builder.task().get();
                this.timeUnit = builder.timeUnit();
            }

            @Override
            public FixedRate build() {
                return FixedRate.create(this);
            }

            @Override
            public FixedRate get() {
                return build();
            }

            @Override
            public long initialDelay() {
                return initialDelay;
            }

            @Override
            public long delay() {
                return delay;
            }

            @Override
            public FixedRate.DelayType delayType() {
                return delayType;
            }

            @Override
            public ScheduledConsumer<FixedRateInvocation> task() {
                return task;
            }

            @Override
            public TimeUnit timeUnit() {
                return timeUnit;
            }

            @Override
            public String toString() {
                return "FixedRateConfig{"
                        + "initialDelay=" + initialDelay + ","
                        + "delay=" + delay + ","
                        + "delayType=" + delayType + ","
                        + "task=" + task + ","
                        + "timeUnit=" + timeUnit
                        + "};"
                        + super.toString();
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof FixedRateConfig other)) {
                    return false;
                }
                return super.equals(other)
                        && initialDelay == other.initialDelay()
                        && delay == other.delay()
                        && Objects.equals(delayType, other.delayType())
                        && Objects.equals(task, other.task())
                        && Objects.equals(timeUnit, other.timeUnit());
            }

            @Override
            public int hashCode() {
                return 31 * super.hashCode() + Objects.hash(initialDelay, delay, delayType, task, timeUnit);
            }

        }

    }

    /**
     * Fluent API builder for {@link FixedRate}.
     */
    class Builder extends FixedRateConfig.BuilderBase<FixedRateConfig.Builder, FixedRateConfig> implements io.helidon.common.Builder<FixedRateConfig.Builder, FixedRate> {

        private Builder() {
        }

        @Override
        public FixedRateConfig buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new FixedRateConfigImpl(this);
        }

        @Override
        public FixedRate build() {
            return FixedRate.create(this.buildPrototype());
        }

    }

}
