/*
 * Copyright (c) 2023 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.scheduling;

import java.util.Objects;
import java.util.Optional;
import java.util.function.Supplier;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Errors;
import io.helidon.common.Generated;
import io.helidon.common.config.Config;

/**
 * Interface generated from definition. Please add javadoc to the definition interface.
 *
 * @see #builder()
 */
@Generated(value = "io.helidon.builder.processor.BlueprintProcessor", trigger = "io.helidon.scheduling.CronConfigBlueprint")
public interface CronConfig extends CronConfigBlueprint, Prototype.Api, TaskConfig {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static CronConfig.Builder builder() {
        return new CronConfig.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static CronConfig.Builder builder(CronConfig instance) {
        return CronConfig.builder().from(instance);
    }

    /**
     * Create a new instance from configuration.
     *
     * @param config used to configure the new instance
     * @return a new instance configured from configuration
     */
    static CronConfig create(Config config) {
        return CronConfig.builder().config(config).buildPrototype();
    }

    /**
     * Fluent API builder base for {@link Cron}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends CronConfig.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends CronConfig> extends TaskConfig.BuilderBase<BUILDER, PROTOTYPE> implements Prototype.ConfiguredBuilder<BUILDER, PROTOTYPE> {

        private boolean concurrentExecution = true;
        private Config config;
        private ScheduledConsumer<CronInvocation> task;
        private String expression;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(CronConfig prototype) {
            super.from(prototype);
            expression(prototype.expression());
            concurrentExecution(prototype.concurrentExecution());
            task(prototype.task());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(CronConfig.BuilderBase<?, ?> builder) {
            super.from(builder);
            builder.expression().ifPresent(this::expression);
            concurrentExecution(builder.concurrentExecution());
            builder.task().ifPresent(this::task);
            return self();
        }

        /**
         * Update builder from configuration (node of this type).
         * If a value is present in configuration, it would override currently configured values.
         *
         * @param config configuration instance used to obtain values to update this builder
         * @return updated builder instance
         */
        @Override
        public BUILDER config(Config config) {
            Objects.requireNonNull(config);
            this.config = config;
            super.config(config);
            config.get("expression").as(String.class).ifPresent(this::expression);
            config.get("concurrent").as(Boolean.class).ifPresent(this::concurrentExecution);
            return self();
        }

        /**
         * Cron expression for specifying period of execution.
         * <p>
         * <b>Examples:</b>
         * <ul>
         * <li>{@code 0/2 * * * * ? *} - Every 2 seconds</li>
         * <li>{@code 0 45 9 ? * *} - Every day at 9:45</li>
         * <li>{@code 0 15 8 ? * MON-FRI} - Every workday at 8:15</li>
         * </ul>
         *
         * @param expression cron expression
         * @return updated builder instance
         * @see #expression()
         */
        public BUILDER expression(String expression) {
            Objects.requireNonNull(expression);
            this.expression = expression;
            return self();
        }

        /**
         * Allow concurrent execution if previous task didn't finish before next execution.
         * Default value is {@code true}.
         *
         * @param concurrentExecution true for allow concurrent execution.
         * @return updated builder instance
         * @see #concurrentExecution()
         */
        public BUILDER concurrentExecution(boolean concurrentExecution) {
            this.concurrentExecution = concurrentExecution;
            return self();
        }

        /**
         * Task to be scheduled for execution.
         *
         * @param task scheduled for execution
         * @return updated builder instance
         * @see #task()
         */
        public BUILDER task(ScheduledConsumer<CronInvocation> task) {
            Objects.requireNonNull(task);
            this.task = task;
            return self();
        }

        /**
         * Cron expression for specifying period of execution.
         * <p>
         * <b>Examples:</b>
         * <ul>
         * <li>{@code 0/2 * * * * ? *} - Every 2 seconds</li>
         * <li>{@code 0 45 9 ? * *} - Every day at 9:45</li>
         * <li>{@code 0 15 8 ? * MON-FRI} - Every workday at 8:15</li>
         * </ul>
         *
         * @return the expression
         */
        public Optional<String> expression() {
            return Optional.ofNullable(expression);
        }

        /**
         * Allow concurrent execution if previous task didn't finish before next execution.
         * Default value is {@code true}.
         *
         * @return the concurrent execution
         */
        public boolean concurrentExecution() {
            return concurrentExecution;
        }

        /**
         * Task to be scheduled for execution.
         *
         * @return the task
         */
        public Optional<ScheduledConsumer<CronInvocation>> task() {
            return Optional.ofNullable(task);
        }

        /**
         * If this instance was configured, this would be the config instance used.
         *
         * @return config node used to configure this builder, or empty if not configured
         */
        public Optional<Config> config() {
            return Optional.ofNullable(config);
        }

        @Override
        public String toString() {
            return "CronConfigBuilder{"
                    + "expression=" + expression + ","
                    + "concurrentExecution=" + concurrentExecution + ","
                    + "task=" + task
                    + "};"
                    + super.toString();
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
            super.preBuildPrototype();
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
            super.validatePrototype();
            Errors.Collector collector = Errors.collector();
            if (expression == null) {
                collector.fatal(getClass(), "Property \"expression\" is required, but not set");
            }
            if (task == null) {
                collector.fatal(getClass(), "Property \"task\" must not be null, but not set");
            }
            collector.collect().checkValid();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class CronConfigImpl extends TaskConfigImpl implements CronConfig, Supplier<Cron> {

            private final boolean concurrentExecution;
            private final ScheduledConsumer<CronInvocation> task;
            private final String expression;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected CronConfigImpl(CronConfig.BuilderBase<?, ?> builder) {
                super(builder);
                this.expression = builder.expression().get();
                this.concurrentExecution = builder.concurrentExecution();
                this.task = builder.task().get();
            }

            @Override
            public Cron build() {
                return Cron.create(this);
            }

            @Override
            public Cron get() {
                return build();
            }

            @Override
            public String expression() {
                return expression;
            }

            @Override
            public boolean concurrentExecution() {
                return concurrentExecution;
            }

            @Override
            public ScheduledConsumer<CronInvocation> task() {
                return task;
            }

            @Override
            public String toString() {
                return "CronConfig{"
                        + "expression=" + expression + ","
                        + "concurrentExecution=" + concurrentExecution + ","
                        + "task=" + task
                        + "};"
                        + super.toString();
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof CronConfig other)) {
                    return false;
                }
                return super.equals(other)
                        && Objects.equals(expression, other.expression())
                        && concurrentExecution == other.concurrentExecution()
                        && Objects.equals(task, other.task());
            }

            @Override
            public int hashCode() {
                return 31 * super.hashCode() + Objects.hash(expression, concurrentExecution, task);
            }

        }

    }

    /**
     * Fluent API builder for {@link Cron}.
     */
    class Builder extends CronConfig.BuilderBase<CronConfig.Builder, CronConfig> implements io.helidon.common.Builder<CronConfig.Builder, Cron> {

        private Builder() {
        }

        @Override
        public CronConfig buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new CronConfigImpl(this);
        }

        @Override
        public Cron build() {
            return Cron.create(this.buildPrototype());
        }

    }

}
