/*
 * Copyright (c) 2020, 2022 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.helidon.reactive.media.jsonp;

import java.nio.charset.Charset;
import java.util.concurrent.Flow.Publisher;

import io.helidon.common.GenericType;
import io.helidon.common.http.DataChunk;
import io.helidon.common.http.HttpMediaType;
import io.helidon.common.mapper.Mapper;
import io.helidon.common.reactive.Single;
import io.helidon.reactive.media.common.CharBuffer;
import io.helidon.reactive.media.common.MessageBodyWriter;
import io.helidon.reactive.media.common.MessageBodyWriterContext;

import jakarta.json.JsonStructure;
import jakarta.json.JsonWriter;
import jakarta.json.JsonWriterFactory;

/**
 * Message body writer for {@link JsonStructure} sub-classes (JSON-P).
 */
class JsonpBodyWriter implements MessageBodyWriter<JsonStructure> {

    private final JsonWriterFactory jsonWriterFactory;

    JsonpBodyWriter(JsonWriterFactory jsonWriterFactory) {
        this.jsonWriterFactory = jsonWriterFactory;
    }

    @Override
    public PredicateResult accept(GenericType<?> type, MessageBodyWriterContext context) {
        return PredicateResult.supports(JsonStructure.class, type);
    }

    @Override
    public Publisher<DataChunk> write(Single<? extends JsonStructure> content,
                                      GenericType<? extends JsonStructure> type,
                                      MessageBodyWriterContext context) {

        HttpMediaType contentType = context.findAccepted(HttpMediaType.JSON_PREDICATE, HttpMediaType.APPLICATION_JSON);
        context.contentType(contentType);
        return content.map(new JsonStructureToChunks(jsonWriterFactory, context.charset()));
    }

    static final class JsonStructureToChunks implements Mapper<JsonStructure, DataChunk> {
        private final JsonWriterFactory factory;
        private final Charset charset;
        private boolean flush = false;

        JsonStructureToChunks(JsonWriterFactory factory, Charset charset) {
            this.factory = factory;
            this.charset = charset;
        }

        JsonStructureToChunks(boolean flush, JsonWriterFactory factory, Charset charset) {
            this.factory = factory;
            this.charset = charset;
            this.flush = flush;
        }

        @Override
        public DataChunk map(JsonStructure item) {
            CharBuffer buffer = new CharBuffer();
            try (JsonWriter writer = factory.createWriter(buffer)) {
                writer.write(item);
                return DataChunk.create(flush, buffer.encode(charset));
            }
        }
    }
}
