/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.microprofile.faulttolerance;

import com.netflix.config.ConfigurationManager;
import io.helidon.microprofile.faulttolerance.FaultToleranceCommand;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Logger;
import org.eclipse.microprofile.faulttolerance.CircuitBreaker;

public class CircuitBreakerHelper {
    private static final Logger LOGGER = Logger.getLogger(CircuitBreakerHelper.class.getName());
    private static final String FORCE_OPEN = "hystrix.command.%s.circuitBreaker.forceOpen";
    private static final String FORCE_CLOSED = "hystrix.command.%s.circuitBreaker.forceClosed";
    private static final Map<String, CommandData> COMMAND_STATS = new ConcurrentHashMap<String, CommandData>();
    private final FaultToleranceCommand command;
    private final CircuitBreaker circuitBreaker;

    CircuitBreakerHelper(FaultToleranceCommand command, CircuitBreaker circuitBreaker) {
        this.command = command;
        this.circuitBreaker = circuitBreaker;
    }

    private CommandData getCommandData() {
        return COMMAND_STATS.computeIfAbsent(this.command.getCommandKey().toString(), d -> new CommandData(this.circuitBreaker.requestVolumeThreshold()));
    }

    void resetCommandData() {
        ReentrantLock lock = this.getCommandData().getLock();
        if (lock.isLocked()) {
            lock.unlock();
        }
        COMMAND_STATS.remove(this.command.getCommandKey().toString());
        LOGGER.info("Discarded command data for " + this.command.getCommandKey());
    }

    void pushResult(boolean result) {
        this.getCommandData().pushResult(result);
    }

    long getCurrentStateNanos() {
        return this.getCommandData().getCurrentStateNanos();
    }

    double getFailureRatio() {
        return this.getCommandData().getFailureRatio();
    }

    State getState() {
        return this.getCommandData().getState();
    }

    void setState(State newState) {
        this.getCommandData().setState(newState);
        if (newState == State.OPEN_MP) {
            this.openBreaker();
        } else {
            this.closeBreaker();
        }
        LOGGER.info("Circuit breaker for " + this.command.getCommandKey() + " now in state " + (Object)((Object)this.getState()));
    }

    int getSuccessCount() {
        return this.getCommandData().getSuccessCount();
    }

    void incSuccessCount() {
        this.getCommandData().incSuccessCount();
    }

    void lock() {
        this.getCommandData().getLock().lock();
    }

    void unlock() {
        this.getCommandData().getLock().unlock();
    }

    long getInStateNanos(State queryState) {
        return this.getCommandData().getInStateNanos(queryState);
    }

    private void openBreaker() {
        if (!this.command.isCircuitBreakerOpen()) {
            ConfigurationManager.getConfigInstance().setProperty(String.format(FORCE_OPEN, this.command.getCommandKey()), (Object)"true");
        }
    }

    private void closeBreaker() {
        if (this.command.isCircuitBreakerOpen()) {
            ConfigurationManager.getConfigInstance().setProperty(String.format(FORCE_OPEN, this.command.getCommandKey()), (Object)"false");
            ConfigurationManager.getConfigInstance().setProperty(String.format(FORCE_CLOSED, this.command.getCommandKey()), (Object)"true");
        }
    }

    static class CommandData {
        private int size;
        private final boolean[] results;
        private State state = State.CLOSED_MP;
        private int successCount;
        private long[] inStateNanos = new long[State.values().length];
        private long lastNanosRead;
        private ReentrantLock lock = new ReentrantLock();

        CommandData(int capacity) {
            this.results = new boolean[capacity];
            this.size = 0;
            this.successCount = 0;
            this.lastNanosRead = System.nanoTime();
        }

        ReentrantLock getLock() {
            return this.lock;
        }

        State getState() {
            return this.state;
        }

        long getCurrentStateNanos() {
            return System.nanoTime() - this.lastNanosRead;
        }

        void setState(State newState) {
            if (this.state != newState) {
                this.updateStateNanos(this.state);
                if (newState == State.HALF_OPEN_MP) {
                    this.successCount = 0;
                }
                this.state = newState;
            }
        }

        long getInStateNanos(State queryState) {
            if (this.state == queryState) {
                this.updateStateNanos(this.state);
            }
            return this.inStateNanos[queryState.value];
        }

        private void updateStateNanos(State state) {
            long currentNanos = System.nanoTime();
            int n = state.value;
            this.inStateNanos[n] = this.inStateNanos[n] + (currentNanos - this.lastNanosRead);
            this.lastNanosRead = currentNanos;
        }

        int getSuccessCount() {
            return this.successCount;
        }

        void incSuccessCount() {
            ++this.successCount;
        }

        boolean isAtCapacity() {
            return this.size == this.results.length;
        }

        void pushResult(boolean result) {
            if (this.isAtCapacity()) {
                this.shift();
            }
            this.results[this.size++] = result;
        }

        double getSuccessRatio() {
            if (this.isAtCapacity()) {
                int success = 0;
                for (int k = 0; k < this.size; ++k) {
                    if (!this.results[k]) continue;
                    ++success;
                }
                return (double)success / (double)this.size;
            }
            return -1.0;
        }

        double getFailureRatio() {
            double successRatio = this.getSuccessRatio();
            return successRatio >= 0.0 ? 1.0 - successRatio : -1.0;
        }

        private void shift() {
            if (this.size > 0) {
                for (int k = 0; k < this.size - 1; ++k) {
                    this.results[k] = this.results[k + 1];
                }
                --this.size;
            }
        }
    }

    static enum State {
        CLOSED_MP(0),
        HALF_OPEN_MP(1),
        OPEN_MP(2);

        private int value;

        private State(int value) {
            this.value = value;
        }
    }
}

