/*
 * Copyright (c) 2024 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.metrics.api;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Supplier;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Errors;
import io.helidon.common.Generated;
import io.helidon.common.config.Config;

/**
 * Configuration settings for metrics.
 *
 * <h2>Scope handling configuration</h2>
 * Helidon allows developers to associate a scope with each meter. The {@value SCOPE_CONFIG_KEY} subsection of the
 * {@value METRICS_CONFIG_KEY} configuration controls
 * <ul>
 *     <li>the default scope value to use if a meter is registered without an explicit scope setting, and</li>
 *     <li>whether and how Helidon records each meter's scope as a tag in the underlying implementation meter registry.
 *     <p>
 *         Specifically, users can specify whether scope tags are used at all and, if so, what tag name to use.
 *     </li>
 * </ul>
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.codegen.BuilderCodegen", trigger = "io.helidon.metrics.api.MetricsConfigBlueprint")
public interface MetricsConfig extends MetricsConfigBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static MetricsConfig.Builder builder() {
        return new MetricsConfig.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static MetricsConfig.Builder builder(MetricsConfig instance) {
        return MetricsConfig.builder().from(instance);
    }

    /**
     * Create a new instance from configuration.
     *
     * @param config used to configure the new instance
     * @return a new instance configured from configuration
     */
    static MetricsConfig create(Config config) {
        return MetricsConfig.builder().config(config).buildPrototype();
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static MetricsConfig create() {
        return MetricsConfig.builder().buildPrototype();
    }

    /**
     * Looks up a single config value within the metrics configuration by config key.
     *
     * @param key           config key to fetch
     * @return config value
     */
    Optional<String> lookupConfig(String key);

    /**
     * Reports whether the specified scope is enabled, according to any scope configuration that
     * is part of this metrics configuration.
     *
     * @param scope         scope name
     * @return true if the scope as a whole is enabled; false otherwise
     */
    boolean isScopeEnabled(String scope);

    /**
     * Reports whether the specified meter within the indicated scope is enabled, according to the metrics configuration.
     *
     * @param name          meter name
     * @param targetScope   scope within which to check
     * @return whether the meter is enabled
     */
    boolean isMeterEnabled(String name, String targetScope);

    /**
     * Fluent API builder base for {@link MetricsConfig}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends MetricsConfig.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends MetricsConfig> implements Prototype.ConfiguredBuilder<BUILDER, PROTOTYPE> {

        private final List<Tag> tags = new ArrayList<>();
        private final List<String> roles = new ArrayList<>(List.of("observe"));
        private boolean enabled = true;
        private boolean isRolesMutated;
        private boolean isTagsMutated;
        private boolean permitAll = true;
        private boolean restRequestEnabled;
        private Config config;
        private KeyPerformanceIndicatorMetricsConfig keyPerformanceIndicatorMetricsConfig;
        private ScopingConfig scoping;
        private String appName;
        private String appTagName;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance. This method disables automatic service discovery.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(MetricsConfig prototype) {
            enabled(prototype.enabled());
            permitAll(prototype.permitAll());
            if (!isRolesMutated) {
                roles.clear();
            }
            addRoles(prototype.roles());
            keyPerformanceIndicatorMetricsConfig(prototype.keyPerformanceIndicatorMetricsConfig());
            if (!isTagsMutated) {
                tags.clear();
            }
            addTags(prototype.tags());
            appName(prototype.appName());
            appTagName(prototype.appTagName());
            scoping(prototype.scoping());
            restRequestEnabled(prototype.restRequestEnabled());
            this.config = prototype.config();
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(MetricsConfig.BuilderBase<?, ?> builder) {
            enabled(builder.enabled());
            permitAll(builder.permitAll());
            if (isRolesMutated) {
                if (builder.isRolesMutated) {
                    addRoles(builder.roles);
                }
            } else {
                roles.clear();
                addRoles(builder.roles);
            }
            builder.keyPerformanceIndicatorMetricsConfig().ifPresent(this::keyPerformanceIndicatorMetricsConfig);
            if (isTagsMutated) {
                if (builder.isTagsMutated) {
                    addTags(builder.tags);
                }
            } else {
                tags.clear();
                addTags(builder.tags);
            }
            builder.appName().ifPresent(this::appName);
            builder.appTagName().ifPresent(this::appTagName);
            builder.scoping().ifPresent(this::scoping);
            restRequestEnabled(builder.restRequestEnabled());
            builder.config().ifPresent(this::config);
            return self();
        }

        /**
         * Update builder from configuration (node of this type).
         * If a value is present in configuration, it would override currently configured values.
         *
         * @param config configuration instance used to obtain values to update this builder
         * @return updated builder instance
         */
        @Override
        public BUILDER config(Config config) {
            Objects.requireNonNull(config);
            this.config = config;
            config.get("enabled").as(Boolean.class).ifPresent(this::enabled);
            config.get("permit-all").as(Boolean.class).ifPresent(this::permitAll);
            config.get("roles").asList(String.class).ifPresent(this::roles);
            config.get("key-performance-indicators").map(KeyPerformanceIndicatorMetricsConfig::create).ifPresent(this::keyPerformanceIndicatorMetricsConfig);
            config.get("tags").map(MetricsConfigBlueprint::createTags).ifPresent(this::tags);
            config.get("app-name").as(String.class).ifPresent(this::appName);
            config.get("app-tag-name").as(String.class).ifPresent(this::appTagName);
            config.get("scoping").map(ScopingConfig::create).ifPresent(this::scoping);
            config.get("rest-request-enabled").as(Boolean.class).ifPresent(this::restRequestEnabled);
            return self();
        }

        /**
         * Whether metrics functionality is enabled.
         *
         * @param enabled if metrics are configured to be enabled
         * @return updated builder instance
         * @see #enabled()
         */
        public BUILDER enabled(boolean enabled) {
            this.enabled = enabled;
            return self();
        }

        /**
         * Whether to allow anybody to access the endpoint.
         *
         * @param permitAll whether to permit access to metrics endpoint to anybody, defaults to {@code true}
         * @return updated builder instance
         * @see #roles()
         * @see #permitAll()
         */
        public BUILDER permitAll(boolean permitAll) {
            this.permitAll = permitAll;
            return self();
        }

        /**
         * Hints for role names the user is expected to be in.
         *
         * @param roles list of hints
         * @return updated builder instance
         * @see #roles()
         */
        public BUILDER roles(List<? extends String> roles) {
            Objects.requireNonNull(roles);
            isRolesMutated = true;
            this.roles.clear();
            this.roles.addAll(roles);
            return self();
        }

        /**
         * Hints for role names the user is expected to be in.
         *
         * @param roles list of hints
         * @return updated builder instance
         * @see #roles()
         */
        public BUILDER addRoles(List<? extends String> roles) {
            Objects.requireNonNull(roles);
            isRolesMutated = true;
            this.roles.addAll(roles);
            return self();
        }

        /**
         * Key performance indicator metrics settings.
         *
         * @param keyPerformanceIndicatorMetricsConfig key performance indicator metrics settings
         * @return updated builder instance
         * @see #keyPerformanceIndicatorMetricsConfig()
         */
        public BUILDER keyPerformanceIndicatorMetricsConfig(KeyPerformanceIndicatorMetricsConfig keyPerformanceIndicatorMetricsConfig) {
            Objects.requireNonNull(keyPerformanceIndicatorMetricsConfig);
            this.keyPerformanceIndicatorMetricsConfig = keyPerformanceIndicatorMetricsConfig;
            return self();
        }

        /**
         * Key performance indicator metrics settings.
         *
         * @param consumer consumer of builder for
         *                 key performance indicator metrics settings
         * @return updated builder instance
         * @see #keyPerformanceIndicatorMetricsConfig()
         */
        public BUILDER keyPerformanceIndicatorMetricsConfig(Consumer<KeyPerformanceIndicatorMetricsConfig.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = KeyPerformanceIndicatorMetricsConfig.builder();
            consumer.accept(builder);
            this.keyPerformanceIndicatorMetricsConfig(builder.build());
            return self();
        }

        /**
         * Key performance indicator metrics settings.
         *
         * @param supplier supplier of
         *                 key performance indicator metrics settings
         * @return updated builder instance
         * @see #keyPerformanceIndicatorMetricsConfig()
         */
        public BUILDER keyPerformanceIndicatorMetricsConfig(Supplier<? extends KeyPerformanceIndicatorMetricsConfig> supplier) {
            Objects.requireNonNull(supplier);
            this.keyPerformanceIndicatorMetricsConfig(supplier.get());
            return self();
        }

        /**
         * Global tags.
         *
         * @param tags name/value pairs for global tags
         * @return updated builder instance
         * @see #tags()
         */
        public BUILDER tags(List<? extends Tag> tags) {
            Objects.requireNonNull(tags);
            isTagsMutated = true;
            this.tags.clear();
            this.tags.addAll(tags);
            return self();
        }

        /**
         * Global tags.
         *
         * @param tags name/value pairs for global tags
         * @return updated builder instance
         * @see #tags()
         */
        public BUILDER addTags(List<? extends Tag> tags) {
            Objects.requireNonNull(tags);
            isTagsMutated = true;
            this.tags.addAll(tags);
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #appName()
         */
        public BUILDER clearAppName() {
            this.appName = null;
            return self();
        }

        /**
         * Value for the application tag to be added to each meter ID.
         *
         * @param appName application tag value
         * @return updated builder instance
         * @see #appName()
         */
        public BUILDER appName(String appName) {
            Objects.requireNonNull(appName);
            this.appName = appName;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #appTagName()
         */
        public BUILDER clearAppTagName() {
            this.appTagName = null;
            return self();
        }

        /**
         * Name for the application tag to be added to each meter ID.
         *
         * @param appTagName application tag name
         * @return updated builder instance
         * @see #appTagName()
         */
        public BUILDER appTagName(String appTagName) {
            Objects.requireNonNull(appTagName);
            this.appTagName = appTagName;
            return self();
        }

        /**
         * Settings related to scoping management.
         *
         * @param scoping scoping settings
         * @return updated builder instance
         * @see #scoping()
         */
        public BUILDER scoping(ScopingConfig scoping) {
            Objects.requireNonNull(scoping);
            this.scoping = scoping;
            return self();
        }

        /**
         * Settings related to scoping management.
         *
         * @param consumer consumer of builder for
         *                 scoping settings
         * @return updated builder instance
         * @see #scoping()
         */
        public BUILDER scoping(Consumer<ScopingConfig.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = ScopingConfig.builder();
            consumer.accept(builder);
            this.scoping(builder.build());
            return self();
        }

        /**
         * Settings related to scoping management.
         *
         * @param supplier supplier of
         *                 scoping settings
         * @return updated builder instance
         * @see #scoping()
         */
        public BUILDER scoping(Supplier<? extends ScopingConfig> supplier) {
            Objects.requireNonNull(supplier);
            this.scoping(supplier.get());
            return self();
        }

        /**
         * Whether automatic REST request metrics should be measured.
         *
         * @param restRequestEnabled true/false
         * @return updated builder instance
         * @see #restRequestEnabled()
         */
        public BUILDER restRequestEnabled(boolean restRequestEnabled) {
            this.restRequestEnabled = restRequestEnabled;
            return self();
        }

        /**
         * Whether metrics functionality is enabled.
         *
         * @return the enabled
         */
        public boolean enabled() {
            return enabled;
        }

        /**
         * Whether to allow anybody to access the endpoint.
         *
         * @return the permit all
         * @see #roles()
         * @see #permitAll()
         */
        public boolean permitAll() {
            return permitAll;
        }

        /**
         * Hints for role names the user is expected to be in.
         *
         * @return the roles
         */
        public List<String> roles() {
            return roles;
        }

        /**
         * Key performance indicator metrics settings.
         *
         * @return the key performance indicator metrics config
         */
        public Optional<KeyPerformanceIndicatorMetricsConfig> keyPerformanceIndicatorMetricsConfig() {
            return Optional.ofNullable(keyPerformanceIndicatorMetricsConfig);
        }

        /**
         * Global tags.
         *
         * @return the tags
         */
        public List<Tag> tags() {
            return tags;
        }

        /**
         * Value for the application tag to be added to each meter ID.
         *
         * @return the app name
         */
        public Optional<String> appName() {
            return Optional.ofNullable(appName);
        }

        /**
         * Name for the application tag to be added to each meter ID.
         *
         * @return the app tag name
         */
        public Optional<String> appTagName() {
            return Optional.ofNullable(appTagName);
        }

        /**
         * Settings related to scoping management.
         *
         * @return the scoping
         */
        public Optional<ScopingConfig> scoping() {
            return Optional.ofNullable(scoping);
        }

        /**
         * Whether automatic REST request metrics should be measured.
         *
         * @return the rest request enabled
         */
        public boolean restRequestEnabled() {
            return restRequestEnabled;
        }

        /**
         * If this instance was configured, this would be the config instance used.
         *
         * @return config node used to configure this builder, or empty if not configured
         */
        public Optional<Config> config() {
            return Optional.ofNullable(config);
        }

        @Override
        public String toString() {
            return "MetricsConfigBuilder{"
                    + "enabled=" + enabled + ","
                    + "permitAll=" + permitAll + ","
                    + "roles=" + roles + ","
                    + "keyPerformanceIndicatorMetricsConfig=" + keyPerformanceIndicatorMetricsConfig + ","
                    + "tags=" + tags + ","
                    + "appName=" + appName + ","
                    + "appTagName=" + appTagName + ","
                    + "scoping=" + scoping + ","
                    + "restRequestEnabled=" + restRequestEnabled
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
            new MetricsConfigSupport.BuilderDecorator().decorate(this);
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
            Errors.Collector collector = Errors.collector();
            if (keyPerformanceIndicatorMetricsConfig == null) {
                collector.fatal(getClass(), "Property \"key-performance-indicators\" must not be null, but not set");
            }
            if (scoping == null) {
                collector.fatal(getClass(), "Property \"scoping\" must not be null, but not set");
            }
            if (config == null) {
                collector.fatal(getClass(), "Property \"config\" must not be null, but not set");
            }
            collector.collect().checkValid();
        }

        /**
         * Value for the application tag to be added to each meter ID.
         *
         * @param appName application tag value
         * @return updated builder instance
         * @see #appName()
         */
        BUILDER appName(Optional<String> appName) {
            Objects.requireNonNull(appName);
            this.appName = appName.map(java.lang.String.class::cast).orElse(this.appName);
            return self();
        }

        /**
         * Name for the application tag to be added to each meter ID.
         *
         * @param appTagName application tag name
         * @return updated builder instance
         * @see #appTagName()
         */
        BUILDER appTagName(Optional<String> appTagName) {
            Objects.requireNonNull(appTagName);
            this.appTagName = appTagName.map(java.lang.String.class::cast).orElse(this.appTagName);
            return self();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class MetricsConfigImpl implements MetricsConfig {

            private final boolean enabled;
            private final boolean permitAll;
            private final boolean restRequestEnabled;
            private final Config config;
            private final KeyPerformanceIndicatorMetricsConfig keyPerformanceIndicatorMetricsConfig;
            private final List<Tag> tags;
            private final List<String> roles;
            private final Optional<String> appName;
            private final Optional<String> appTagName;
            private final ScopingConfig scoping;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected MetricsConfigImpl(MetricsConfig.BuilderBase<?, ?> builder) {
                this.enabled = builder.enabled();
                this.permitAll = builder.permitAll();
                this.roles = List.copyOf(builder.roles());
                this.keyPerformanceIndicatorMetricsConfig = builder.keyPerformanceIndicatorMetricsConfig().get();
                this.tags = List.copyOf(builder.tags());
                this.appName = builder.appName();
                this.appTagName = builder.appTagName();
                this.scoping = builder.scoping().get();
                this.restRequestEnabled = builder.restRequestEnabled();
                this.config = builder.config().get();
            }

            @Override
            public Optional<String> lookupConfig(String key) {
                return MetricsConfigSupport.lookupConfig(this, key);
            }

            @Override
            public boolean isScopeEnabled(String scope) {
                return MetricsConfigSupport.isScopeEnabled(this, scope);
            }

            @Override
            public boolean isMeterEnabled(String name, String targetScope) {
                return MetricsConfigSupport.isMeterEnabled(this, name, targetScope);
            }

            @Override
            public boolean enabled() {
                return enabled;
            }

            @Override
            public boolean permitAll() {
                return permitAll;
            }

            @Override
            public List<String> roles() {
                return roles;
            }

            @Override
            public KeyPerformanceIndicatorMetricsConfig keyPerformanceIndicatorMetricsConfig() {
                return keyPerformanceIndicatorMetricsConfig;
            }

            @Override
            public List<Tag> tags() {
                return tags;
            }

            @Override
            public Optional<String> appName() {
                return appName;
            }

            @Override
            public Optional<String> appTagName() {
                return appTagName;
            }

            @Override
            public ScopingConfig scoping() {
                return scoping;
            }

            @Override
            public boolean restRequestEnabled() {
                return restRequestEnabled;
            }

            @Override
            public Config config() {
                return config;
            }

            @Override
            public String toString() {
                return "MetricsConfig{"
                        + "enabled=" + enabled + ","
                        + "permitAll=" + permitAll + ","
                        + "roles=" + roles + ","
                        + "keyPerformanceIndicatorMetricsConfig=" + keyPerformanceIndicatorMetricsConfig + ","
                        + "tags=" + tags + ","
                        + "appName=" + appName + ","
                        + "appTagName=" + appTagName + ","
                        + "scoping=" + scoping + ","
                        + "restRequestEnabled=" + restRequestEnabled
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof MetricsConfig other)) {
                    return false;
                }
                return enabled == other.enabled()
                    && permitAll == other.permitAll()
                    && Objects.equals(roles, other.roles())
                    && Objects.equals(keyPerformanceIndicatorMetricsConfig, other.keyPerformanceIndicatorMetricsConfig())
                    && Objects.equals(tags, other.tags())
                    && Objects.equals(appName, other.appName())
                    && Objects.equals(appTagName, other.appTagName())
                    && Objects.equals(scoping, other.scoping())
                    && restRequestEnabled == other.restRequestEnabled();
            }

            @Override
            public int hashCode() {
                return Objects.hash(enabled, permitAll, roles, keyPerformanceIndicatorMetricsConfig, tags, appName, appTagName, scoping, restRequestEnabled);
            }

        }

    }

    /**
     * Fluent API builder for {@link MetricsConfig}.
     */
    class Builder extends MetricsConfig.BuilderBase<MetricsConfig.Builder, MetricsConfig> implements io.helidon.common.Builder<MetricsConfig.Builder, MetricsConfig> {

        private Builder() {
        }

        @Override
        public MetricsConfig buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new MetricsConfigImpl(this);
        }

        @Override
        public MetricsConfig build() {
            return buildPrototype();
        }

    }

}
