/*
 * Copyright (c) 2024 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.metrics.api;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Generated;
import io.helidon.common.config.Config;

/**
 * Interface generated from definition. Please add javadoc to the definition interface.
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.codegen.BuilderCodegen", trigger = "io.helidon.metrics.api.ScopingConfigBlueprint")
public interface ScopingConfig extends ScopingConfigBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static ScopingConfig.Builder builder() {
        return new ScopingConfig.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static ScopingConfig.Builder builder(ScopingConfig instance) {
        return ScopingConfig.builder().from(instance);
    }

    /**
     * Create a new instance from configuration.
     *
     * @param config used to configure the new instance
     * @return a new instance configured from configuration
     */
    static ScopingConfig create(Config config) {
        return ScopingConfig.builder().config(config).buildPrototype();
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static ScopingConfig create() {
        return ScopingConfig.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link ScopingConfig}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends ScopingConfig.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends ScopingConfig> implements Prototype.ConfiguredBuilder<BUILDER, PROTOTYPE> {

        private final Map<String, ScopeConfig> scopes = new LinkedHashMap<>();
        private Config config;
        private String defaultValue = "application";
        private String tagName = "scope";

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(ScopingConfig prototype) {
            defaultValue(prototype.defaultValue());
            tagName(prototype.tagName());
            addScopes(prototype.scopes());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(ScopingConfig.BuilderBase<?, ?> builder) {
            builder.defaultValue().ifPresent(this::defaultValue);
            builder.tagName().ifPresent(this::tagName);
            addScopes(builder.scopes());
            return self();
        }

        /**
         * Update builder from configuration (node of this type).
         * If a value is present in configuration, it would override currently configured values.
         *
         * @param config configuration instance used to obtain values to update this builder
         * @return updated builder instance
         */
        @Override
        public BUILDER config(Config config) {
            Objects.requireNonNull(config);
            this.config = config;
            config.get("default").as(String.class).ifPresent(this::defaultValue);
            config.get("tag-name").as(String.class).ifPresent(this::tagName);
            config.get("scopes").asNodeList().ifPresent(nodes -> nodes.forEach(node -> scopes.put(node.get("name").asString().orElse(node.name()), node.map(ScopeConfig::create).get())));
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #defaultValue()
         */
        public BUILDER clearDefaultValue() {
            this.defaultValue = null;
            return self();
        }

        /**
         * Default scope value to associate with meters that are registered without an explicit setting; no setting means meters
         * are assigned scope {@value io.helidon.metrics.api.Meter.Scope#DEFAULT}.
         *
         * @param defaultValue default scope value
         * @return updated builder instance
         * @see #defaultValue()
         */
        public BUILDER defaultValue(String defaultValue) {
            Objects.requireNonNull(defaultValue);
            this.defaultValue = defaultValue;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #tagName()
         */
        public BUILDER clearTagName() {
            this.tagName = null;
            return self();
        }

        /**
         * Tag name for storing meter scope values in the underlying implementation meter registry.
         *
         * @param tagName tag name for storing scope values
         * @return updated builder instance
         * @see #tagName()
         */
        public BUILDER tagName(String tagName) {
            Objects.requireNonNull(tagName);
            this.tagName = tagName;
            return self();
        }

        /**
         * This method replaces all values with the new ones.
         *
         * @param scopes scope settings
         * @return updated builder instance
         * @see #scopes()
         */
        public BUILDER scopes(Map<? extends String, ? extends ScopeConfig> scopes) {
            Objects.requireNonNull(scopes);
            this.scopes.clear();
            this.scopes.putAll(scopes);
            return self();
        }

        /**
         * This method keeps existing values, then puts all new values into the map.
         *
         * @param scopes scope settings
         * @return updated builder instance
         * @see #scopes()
         */
        public BUILDER addScopes(Map<? extends String, ? extends ScopeConfig> scopes) {
            Objects.requireNonNull(scopes);
            this.scopes.putAll(scopes);
            return self();
        }

        /**
         * This method adds a new value to the map, or replaces it if the key already exists.
         *
         * @param key key to add or replace
         * @param scope new value for the key
         * @return updated builder instance
         * @see #scopes()
         */
        public BUILDER putScope(String key, ScopeConfig scope) {
            Objects.requireNonNull(key);
            Objects.requireNonNull(scope);
            this.scopes.put(key, scope);
            return self();
        }

        /**
         * This method adds a new value to the map, or replaces it if the key already exists.
         *
         * @param key key to add or replace
         * @param consumer builder consumer to create new value for the key
         * @return updated builder instance
         * @see #scopes()
         */
        public BUILDER putScope(String key, Consumer<ScopeConfig.Builder> consumer) {
            Objects.requireNonNull(key);
            Objects.requireNonNull(consumer);
            var builder = ScopeConfig.builder();
            consumer.accept(builder);
            this.putScope(key, builder.build());
            return self();
        }

        /**
         * Default scope value to associate with meters that are registered without an explicit setting; no setting means meters
         * are assigned scope {@value io.helidon.metrics.api.Meter.Scope#DEFAULT}.
         *
         * @return the default value
         */
        public Optional<String> defaultValue() {
            return Optional.ofNullable(defaultValue);
        }

        /**
         * Tag name for storing meter scope values in the underlying implementation meter registry.
         *
         * @return the tag name
         */
        public Optional<String> tagName() {
            return Optional.ofNullable(tagName);
        }

        /**
         * Settings for individual scopes.
         *
         * @return the scopes
         */
        public Map<String, ScopeConfig> scopes() {
            return scopes;
        }

        /**
         * If this instance was configured, this would be the config instance used.
         *
         * @return config node used to configure this builder, or empty if not configured
         */
        public Optional<Config> config() {
            return Optional.ofNullable(config);
        }

        @Override
        public String toString() {
            return "ScopingConfigBuilder{"
                    + "defaultValue=" + defaultValue + ","
                    + "tagName=" + tagName + ","
                    + "scopes=" + scopes
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
        }

        /**
         * Default scope value to associate with meters that are registered without an explicit setting; no setting means meters
         * are assigned scope {@value io.helidon.metrics.api.Meter.Scope#DEFAULT}.
         *
         * @param defaultValue default scope value
         * @return updated builder instance
         * @see #defaultValue()
         */
        BUILDER defaultValue(Optional<String> defaultValue) {
            Objects.requireNonNull(defaultValue);
            this.defaultValue = defaultValue.map(java.lang.String.class::cast).orElse(this.defaultValue);
            return self();
        }

        /**
         * Tag name for storing meter scope values in the underlying implementation meter registry.
         *
         * @param tagName tag name for storing scope values
         * @return updated builder instance
         * @see #tagName()
         */
        BUILDER tagName(Optional<String> tagName) {
            Objects.requireNonNull(tagName);
            this.tagName = tagName.map(java.lang.String.class::cast).orElse(this.tagName);
            return self();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class ScopingConfigImpl implements ScopingConfig {

            private final Map<String, ScopeConfig> scopes;
            private final Optional<String> defaultValue;
            private final Optional<String> tagName;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected ScopingConfigImpl(ScopingConfig.BuilderBase<?, ?> builder) {
                this.defaultValue = builder.defaultValue();
                this.tagName = builder.tagName();
                this.scopes = Collections.unmodifiableMap(new LinkedHashMap<>(builder.scopes()));
            }

            @Override
            public Optional<String> defaultValue() {
                return defaultValue;
            }

            @Override
            public Optional<String> tagName() {
                return tagName;
            }

            @Override
            public Map<String, ScopeConfig> scopes() {
                return scopes;
            }

            @Override
            public String toString() {
                return "ScopingConfig{"
                        + "defaultValue=" + defaultValue + ","
                        + "tagName=" + tagName + ","
                        + "scopes=" + scopes
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof ScopingConfig other)) {
                    return false;
                }
                return Objects.equals(defaultValue, other.defaultValue())
                    && Objects.equals(tagName, other.tagName())
                    && Objects.equals(scopes, other.scopes());
            }

            @Override
            public int hashCode() {
                return Objects.hash(defaultValue, tagName, scopes);
            }

        }

    }

    /**
     * Fluent API builder for {@link ScopingConfig}.
     */
    class Builder extends ScopingConfig.BuilderBase<ScopingConfig.Builder, ScopingConfig> implements io.helidon.common.Builder<ScopingConfig.Builder, ScopingConfig> {

        private Builder() {
        }

        @Override
        public ScopingConfig buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new ScopingConfigImpl(this);
        }

        @Override
        public ScopingConfig build() {
            return buildPrototype();
        }

    }

}
