/*
 * Copyright (c) 2023 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.inject.processor;

import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import javax.annotation.processing.ProcessingEnvironment;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Generated;
import io.helidon.common.types.TypedElementInfo;

/**
 * Attributes that can be observed via {@link io.helidon.inject.processor.spi.InjectionAnnotationProcessorObserver}.
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.processor.BlueprintProcessor", trigger = "io.helidon.inject.processor.ProcessingEventBlueprint")
public interface ProcessingEvent extends ProcessingEventBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static ProcessingEvent.Builder builder() {
        return new ProcessingEvent.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static ProcessingEvent.Builder builder(ProcessingEvent instance) {
        return ProcessingEvent.builder().from(instance);
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static ProcessingEvent create() {
        return ProcessingEvent.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link ProcessingEvent}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends ProcessingEvent.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends ProcessingEvent> implements Prototype.Builder<BUILDER, PROTOTYPE> {

        private final Set<TypedElementInfo> elementsOfInterest = new LinkedHashSet<>();
        private ProcessingEnvironment processingEnvironment;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(ProcessingEvent prototype) {
            processingEnvironment(prototype.processingEnvironment());
            addElementsOfInterest(prototype.elementsOfInterest());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(ProcessingEvent.BuilderBase<?, ?> builder) {
            builder.processingEnvironment().ifPresent(this::processingEnvironment);
            addElementsOfInterest(builder.elementsOfInterest());
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #processingEnvironment()
         */
        public BUILDER clearProcessingEnvironment() {
            this.processingEnvironment = null;
            return self();
        }

        /**
         * Optionally, the active {@link javax.annotation.processing.ProcessingEnvironment} if it is available.
         *
         * @param processingEnvironment the processing environment if it is available
         * @return updated builder instance
         * @see #processingEnvironment()
         */
        public BUILDER processingEnvironment(ProcessingEnvironment processingEnvironment) {
            Objects.requireNonNull(processingEnvironment);
            this.processingEnvironment = processingEnvironment;
            return self();
        }

        /**
         * The {@code jakarta.inject.Inject}'able type elements, and possibly any other elements that are found to be of interest for
         * processing. The set of processed elements are subject to change in the future. The implementor is therefore encouraged
         * to not make assumptions about the set of elements that are in this set.
         *
         * @param elementsOfInterest the set of injectable elements, and any other elements of interest to the APT
         * @return updated builder instance
         * @see #elementsOfInterest()
         */
        public BUILDER elementsOfInterest(Set<? extends TypedElementInfo> elementsOfInterest) {
            Objects.requireNonNull(elementsOfInterest);
            this.elementsOfInterest.clear();
            this.elementsOfInterest.addAll(elementsOfInterest);
            return self();
        }

        /**
         * The {@code jakarta.inject.Inject}'able type elements, and possibly any other elements that are found to be of interest for
         * processing. The set of processed elements are subject to change in the future. The implementor is therefore encouraged
         * to not make assumptions about the set of elements that are in this set.
         *
         * @param elementsOfInterest the set of injectable elements, and any other elements of interest to the APT
         * @return updated builder instance
         * @see #elementsOfInterest()
         */
        public BUILDER addElementsOfInterest(Set<? extends TypedElementInfo> elementsOfInterest) {
            Objects.requireNonNull(elementsOfInterest);
            this.elementsOfInterest.addAll(elementsOfInterest);
            return self();
        }

        /**
         * Optionally, the active {@link javax.annotation.processing.ProcessingEnvironment} if it is available.
         *
         * @return the processing environment
         */
        public Optional<ProcessingEnvironment> processingEnvironment() {
            return Optional.ofNullable(processingEnvironment);
        }

        /**
         * The {@code jakarta.inject.Inject}'able type elements, and possibly any other elements that are found to be of interest for
         * processing. The set of processed elements are subject to change in the future. The implementor is therefore encouraged
         * to not make assumptions about the set of elements that are in this set.
         *
         * @return the elements of interest
         */
        public Set<TypedElementInfo> elementsOfInterest() {
            return elementsOfInterest;
        }

        @Override
        public String toString() {
            return "ProcessingEventBuilder{"
                    + "processingEnvironment=" + processingEnvironment + ","
                    + "elementsOfInterest=" + elementsOfInterest
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
        }

        /**
         * Optionally, the active {@link javax.annotation.processing.ProcessingEnvironment} if it is available.
         *
         * @param processingEnvironment the processing environment if it is available
         * @return updated builder instance
         * @see #processingEnvironment()
         */
        BUILDER processingEnvironment(Optional<? extends ProcessingEnvironment> processingEnvironment) {
            Objects.requireNonNull(processingEnvironment);
            this.processingEnvironment = processingEnvironment.orElse(null);
            return self();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class ProcessingEventImpl implements ProcessingEvent {

            private final Optional<ProcessingEnvironment> processingEnvironment;
            private final Set<TypedElementInfo> elementsOfInterest;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected ProcessingEventImpl(ProcessingEvent.BuilderBase<?, ?> builder) {
                this.processingEnvironment = builder.processingEnvironment();
                this.elementsOfInterest = Collections.unmodifiableSet(new LinkedHashSet<>(builder.elementsOfInterest()));
            }

            @Override
            public Optional<ProcessingEnvironment> processingEnvironment() {
                return processingEnvironment;
            }

            @Override
            public Set<TypedElementInfo> elementsOfInterest() {
                return elementsOfInterest;
            }

            @Override
            public String toString() {
                return "ProcessingEvent{"
                        + "processingEnvironment=" + processingEnvironment + ","
                        + "elementsOfInterest=" + elementsOfInterest
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof ProcessingEvent other)) {
                    return false;
                }
                return Objects.equals(processingEnvironment, other.processingEnvironment())
                        && Objects.equals(elementsOfInterest, other.elementsOfInterest());
            }

            @Override
            public int hashCode() {
                return Objects.hash(processingEnvironment, elementsOfInterest);
            }

        }

    }

    /**
     * Fluent API builder for {@link ProcessingEvent}.
     */
    class Builder extends ProcessingEvent.BuilderBase<ProcessingEvent.Builder, ProcessingEvent> implements io.helidon.common.Builder<ProcessingEvent.Builder, ProcessingEvent> {

        private Builder() {
        }

        @Override
        public ProcessingEvent buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new ProcessingEventImpl(this);
        }

        @Override
        public ProcessingEvent build() {
            return buildPrototype();
        }

    }

}
