/*
 * Copyright (c) 2023 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.inject.api;

import java.util.Objects;
import java.util.Optional;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Generated;

/**
 * Metrics.
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.processor.BlueprintProcessor", trigger = "io.helidon.inject.api.MetricsBlueprint")
public interface Metrics extends MetricsBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static Metrics.Builder builder() {
        return new Metrics.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static Metrics.Builder builder(Metrics instance) {
        return Metrics.builder().from(instance);
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static Metrics create() {
        return Metrics.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link Metrics}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends Metrics.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends Metrics> implements Prototype.Builder<BUILDER, PROTOTYPE> {

        private Integer cacheHitCount;
        private Integer cacheLookupCount;
        private Integer lookupCount;
        private Integer serviceCount;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(Metrics prototype) {
            serviceCount(prototype.serviceCount());
            lookupCount(prototype.lookupCount());
            cacheLookupCount(prototype.cacheLookupCount());
            cacheHitCount(prototype.cacheHitCount());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(Metrics.BuilderBase<?, ?> builder) {
            builder.serviceCount().ifPresent(this::serviceCount);
            builder.lookupCount().ifPresent(this::lookupCount);
            builder.cacheLookupCount().ifPresent(this::cacheLookupCount);
            builder.cacheHitCount().ifPresent(this::cacheHitCount);
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #serviceCount()
         */
        public BUILDER clearServiceCount() {
            this.serviceCount = null;
            return self();
        }

        /**
         * The total service count in the registry.
         *
         * @param serviceCount total service count
         * @return updated builder instance
         * @see #serviceCount()
         */
        public BUILDER serviceCount(int serviceCount) {
            Objects.requireNonNull(serviceCount);
            this.serviceCount = serviceCount;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #lookupCount()
         */
        public BUILDER clearLookupCount() {
            this.lookupCount = null;
            return self();
        }

        /**
         * The total number of {@code Services::lookup()} calls since jvm start, or since last reset.
         *
         * @param lookupCount lookup count
         * @return updated builder instance
         * @see #lookupCount()
         */
        public BUILDER lookupCount(int lookupCount) {
            Objects.requireNonNull(lookupCount);
            this.lookupCount = lookupCount;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #cacheLookupCount()
         */
        public BUILDER clearCacheLookupCount() {
            this.cacheLookupCount = null;
            return self();
        }

        /**
         * The total number of {@code Services::lookup()} calls that were attempted against the lookup cache. This will be empty
         * if caching is disabled.
         *
         * @param cacheLookupCount cache lookup count
         * @return updated builder instance
         * @see #cacheLookupCount()
         */
        public BUILDER cacheLookupCount(int cacheLookupCount) {
            Objects.requireNonNull(cacheLookupCount);
            this.cacheLookupCount = cacheLookupCount;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #cacheHitCount()
         */
        public BUILDER clearCacheHitCount() {
            this.cacheHitCount = null;
            return self();
        }

        /**
         * The total number of {@code Services:lookup()} calls that were successfully resolved via cache. This will be a value less
         * than or equal to {@link #cacheLookupCount()}.
         *
         * @param cacheHitCount cache hit count
         * @return updated builder instance
         * @see #cacheHitCount()
         */
        public BUILDER cacheHitCount(int cacheHitCount) {
            Objects.requireNonNull(cacheHitCount);
            this.cacheHitCount = cacheHitCount;
            return self();
        }

        /**
         * The total service count in the registry.
         *
         * @return the service count
         */
        public Optional<Integer> serviceCount() {
            return Optional.ofNullable(serviceCount);
        }

        /**
         * The total number of {@code Services::lookup()} calls since jvm start, or since last reset.
         *
         * @return the lookup count
         */
        public Optional<Integer> lookupCount() {
            return Optional.ofNullable(lookupCount);
        }

        /**
         * The total number of {@code Services::lookup()} calls that were attempted against the lookup cache. This will be empty
         * if caching is disabled.
         *
         * @return the cache lookup count
         */
        public Optional<Integer> cacheLookupCount() {
            return Optional.ofNullable(cacheLookupCount);
        }

        /**
         * The total number of {@code Services:lookup()} calls that were successfully resolved via cache. This will be a value less
         * than or equal to {@link #cacheLookupCount()}.
         *
         * @return the cache hit count
         */
        public Optional<Integer> cacheHitCount() {
            return Optional.ofNullable(cacheHitCount);
        }

        @Override
        public String toString() {
            return "MetricsBuilder{"
                    + "serviceCount=" + serviceCount + ","
                    + "lookupCount=" + lookupCount + ","
                    + "cacheLookupCount=" + cacheLookupCount + ","
                    + "cacheHitCount=" + cacheHitCount
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
        }

        /**
         * The total service count in the registry.
         *
         * @param serviceCount total service count
         * @return updated builder instance
         * @see #serviceCount()
         */
        BUILDER serviceCount(Optional<Integer> serviceCount) {
            Objects.requireNonNull(serviceCount);
            this.serviceCount = serviceCount.map(java.lang.Integer.class::cast).orElse(this.serviceCount);
            return self();
        }

        /**
         * The total number of {@code Services::lookup()} calls since jvm start, or since last reset.
         *
         * @param lookupCount lookup count
         * @return updated builder instance
         * @see #lookupCount()
         */
        BUILDER lookupCount(Optional<Integer> lookupCount) {
            Objects.requireNonNull(lookupCount);
            this.lookupCount = lookupCount.map(java.lang.Integer.class::cast).orElse(this.lookupCount);
            return self();
        }

        /**
         * The total number of {@code Services::lookup()} calls that were attempted against the lookup cache. This will be empty
         * if caching is disabled.
         *
         * @param cacheLookupCount cache lookup count
         * @return updated builder instance
         * @see #cacheLookupCount()
         */
        BUILDER cacheLookupCount(Optional<Integer> cacheLookupCount) {
            Objects.requireNonNull(cacheLookupCount);
            this.cacheLookupCount = cacheLookupCount.map(java.lang.Integer.class::cast).orElse(this.cacheLookupCount);
            return self();
        }

        /**
         * The total number of {@code Services:lookup()} calls that were successfully resolved via cache. This will be a value less
         * than or equal to {@link #cacheLookupCount()}.
         *
         * @param cacheHitCount cache hit count
         * @return updated builder instance
         * @see #cacheHitCount()
         */
        BUILDER cacheHitCount(Optional<Integer> cacheHitCount) {
            Objects.requireNonNull(cacheHitCount);
            this.cacheHitCount = cacheHitCount.map(java.lang.Integer.class::cast).orElse(this.cacheHitCount);
            return self();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class MetricsImpl implements Metrics {

            private final Optional<Integer> cacheHitCount;
            private final Optional<Integer> cacheLookupCount;
            private final Optional<Integer> lookupCount;
            private final Optional<Integer> serviceCount;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected MetricsImpl(Metrics.BuilderBase<?, ?> builder) {
                this.serviceCount = builder.serviceCount();
                this.lookupCount = builder.lookupCount();
                this.cacheLookupCount = builder.cacheLookupCount();
                this.cacheHitCount = builder.cacheHitCount();
            }

            @Override
            public Optional<Integer> serviceCount() {
                return serviceCount;
            }

            @Override
            public Optional<Integer> lookupCount() {
                return lookupCount;
            }

            @Override
            public Optional<Integer> cacheLookupCount() {
                return cacheLookupCount;
            }

            @Override
            public Optional<Integer> cacheHitCount() {
                return cacheHitCount;
            }

            @Override
            public String toString() {
                return "Metrics{"
                        + "serviceCount=" + serviceCount + ","
                        + "lookupCount=" + lookupCount + ","
                        + "cacheLookupCount=" + cacheLookupCount + ","
                        + "cacheHitCount=" + cacheHitCount
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof Metrics other)) {
                    return false;
                }
                return Objects.equals(serviceCount, other.serviceCount())
                        && Objects.equals(lookupCount, other.lookupCount())
                        && Objects.equals(cacheLookupCount, other.cacheLookupCount())
                        && Objects.equals(cacheHitCount, other.cacheHitCount());
            }

            @Override
            public int hashCode() {
                return Objects.hash(serviceCount, lookupCount, cacheLookupCount, cacheHitCount);
            }

        }

    }

    /**
     * Fluent API builder for {@link Metrics}.
     */
    class Builder extends Metrics.BuilderBase<Metrics.Builder, Metrics> implements io.helidon.common.Builder<Metrics.Builder, Metrics> {

        private Builder() {
        }

        @Override
        public Metrics buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new MetricsImpl(this);
        }

        @Override
        public Metrics build() {
            return buildPrototype();
        }

    }

}
